Add-Type -AssemblyName PresentationFramework

# Liste des serveurs  surveiller
$Servers = @(
    @{ Name = "HFS"; Host = "192.168.1.88"; Port = 80 },
    @{ Name = "FRN"; Host = "192.168.1.88"; Port = 10024 },
    @{ Name = "MUMBLE"; Host = "192.168.1.88"; Port = 64738 },
    @{ Name = "T-BEAM"; Host = "192.168.1.55"; Port = 80 },
    @{ Name = "TTGO-1"; Host = "192.168.1.58"; Port = 80 },
    @{ Name = "TTGO-2"; Host = "192.168.1.53"; Port = 80 },
    @{ Name = "SDR"; Host = "192.168.1.86"; Port = 50101 }
)

# Intervalle de surveillance (en secondes)
$INTERVALLE = 5

# Chemin du fichier log (sur le Bureau)
$logFile = [System.IO.Path]::Combine([Environment]::GetFolderPath("Desktop"), "log.txt")

# Cration de la fentre WPF principale
[xml]$xaml = @"
<Window xmlns="http://schemas.microsoft.com/winfx/2006/xaml/presentation"
        Title="Mon Reseau" Height="120" Width="150" ResizeMode="NoResize" WindowStartupLocation="Manual">
    <StackPanel Margin="10" HorizontalAlignment="Center">
        <StackPanel Orientation="Horizontal" HorizontalAlignment="Center" VerticalAlignment="Center">
            <TextBlock Name="statusText" Text="- En Fonction -" Margin="0,0,10,0" FontSize="14" FontStyle="Italic" VerticalAlignment="Center"/>
            <Ellipse Name="statusLed" Width="12" Height="12" Fill="#00FF00" VerticalAlignment="Center"/>
        </StackPanel>
        <TextBlock Text="" Height="5"/>
        <Button Name="quitButton" Content="Quitter" Width="70" Height="30" HorizontalAlignment="Center"/>
    </StackPanel>
</Window>
"@

$reader = (New-Object System.Xml.XmlNodeReader $xaml)
$window = [Windows.Markup.XamlReader]::Load($reader)

$statusText = $window.FindName("statusText")
$quitButton = $window.FindName("quitButton")
$statusLed = $window.FindName("statusLed")

# Positionner la fentre en bas  droite de l'cran
$screenWidth = [System.Windows.SystemParameters]::PrimaryScreenWidth
$screenHeight = [System.Windows.SystemParameters]::PrimaryScreenHeight
$window.Width = 150
$window.Height = 120
$window.Left = $screenWidth - $window.Width - -5
$window.Top = $screenHeight - $window.Height - 33

# Rendre la fentre "toujours au-dessus"
$window.Topmost = $true

# Variable d'arrt globale
$stopMonitoring = $false

# Action bouton Quitter
$quitButton.Add_Click({
    $stopMonitoring = $true
    $window.Close()
})

# Lancer la boucle de surveillance dans un thread spar (runspace)
$runspace = [runspacefactory]::CreateRunspace()
$runspace.ApartmentState = "STA"
$runspace.ThreadOptions = "ReuseThread"
$runspace.Open()
$ps = [powershell]::Create()
$ps.Runspace = $runspace

$ps.AddScript({
    param($Servers, $INTERVALLE, [ref]$stopFlag, $statusText, $statusLed, $logFile)

    # Fonction de test TCP
    function Test-TcpPortFast {
        param (
            [string]$ComputerName,
            [int]$Port,
            [int]$Timeout = 1000
        )
        try {
            $client = New-Object System.Net.Sockets.TcpClient
            $async = $client.BeginConnect($ComputerName, $Port, $null, $null)
            $wait = $async.AsyncWaitHandle.WaitOne($Timeout, $false)
            if (!$wait) {
                $client.Close()
                return $false
            }
            $client.EndConnect($async)
            $client.Close()
            return $true
        } catch {
            return $false
        }
    }

    $previousStates = @{}  # Dictionnaire pour garder la trace de l'tat prcdent des serveurs

    while (-not $stopFlag.Value) {
        $statusOk = $true  # Variable pour vrifier si tous les serveurs sont en ligne

        foreach ($server in $Servers) {
            $currentState = Test-TcpPortFast -ComputerName $server.Host -Port $server.Port
            $previousState = $previousStates[$server.Name]

            if ($currentState) {
                # Le service est en ligne, vrifier s'il tait hors ligne prcdemment
                if ($previousState -eq $false) {
                    # Ajouter au log lorsque le service revient en ligne
                    $alertMessage = "$(Get-Date -Format 'yyyy-MM-dd HH:mm:ss') - $($server.Name) Remis en Ligne - IP: $($server.Host) - Port: $($server.Port)"
                    Add-Content -Path $logFile -Value $alertMessage
                }
                Write-Output "$(Get-Date -Format 'yyyy-MM-dd HH:mm:ss') - $($server.Name) OK"
            } else {
                # Le service est hors ligne, vrifier s'il tait en ligne prcdemment
                if ($previousState -eq $true) {
                    # Ajouter au log lorsqu'un service passe hors ligne
                    $alertMessage = "$(Get-Date -Format 'yyyy-MM-dd HH:mm:ss') - $($server.Name) Hors Ligne - IP: $($server.Host) - Port: $($server.Port)"
                    Add-Content -Path $logFile -Value $alertMessage
                }
                Write-Output "$(Get-Date -Format 'yyyy-MM-dd HH:mm:ss') - $($server.Name) Hors Ligne"
                $statusOk = $false  # Si un serveur est hors ligne, on change l'tat du voyant et du texte

                # Mettre  jour immdiatement le texte et le voyant pour signaler un problme
                $statusLed.Dispatcher.Invoke({
                    $statusLed.Fill = 'Red'  # Le voyant devient rouge en cas de service hors ligne
                })
                $statusText.Dispatcher.Invoke({
                    $statusText.Text = "- Alerte -"  # Afficher "Alerte" quand un service est hors ligne
                })
                
                # Ajouter les bips sonores
                for ($i = 0; $i -lt 6; $i++) {
                    [console]::Beep(2500, 500)  # Frquence 2500Hz pendant 500ms
                    Start-Sleep -Milliseconds 800  # Attente de 800ms entre les bips
                }

                # Afficher l'alerte et s'assurer qu'elle reste au premier plan
                $message = "Le $($server.Name) est Hors Ligne !`nIP : $($server.Host)`nPort : $($server.Port)"
                $alertTime = Get-Date -Format "dd/MM/yyyy HH:mm:ss"

                # Crer une fentre d'alerte
                $alertWindow = New-Object System.Windows.Window
                $alertWindow.Title = "Alerte - $($server.Name)"
                $alertWindow.Topmost = $true
                $alertWindow.Width = 400
                $alertWindow.Height = 250
                $alertWindow.ResizeMode = "NoResize"
                $alertWindow.Background = [System.Windows.Media.Brushes]::LightGray  # Fond gris clair

                # Calculer la position centrale
                $screenWidth = [System.Windows.SystemParameters]::PrimaryScreenWidth
                $screenHeight = [System.Windows.SystemParameters]::PrimaryScreenHeight
                $alertWindow.Left = ($screenWidth - $alertWindow.Width) / 2
                $alertWindow.Top = ($screenHeight - $alertWindow.Height) / 2

                # StackPanel avec un voyant et texte d'alerte
                $stackPanel = New-Object System.Windows.Controls.StackPanel
                $stackPanel.VerticalAlignment = "Center"
                $stackPanel.HorizontalAlignment = "Center"
                $stackPanel.Orientation = "Vertical"
                $alertWindow.Content = $stackPanel

                # Voyant rouge plus petit
                $alertLed = New-Object System.Windows.Shapes.Ellipse
                $alertLed.Width = 20
                $alertLed.Height = 20
                $alertLed.Fill = [System.Windows.Media.Brushes]::Red
                $stackPanel.Children.Add($alertLed)

                # Texte d'alerte (en italique, rouge et gras)
                $alertText = New-Object System.Windows.Controls.TextBlock
                $alertText.Text = $message
                $alertText.HorizontalAlignment = "Center"
                $alertText.Margin = "10"
                $alertText.FontStyle = [System.Windows.FontStyles]::Italic  # Texte en italique
                $alertText.Foreground = [System.Windows.Media.Brushes]::black  # Texte en noir
                $alertText.FontWeight = [System.Windows.FontWeights]::Bold  # Texte en gras
                $stackPanel.Children.Add($alertText)

                # Ajouter la date et l'heure de la premire alerte
                $alertTimeText = New-Object System.Windows.Controls.TextBlock
                $alertTimeText.Text = "Premire Alerte : $alertTime"
                $alertTimeText.HorizontalAlignment = "Center"
                $alertTimeText.Margin = "10"
                $alertTimeText.FontStyle = [System.Windows.FontStyles]::Italic
                $alertTimeText.Foreground = [System.Windows.Media.Brushes]::DarkBlue  # Texte en bleu
                $alertTimeText.FontWeight = [System.Windows.FontWeights]::Normal
                $stackPanel.Children.Add($alertTimeText)

                # Bouton de fermeture
                $closeButton = New-Object System.Windows.Controls.Button
                $closeButton.Content = "Fermer"
                $closeButton.Width = 100
                $closeButton.Height = 30
                $closeButton.HorizontalAlignment = "Center"
                $closeButton.Margin = "10"
                $stackPanel.Children.Add($closeButton)

                # Lorsque le bouton est cliqu, fermer la fentre
                $closeButton.Add_Click({
                    $alertWindow.Close()
                })

                # Afficher la fentre d'alerte
                $alertWindow.ShowDialog()
            }

            # Mettre  jour l'tat prcdent
            $previousStates[$server.Name] = $currentState
        }

        # Si tous les serveurs sont en ligne, rinitialiser le texte et le voyant
        if ($statusOk) {
            $statusLed.Dispatcher.Invoke({
                $statusLed.Fill = '#00FF00'  # Le voyant devient vert lorsque tous les serveurs sont en ligne
            })
            $statusText.Dispatcher.Invoke({
                $statusText.Text = "- En Fonction -"  # Rafficher "En Fonction" si tout est ok
            })
        }

        Start-Sleep -Seconds $INTERVALLE
    }
}) | Out-Null

$ps.AddArgument($Servers)
$ps.AddArgument($INTERVALLE)
$ps.AddArgument([ref]$stopMonitoring)
$ps.AddArgument($statusText)
$ps.AddArgument($statusLed)
$ps.AddArgument($logFile)
$ps.BeginInvoke()

# Afficher la fentre principale
$window.ShowDialog() | Out-Null

# Nettoyage
$ps.Dispose()
$runspace.Close()
$runspace.Dispose()
