class HamCallsign(object):
    @staticmethod
    def getCountry(callsign: str):
        callsign = callsign.upper()
        for x in range(4, 0, -1):
            pfx = callsign[0:x]
            if pfx in CALL2COUNTRY:
                return CALL2COUNTRY[pfx]
        return None

    @staticmethod
    def getCountryCode(callsign: str):
        country = HamCallsign.getCountry(callsign)
        return country[0] if country else None

    @staticmethod
    def getCountryName(callsign: str):
        country = HamCallsign.findCountry(callsign)
        return country[1] if country else None


class MmsiNumber(object):
    @staticmethod
    def getMid(mmsi: str):
        return mmsi[0:3]

    @staticmethod
    def getCountry(mmsi: str):
        mid = MmsiNumber.getMid(mmsi)
        return MID2COUNTRY[mid] if mid in MID2COUNTRY else None

    @staticmethod
    def getCountryCode(mmsi: str):
        mid = MmsiNumber.getMid(mmsi)
        return MID2COUNTRY[mid][0] if mid in MID2COUNTRY else None

    @staticmethod
    def getCountryName(mmsi: str):
        mid = MmsiNumber.getMid(mmsi)
        return MID2COUNTRY[mid][1] if mid in MID2COUNTRY else None


class CountryCode(object):
    @staticmethod
    def getCountry(ccode: str):
        ccode = ccode.upper()
        return CCODE2COUNTRY[ccode] if ccode in CCODE2COUNTRY else None


#
# Mapping from HAM callsign prefixes to countries and locales
#

CALL2COUNTRY = {
    "0S"  : (None, "Principality of Seborga"),
    "1A"  : (None, "Sovereign Military Order of Malta"),
    "1B"  : (None, "Northern Cyprus or Blenheim Reef"),
    "1G"  : (None, "Geyser Reef"),
    "1L"  : (None, "Liberland"),
    "1M"  : (None, "Minerva Reefs"),
    "1S"  : (None, "Principality of Sealand"),
    "1Z"  : (None, "Kayin State"),
    "2"   : ("GB", "United Kingdom"),
    "3A"  : ("MC", "Monaco"),
    "3B"  : ("MU", "Mauritius"),
    "3C"  : ("GQ", "Equatorial Guinea"),
    "3DA" : ("SZ", "Swaziland"),
    "3DB" : ("SZ", "Swaziland"),
    "3DC" : ("SZ", "Swaziland"),
    "3DD" : ("SZ", "Swaziland"),
    "3DE" : ("SZ", "Swaziland"),
    "3DF" : ("SZ", "Swaziland"),
    "3DG" : ("SZ", "Swaziland"),
    "3DH" : ("SZ", "Swaziland"),
    "3DI" : ("SZ", "Swaziland"),
    "3DJ" : ("SZ", "Swaziland"),
    "3DK" : ("SZ", "Swaziland"),
    "3DL" : ("SZ", "Swaziland"),
    "3DM" : ("SZ", "Swaziland"),
    "3DN" : ("FJ", "Fiji"),
    "3DO" : ("FJ", "Fiji"),
    "3DP" : ("FJ", "Fiji"),
    "3DQ" : ("FJ", "Fiji"),
    "3DR" : ("FJ", "Fiji"),
    "3DS" : ("FJ", "Fiji"),
    "3DT" : ("FJ", "Fiji"),
    "3DU" : ("FJ", "Fiji"),
    "3DV" : ("FJ", "Fiji"),
    "3DW" : ("FJ", "Fiji"),
    "3DX" : ("FJ", "Fiji"),
    "3DY" : ("FJ", "Fiji"),
    "3DZ" : ("FJ", "Fiji"),
    "3E"  : ("PA", "Panama"),
    "3F"  : ("PA", "Panama"),
    "3G"  : ("CL", "Chile"),
    "3H"  : ("CN", "China"),
    "3I"  : ("CN", "China"),
    "3J"  : ("CN", "China"),
    "3K"  : ("CN", "China"),
    "3L"  : ("CN", "China"),
    "3M"  : ("CN", "China"),
    "3N"  : ("CN", "China"),
    "3O"  : ("CN", "China"),
    "3P"  : ("CN", "China"),
    "3Q"  : ("CN", "China"),
    "3R"  : ("CN", "China"),
    "3S"  : ("CN", "China"),
    "3T"  : ("CN", "China"),
    "3U"  : ("CN", "China"),
    "3V"  : ("TN", "Tunisia"),
    "3W"  : ("VN", "Vietnam"),
    "3X"  : ("GN", "Guinea"),
    "3Y"  : ("NO", "Norway"),
    "3Z"  : ("PL", "Poland"),
    "4A"  : ("MX", "Mexico"),
    "4B"  : ("MX", "Mexico"),
    "4C"  : ("MX", "Mexico"),
    "4D"  : ("PH", "Philippines"),
    "4E"  : ("PH", "Philippines"),
    "4F"  : ("PH", "Philippines"),
    "4G"  : ("PH", "Philippines"),
    "4H"  : ("PH", "Philippines"),
    "4I"  : ("PH", "Philippines"),
    "4J"  : ("AZ", "Azerbaijan"),
    "4K"  : ("AZ", "Azerbaijan"),
    "4L"  : ("GE", "Georgia"),
    "4M"  : ("VE", "Venezuela"),
    "4O"  : ("ME", "Montenegro"),
    "4P"  : ("LK", "Sri Lanka"),
    "4Q"  : ("LK", "Sri Lanka"),
    "4R"  : ("LK", "Sri Lanka"),
    "4S"  : ("LK", "Sri Lanka"),
    "4T"  : ("PE", "Peru"),
    "4U"  : (None, "United Nations"),
    "4V"  : ("HT", "Haiti"),
    "4W"  : ("TL", "East Timor"),
    "4X"  : ("IL", "Israel"),
    "4Y"  : (None, "International Civil Aviation Organization"),
    "4Z"  : ("IL", "Israel"),
    "5A"  : ("LY", "Libya"),
    "5B"  : ("CY", "Cyprus"),
    "5C"  : ("MA", "Morocco"),
    "5D"  : ("MA", "Morocco"),
    "5E"  : ("MA", "Morocco"),
    "5F"  : ("MA", "Morocco"),
    "5G"  : ("MA", "Morocco"),
    "5H"  : ("TZ", "Tanzania"),
    "5I"  : ("TZ", "Tanzania"),
    "5J"  : ("CO", "Colombia"),
    "5K"  : ("CO", "Colombia"),
    "5L"  : ("LR", "Liberia"),
    "5M"  : ("LR", "Liberia"),
    "5N"  : ("NG", "Nigeria"),
    "5O"  : ("NG", "Nigeria"),
    "5P"  : ("DK", "Denmark"),
    "5Q"  : ("DK", "Denmark"),
    "5R"  : ("MG", "Madagascar"),
    "5S"  : ("MG", "Madagascar"),
    "5T"  : ("MR", "Mauritania"),
    "5U"  : ("NE", "Niger"),
    "5V"  : ("TG", "Togo"),
    "5W"  : ("WS", "Western Samoa"),
    "5X"  : ("UG", "Uganda"),
    "5Y"  : ("KE", "Kenya"),
    "5Z"  : ("KE", "Kenya"),
    "6A"  : ("EG", "Egypt"),
    "6B"  : ("EG", "Egypt"),
    "6C"  : ("SY", "Syria"),
    "6D"  : ("MX", "Mexico"),
    "6E"  : ("MX", "Mexico"),
    "6F"  : ("MX", "Mexico"),
    "6G"  : ("MX", "Mexico"),
    "6H"  : ("MX", "Mexico"),
    "6I"  : ("MX", "Mexico"),
    "6J"  : ("MX", "Mexico"),
    "6K"  : ("HK", "South Korea"),
    "6L"  : ("HK", "South Korea"),
    "6M"  : ("HK", "South Korea"),
    "6N"  : ("HK", "South Korea"),
    "6O"  : ("SO", "Somalia"),
    "6P"  : ("PK", "Pakistan"),
    "6Q"  : ("PK", "Pakistan"),
    "6R"  : ("PK", "Pakistan"),
    "6S"  : ("PK", "Pakistan"),
    "6T"  : ("SD", "Sudan"),
    "6U"  : ("SD", "Sudan"),
    "6V"  : ("SN", "Senegal"),
    "6W"  : ("SN", "Senegal"),
    "6X"  : ("MG", "Madagascar"),
    "6Y"  : ("JM", "Jamaica"),
    "6Z"  : ("LR", "Liberia"),
    "7A"  : ("ID", "Indonesia"),
    "7B"  : ("ID", "Indonesia"),
    "7C"  : ("ID", "Indonesia"),
    "7D"  : ("ID", "Indonesia"),
    "7E"  : ("ID", "Indonesia"),
    "7F"  : ("ID", "Indonesia"),
    "7G"  : ("ID", "Indonesia"),
    "7H"  : ("ID", "Indonesia"),
    "7I"  : ("ID", "Indonesia"),
    "7J"  : ("JP", "Japan"),
    "7K"  : ("JP", "Japan"),
    "7L"  : ("JP", "Japan"),
    "7M"  : ("JP", "Japan"),
    "7N"  : ("JP", "Japan"),
    "7O"  : ("YE", "Yemen"),
    "7P"  : ("LS", "Lesotho"),
    "7Q"  : ("MW", "Malawi"),
    "7R"  : ("DZ", "Algeria"),
    "7S"  : ("SE", "Sweden"),
    "7T"  : ("DZ", "Algeria"),
    "7U"  : ("DZ", "Algeria"),
    "7V"  : ("DZ", "Algeria"),
    "7W"  : ("DZ", "Algeria"),
    "7X"  : ("DZ", "Algeria"),
    "7Y"  : ("DZ", "Algeria"),
    "7Z"  : ("SA", "Saudi Arabia"),
    "8A"  : ("ID", "Indonesia"),
    "8B"  : ("ID", "Indonesia"),
    "8C"  : ("ID", "Indonesia"),
    "8D"  : ("ID", "Indonesia"),
    "8E"  : ("ID", "Indonesia"),
    "8F"  : ("ID", "Indonesia"),
    "8G"  : ("ID", "Indonesia"),
    "8H"  : ("ID", "Indonesia"),
    "8I"  : ("ID", "Indonesia"),
    "8J"  : ("JP", "Japan"),
    "8K"  : ("JP", "Japan"),
    "8L"  : ("JP", "Japan"),
    "8M"  : ("JP", "Japan"),
    "8N"  : ("JP", "Japan"),
    "8O"  : ("BW", "Botswana"),
    "8P"  : ("BB", "Barbados"),
    "8Q"  : ("MV", "Maldives"),
    "8R"  : ("GY", "Guyana"),
    "8S"  : ("SE", "Sweden"),
    "8T"  : ("IN", "India"),
    "8U"  : ("IN", "India"),
    "8V"  : ("IN", "India"),
    "8W"  : ("IN", "India"),
    "8X"  : ("IN", "India"),
    "8Y"  : ("IN", "India"),
    "8Z"  : ("SA", "Saudi Arabia"),
    "9A"  : ("HR", "Croatia"),
    "9B"  : ("IR", "Iran"),
    "9C"  : ("IR", "Iran"),
    "9D"  : ("IR", "Iran"),
    "9E"  : ("ET", "Ethiopia"),
    "9F"  : ("ET", "Ethiopia"),
    "9G"  : ("GH", "Ghana"),
    "9H"  : ("MT", "Malta"),
    "9I"  : ("ZM", "Zambia"),
    "9J"  : ("ZM", "Zambia"),
    "9K"  : ("KW", "Kuwait"),
    "9L"  : ("SL", "Sierra Leone"),
    "9M0" : ("MY", "Spratly Islands"),
    "9M"  : ("MY", "Malaysia"),
    "9N"  : ("NP", "Nepal"),
    "9O"  : ("CD", "Democratic Republic of Congo"),
    "9P"  : ("CD", "Democratic Republic of Congo"),
    "9Q"  : ("CD", "Democratic Republic of Congo"),
    "9R"  : ("CD", "Democratic Republic of Congo"),
    "9S"  : ("CD", "Democratic Republic of Congo"),
    "9T"  : ("CD", "Democratic Republic of Congo"),
    "9U"  : ("BI", "Burundi"),
    "9V"  : ("SG", "Singapore"),
    "9W"  : ("MY", "Malaysia"),
    "9X"  : ("RW", "Rwanda"),
    "9Y"  : ("TT", "Trinidad and Tobago"),
    "9Z"  : ("TT", "Trinidad and Tobago"),
    "A2"  : ("BW", "Botswana"),
    "A3"  : ("TO", "Tonga"),
    "A4"  : ("OM", "Oman"),
    "A5"  : ("BT", "Bhutan"),
    "A6"  : ("AE", "United Arab Emirates"),
    "A7"  : ("QA", "Qatar"),
    "A8"  : ("LR", "Liberia"),
    "A9"  : ("BH", "Bahrain"),
    "AA"  : ("US", "United States"),
    "AB"  : ("US", "United States"),
    "AC"  : ("US", "United States"),
    "AD"  : ("US", "United States"),
    "AE"  : ("US", "United States"),
    "AF"  : ("US", "United States"),
    "AG"  : ("US", "United States"),
    "AH"  : ("US", "United States"),
    "AI"  : ("US", "United States"),
    "AJ"  : ("US", "United States"),
    "AK"  : ("US", "United States"),
    "AL"  : ("US", "United States"),
    "AM"  : ("ES", "Spain"),
    "AN"  : ("ES", "Spain"),
    "AO"  : ("ES", "Spain"),
    "AP"  : ("PK", "Pakistan"),
    "AQ"  : ("PK", "Pakistan"),
    "AR"  : ("PK", "Pakistan"),
    "AS"  : ("PK", "Pakistan"),
    "AT"  : ("IN", "India"),
    "AU"  : ("IN", "India"),
    "AV"  : ("IN", "India"),
    "AW"  : ("IN", "India"),
    "AX"  : ("AU", "Australia"),
    "AY"  : ("AR", "Argentina"),
    "AZ"  : ("AR", "Argentina"),
    "BM"  : ("TW", "Taiwan"),
    "BN"  : ("TW", "Taiwan"),
    "BO"  : ("TW", "Taiwan"),
    "BP"  : ("TW", "Taiwan"),
    "BQ"  : ("TW", "Taiwan"),
    "BU"  : ("TW", "Taiwan"),
    "BV9" : ("TW", "Spratly Islands"),
    "BV"  : ("TW", "Taiwan"),
    "BW"  : ("TW", "Taiwan"),
    "BX"  : ("TW", "Taiwan"),
    "B"   : ("CN", "China"),
    "C2"  : ("NR", "Nauru"),
    "C3"  : ("AD", "Andorra"),
    "C4"  : ("CY", "Cyprus"),
    "C5"  : ("GM", "Gambia"),
    "C6"  : ("BS", "Bahamas"),
    "C7"  : (None, "World Meteorological Organization"),
    "C8"  : ("MZ", "Mozambique"),
    "C9"  : ("MZ", "Mozambique"),
    "CA"  : ("CL", "Chile"),
    "CB"  : ("CL", "Chile"),
    "CC"  : ("CL", "Chile"),
    "CD"  : ("CL", "Chile"),
    "CE"  : ("CL", "Chile"),
    "CF"  : ("CA", "Canada"),
    "CG"  : ("CA", "Canada"),
    "CH"  : ("CA", "Canada"),
    "CI"  : ("CA", "Canada"),
    "CJ"  : ("CA", "Canada"),
    "CK"  : ("CA", "Canada"),
    "CL"  : ("CU", "Cuba"),
    "CM"  : ("CU", "Cuba"),
    "CN"  : ("MA", "Morocco"),
    "CO"  : ("CU", "Cuba"),
    "CP"  : ("BO", "Bolivia"),
    "CQ"  : ("PT", "Portugal"),
    "CR"  : ("PT", "Portugal"),
    "CS"  : ("PT", "Portugal"),
    "CT"  : ("PT", "Portugal"),
    "CU"  : ("PT", "Portugal"),
    "CV"  : ("UY", "Uruguay"),
    "CW"  : ("UY", "Uruguay"),
    "CX"  : ("UY", "Uruguay"),
    "CY"  : ("CA", "Canada"),
    "CZ"  : ("CA", "Canada"),
    "D0"  : ("UA", "Donetsk"),
    "D1"  : ("UA", "Donetsk"),
    "D2"  : ("AO", "Angola"),
    "D3"  : ("AO", "Angola"),
    "D4"  : ("CV", "Cape Verde"),
    "D5"  : ("LR", "Liberia"),
    "D6"  : ("KM", "Comoros"),
    "D7"  : ("HK", "South Korea"),
    "D8"  : ("HK", "South Korea"),
    "D9"  : ("HK", "South Korea"),
    "DA"  : ("DE", "Germany"),
    "DB"  : ("DE", "Germany"),
    "DC"  : ("DE", "Germany"),
    "DD"  : ("DE", "Germany"),
    "DE"  : ("DE", "Germany"),
    "DF"  : ("DE", "Germany"),
    "DG"  : ("DE", "Germany"),
    "DH"  : ("DE", "Germany"),
    "DI"  : ("DE", "Germany"),
    "DJ"  : ("DE", "Germany"),
    "DK"  : ("DE", "Germany"),
    "DL"  : ("DE", "Germany"),
    "DM"  : ("DE", "Germany"),
    "DN"  : ("DE", "Germany"),
    "DO"  : ("DE", "Germany"),
    "DP"  : ("DE", "Germany"),
    "DQ"  : ("DE", "Germany"),
    "DR"  : ("DE", "Germany"),
    "DS"  : ("HK", "South Korea"),
    "DT"  : ("HK", "South Korea"),
    "DU"  : ("PH", "Philippines"),
    "DV"  : ("PH", "Philippines"),
    "DW"  : ("PH", "Philippines"),
    "DX0" : ("PH", "Spratly Islands"),
    "DX"  : ("PH", "Philippines"),
    "DY"  : ("PH", "Philippines"),
    "DZ"  : ("PH", "Philippines"),
    "E2"  : ("TH", "Thailand"),
    "E3"  : ("ER", "Eritrea"),
    "E4"  : ("PS", "Palestine"),
    "E5"  : ("CK", "Cook Islands"),
    "E6"  : ("NU", "Niue"),
    "E7"  : ("BA", "Bosnia and Herzegovina"),
    "EA"  : ("ES", "Spain"),
    "EB"  : ("ES", "Spain"),
    "EC"  : ("ES", "Spain"),
    "ED"  : ("ES", "Spain"),
    "EE"  : ("ES", "Spain"),
    "EF"  : ("ES", "Spain"),
    "EG"  : ("ES", "Spain"),
    "EH"  : ("ES", "Spain"),
    "EI"  : ("IE", "Ireland"),
    "EJ"  : ("IE", "Ireland"),
    "EK"  : ("AM", "Armenia"),
    "EL"  : ("LR", "Liberia"),
    "EM"  : ("UA", "Ukraine"),
    "EN"  : ("UA", "Ukraine"),
    "EO"  : ("UA", "Ukraine"),
    "EP"  : ("IR", "Iran"),
    "EQ"  : ("IR", "Iran"),
    "ER"  : ("MD", "Moldova"),
    "ES"  : ("EE", "Estonia"),
    "ET"  : ("ET", "Ethiopia"),
    "EU"  : ("BY", "Belarus"),
    "EV"  : ("BY", "Belarus"),
    "EW"  : ("BY", "Belarus"),
    "EX"  : ("KG", "Kyrgyzstan"),
    "EY"  : ("TJ", "Tajikistan"),
    "EZ"  : ("TM", "Turkmenistan"),
    "F"   : ("FR", "France"),
    "G"   : ("GB", "United Kingdom"),
    "H2"  : ("CY", "Cyprus"),
    "H3"  : ("PA", "Panama"),
    "H4"  : ("SB", "Solomon Islands"),
    "H5"  : (None, "Bophuthatswana"),
    "H6"  : ("NI", "Nicaragua"),
    "H7"  : ("NI", "Nicaragua"),
    "H8"  : ("PA", "Panama"),
    "H9"  : ("PA", "Panama"),
    "HA"  : ("HU", "Hungary"),
    "HB3Y" : ("LI", "Liechtenstein"),
    "HB0" : ("LI", "Liechtenstein"),
    "HBL" : ("LI", "Liechtenstein"),
    "HB"  : ("CH", "Switzerland"),
    "HC"  : ("EC", "Ecuador"),
    "HD"  : ("EC", "Ecuador"),
    "HE"  : ("CH", "Switzerland"),
    "HF"  : ("PL", "Poland"),
    "HG"  : ("HU", "Hungary"),
    "HH"  : ("HT", "Haiti"),
    "HI"  : ("DO", "Dominican Republic"),
    "HJ"  : ("CO", "Colombia"),
    "HK"  : ("CO", "Colombia"),
    "HL"  : ("HK", "South Korea"),
    "HM"  : ("KP", "North Korea"),
    "HN"  : ("IQ", "Iraq"),
    "HO"  : ("PA", "Panama"),
    "HP"  : ("PA", "Panama"),
    "HQ"  : ("HN", "Honduras"),
    "HR"  : ("HN", "Honduras"),
    "HS"  : ("TH", "Thailand"),
    "HT"  : ("NI", "Nicaragua"),
    "HU"  : ("SV", "El Salvador"),
    "HV"  : ("VA", "Vatican"),
    "HW"  : ("FR", "France"),
    "HX"  : ("FR", "France"),
    "HY"  : ("FR", "France"),
    "HZ"  : ("SA", "Saudi Arabia"),
    "I"   : ("IT", "Italy"),
    "J2"  : ("DJ", "Djibouti"),
    "J3"  : ("GD", "Grenada"),
    "J4"  : ("GR", "Greece"),
    "J5"  : ("GW", "Guinea-Bissau"),
    "J6"  : ("LC", "Saint Lucia"),
    "J7"  : ("DM", "Dominica"),
    "J8"  : ("VC", "Saint Vincent and Grenadines"),
    "JA"  : ("JP", "Japan"),
    "JB"  : ("JP", "Japan"),
    "JC"  : ("JP", "Japan"),
    "JD"  : ("JP", "Japan"),
    "JE"  : ("JP", "Japan"),
    "JF"  : ("JP", "Japan"),
    "JG"  : ("JP", "Japan"),
    "JH"  : ("JP", "Japan"),
    "JI"  : ("JP", "Japan"),
    "JJ"  : ("JP", "Japan"),
    "JK"  : ("JP", "Japan"),
    "JL"  : ("JP", "Japan"),
    "JM"  : ("JP", "Japan"),
    "JN"  : ("JP", "Japan"),
    "JO"  : ("JP", "Japan"),
    "JP"  : ("JP", "Japan"),
    "JQ"  : ("JP", "Japan"),
    "JR"  : ("JP", "Japan"),
    "JS"  : ("JP", "Japan"),
    "JT"  : ("MN", "Mongolia"),
    "JU"  : ("MN", "Mongolia"),
    "JV"  : ("MN", "Mongolia"),
    "JW"  : ("NO", "Norway"),
    "JX"  : ("NO", "Norway"),
    "JY"  : ("JO", "Jordan"),
    "JZ"  : ("ID", "Indonesia"),
    "K"   : ("US", "United States"),
    "L2"  : ("AR", "Argentina"),
    "L3"  : ("AR", "Argentina"),
    "L4"  : ("AR", "Argentina"),
    "L5"  : ("AR", "Argentina"),
    "L6"  : ("AR", "Argentina"),
    "L7"  : ("AR", "Argentina"),
    "L8"  : ("AR", "Argentina"),
    "L9"  : ("AR", "Argentina"),
    "LA"  : ("NO", "Norway"),
    "LB"  : ("NO", "Norway"),
    "LC"  : ("NO", "Norway"),
    "LD"  : ("NO", "Norway"),
    "LE"  : ("NO", "Norway"),
    "LF"  : ("NO", "Norway"),
    "LG"  : ("NO", "Norway"),
    "LH"  : ("NO", "Norway"),
    "LI"  : ("NO", "Norway"),
    "LJ"  : ("NO", "Norway"),
    "LK"  : ("NO", "Norway"),
    "LL"  : ("NO", "Norway"),
    "LM"  : ("NO", "Norway"),
    "LN"  : ("NO", "Norway"),
    "LO"  : ("AR", "Argentina"),
    "LP"  : ("AR", "Argentina"),
    "LQ"  : ("AR", "Argentina"),
    "LR"  : ("AR", "Argentina"),
    "LS"  : ("AR", "Argentina"),
    "LT"  : ("AR", "Argentina"),
    "LU"  : ("AR", "Argentina"),
    "LV"  : ("AR", "Argentina"),
    "LW"  : ("AR", "Argentina"),
    "LX"  : ("LU", "Luxembourg"),
    "LY"  : ("LT", "Lithuania"),
    "LZ"  : ("BG", "Bulgaria"),
    "M"   : ("GB", "United Kingdom"),
    "N"   : ("US", "United States"),
    "O1"  : ("GE", "South Ossetia"),
    "OA"  : ("PE", "Peru"),
    "OB"  : ("PE", "Peru"),
    "OC"  : ("PE", "Peru"),
    "OD"  : ("LB", "Lebanon"),
    "OE"  : ("AT", "Austria"),
    "OF"  : ("FI", "Finland"),
    "OG"  : ("FI", "Finland"),
    "OH"  : ("FI", "Finland"),
    "OI"  : ("FI", "Finland"),
    "OJ"  : ("FI", "Finland"),
    "OK"  : ("CZ", "Czech Republic"),
    "OL"  : ("CZ", "Czech Republic"),
    "OM"  : ("SK", "Slovakia"),
    "ON"  : ("BE", "Belgium"),
    "OO"  : ("BE", "Belgium"),
    "OP"  : ("BE", "Belgium"),
    "OQ"  : ("BE", "Belgium"),
    "OR"  : ("BE", "Belgium"),
    "OS"  : ("BE", "Belgium"),
    "OT"  : ("BE", "Belgium"),
    "OU"  : ("DK", "Denmark"),
    "OV"  : ("DK", "Denmark"),
    "OW"  : ("DK", "Denmark"),
    "OX"  : ("DK", "Denmark"),
    "OY"  : ("DK", "Denmark"),
    "OZ"  : ("DK", "Denmark"),
    "P2"  : ("PG", "Papua New Guinea"),
    "P3"  : ("CY", "Cyprus"),
    "P4"  : ("AW", "Aruba"),
    "P5"  : ("KP", "North Korea"),
    "P6"  : ("KP", "North Korea"),
    "P7"  : ("KP", "North Korea"),
    "P8"  : ("KP", "North Korea"),
    "P9"  : ("KP", "North Korea"),
    "PA"  : ("NL", "Netherlands"),
    "PB"  : ("NL", "Netherlands"),
    "PC"  : ("NL", "Netherlands"),
    "PD"  : ("NL", "Netherlands"),
    "PE"  : ("NL", "Netherlands"),
    "PF"  : ("NL", "Netherlands"),
    "PG"  : ("NL", "Netherlands"),
    "PH"  : ("NL", "Netherlands"),
    "PI"  : ("NL", "Netherlands"),
    "PJ"  : ("AN", "Antilles (Netherlands)"),
    "PK"  : ("ID", "Indonesia"),
    "PL"  : ("ID", "Indonesia"),
    "PM"  : ("ID", "Indonesia"),
    "PN"  : ("ID", "Indonesia"),
    "PO"  : ("ID", "Indonesia"),
    "PP"  : ("BR", "Brazil"),
    "PQ"  : ("BR", "Brazil"),
    "PR"  : ("BR", "Brazil"),
    "PS"  : ("BR", "Brazil"),
    "PT"  : ("BR", "Brazil"),
    "PU"  : ("BR", "Brazil"),
    "PV"  : ("BR", "Brazil"),
    "PW"  : ("BR", "Brazil"),
    "PX"  : ("BR", "Brazil"),
    "PY"  : ("BR", "Brazil"),
    "PZ"  : ("SR", "Suriname"),
    "RA6" : ("RU", "Chechnya"),
    "R"   : ("RU", "Russia"),
    "S0"  : ("EH", "Western Sahara"),
    "S2"  : ("BD", "Bangladesh"),
    "S3"  : ("BD", "Bangladesh"),
    "S5"  : ("SI", "Slovenia"),
    "S6"  : ("SG", "Singapore"),
    "S7"  : ("SC", "Seychelles"),
    "S8"  : ("ZA", "South Africa"),
    "S9"  : ("ST", "Sao Tome and Principe"),
    "SA"  : ("SE", "Sweden"),
    "SB"  : ("SE", "Sweden"),
    "SC"  : ("SE", "Sweden"),
    "SD"  : ("SE", "Sweden"),
    "SE"  : ("SE", "Sweden"),
    "SF"  : ("SE", "Sweden"),
    "SG"  : ("SE", "Sweden"),
    "SH"  : ("SE", "Sweden"),
    "SI"  : ("SE", "Sweden"),
    "SJ"  : ("SE", "Sweden"),
    "SK"  : ("SE", "Sweden"),
    "SL"  : ("SE", "Sweden"),
    "SM"  : ("SE", "Sweden"),
    "SN"  : ("PL", "Poland"),
    "SO"  : ("PL", "Poland"),
    "SP"  : ("PL", "Poland"),
    "SQ"  : ("PL", "Poland"),
    "SR"  : ("PL", "Poland"),
    "SSA" : ("EG", "Egypt"),
    "SSB" : ("EG", "Egypt"),
    "SSC" : ("EG", "Egypt"),
    "SSD" : ("EG", "Egypt"),
    "SSE" : ("EG", "Egypt"),
    "SSF" : ("EG", "Egypt"),
    "SSG" : ("EG", "Egypt"),
    "SSH" : ("EG", "Egypt"),
    "SSI" : ("EG", "Egypt"),
    "SSJ" : ("EG", "Egypt"),
    "SSK" : ("EG", "Egypt"),
    "SSL" : ("EG", "Egypt"),
    "SSM" : ("EG", "Egypt"),
    "SS"  : ("SD", "Sudan"),
    "ST"  : ("SD", "Sudan"),
    "SU"  : ("EG", "Egypt"),
    "SV"  : ("GR", "Greece"),
    "SW"  : ("GR", "Greece"),
    "SX"  : ("GR", "Greece"),
    "SY"  : ("GR", "Greece"),
    "SZ"  : ("GR", "Greece"),
    "T0"  : (None, "Principality of Seborga"),
    "T1"  : ("MD", "Transnistria"),
    "T2"  : ("TV", "Tuvalu"),
    "T3"  : ("KI", "Kiribati"),
    "T4"  : ("CU", "Cuba"),
    "T5"  : ("SO", "Somalia"),
    "T6"  : ("AF", "Afghanistan"),
    "T7"  : ("SM", "San Marino"),
    "T8"  : ("PW", "Palau"),
    "TA"  : ("TR", "Turkey"),
    "TB"  : ("TR", "Turkey"),
    "TC"  : ("TR", "Turkey"),
    "TD"  : ("GT", "Guatemala"),
    "TE"  : ("CR", "Costa Rica"),
    "TF"  : ("IS", "Iceland"),
    "TG"  : ("GT", "Guatemala"),
    "TH"  : ("FR", "France"),
    "TI"  : ("CR", "Costa Rica"),
    "TJ"  : ("CM", "Cameroon"),
    "TK"  : ("FR", "France"),
    "TL"  : ("CF", "Central African Republic"),
    "TM"  : ("FR", "France"),
    "TN"  : ("CG", "Republic of Congo"),
    "TO"  : ("FR", "France"),
    "TP"  : ("FR", "France"),
    "TQ"  : ("FR", "France"),
    "TR"  : ("GA", "Gabon"),
    "TS"  : ("TN", "Tunisia"),
    "TT"  : ("TD", "Chad"),
    "TU"  : ("CI", "Ivory Coast"),
    "TV"  : ("FR", "France"),
    "TW"  : ("FR", "France"),
    "TX"  : ("FR", "France"),
    "TY"  : ("BJ", "Benin"),
    "TZ"  : ("ML", "Mali"),
    "UA"  : ("RU", "Russia"),
    "UB"  : ("RU", "Russia"),
    "UC"  : ("RU", "Russia"),
    "UD"  : ("RU", "Russia"),
    "UE"  : ("RU", "Russia"),
    "UF"  : ("RU", "Russia"),
    "UG"  : ("RU", "Russia"),
    "UH"  : ("RU", "Russia"),
    "UI"  : ("RU", "Russia"),
    "UJ"  : ("UZ", "Uzbekistan"),
    "UK"  : ("UZ", "Uzbekistan"),
    "UL"  : ("UZ", "Uzbekistan"),
    "UM"  : ("UZ", "Uzbekistan"),
    "UN"  : ("KZ", "Kazakhstan"),
    "UO"  : ("KZ", "Kazakhstan"),
    "UP"  : ("KZ", "Kazakhstan"),
    "UQ"  : ("KZ", "Kazakhstan"),
    "UR"  : ("UA", "Ukraine"),
    "US"  : ("UA", "Ukraine"),
    "UT"  : ("UA", "Ukraine"),
    "UU"  : ("UA", "Ukraine"),
    "UV"  : ("UA", "Ukraine"),
    "UW"  : ("UA", "Ukraine"),
    "UX"  : ("UA", "Ukraine"),
    "UY"  : ("UA", "Ukraine"),
    "UZ"  : ("UA", "Ukraine"),
    "V2"  : ("AG", "Antigua and Barbuda"),
    "V3"  : ("BZ", "Belize"),
    "V4"  : ("KN", "Saint Kitts and Nevis"),
    "V5"  : ("NA", "Namibia"),
    "V6"  : ("FM", "Micronesia"),
    "V7"  : ("MH", "Marshall Islands"),
    "V8"  : ("BN", "Brunei"),
    "VA"  : ("CA", "Canada"),
    "VB"  : ("CA", "Canada"),
    "VC"  : ("CA", "Canada"),
    "VD"  : ("CA", "Canada"),
    "VE"  : ("CA", "Canada"),
    "VF"  : ("CA", "Canada"),
    "VG"  : ("CA", "Canada"),
    "VH"  : ("AU", "Australia"),
    "VI"  : ("AU", "Australia"),
    "VJ"  : ("AU", "Australia"),
    "VK"  : ("AU", "Australia"),
    "VL"  : ("AU", "Australia"),
    "VM"  : ("AU", "Australia"),
    "VN"  : ("AU", "Australia"),
    "VO"  : ("CA", "Newfoundland (Canada)"),
    "VP"  : ("GB", "United Kingdom"),
    "VQ"  : ("GB", "United Kingdom"),
    "VR"  : ("HK", "Hong Kong"),
    "VS"  : ("GB", "United Kingdom"),
    "VT"  : ("IN", "India"),
    "VU"  : ("IN", "India"),
    "VV"  : ("IN", "India"),
    "VW"  : ("IN", "India"),
    "VX"  : ("CA", "Canada"),
    "VY"  : ("CA", "Canada"),
    "VZ"  : ("AU", "Australia"),
    "W"   : ("US", "United States"),
    "XA"  : ("MX", "Mexico"),
    "XB"  : ("MX", "Mexico"),
    "XC"  : ("MX", "Mexico"),
    "XD"  : ("MX", "Mexico"),
    "XE"  : ("MX", "Mexico"),
    "XF"  : ("MX", "Mexico"),
    "XG"  : ("MX", "Mexico"),
    "XH"  : ("MX", "Mexico"),
    "XI"  : ("MX", "Mexico"),
    "XJ"  : ("CA", "Canada"),
    "XK"  : ("CA", "Canada"),
    "XL"  : ("CA", "Canada"),
    "XM"  : ("CA", "Canada"),
    "XN"  : ("CA", "Canada"),
    "XO"  : ("CA", "Canada"),
    "XP"  : ("DK", "Denmark"),
    "XQ"  : ("CL", "Chile"),
    "XR"  : ("CL", "Chile"),
    "XS"  : ("CN", "China"),
    "XT"  : ("BF", "Burkina Faso"),
    "XU"  : ("KH", "Cambodia"),
    "XV"  : ("VN", "Vietnam"),
    "XW"  : ("LA", "Laos"),
    "XX"  : ("MO", "Macao"),
    "XY"  : ("MM", "Myanmar"),
    "XZ"  : ("MM", "Myanmar"),
    "Y2"  : ("DE", "Germany"),
    "Y3"  : ("DE", "Germany"),
    "Y4"  : ("DE", "Germany"),
    "Y5"  : ("DE", "Germany"),
    "Y6"  : ("DE", "Germany"),
    "Y7"  : ("DE", "Germany"),
    "Y8"  : ("DE", "Germany"),
    "Y9"  : ("DE", "Germany"),
    "YA"  : ("AF", "Afghanistan"),
    "YB"  : ("ID", "Indonesia"),
    "YC"  : ("ID", "Indonesia"),
    "YD"  : ("ID", "Indonesia"),
    "YE"  : ("ID", "Indonesia"),
    "YF"  : ("ID", "Indonesia"),
    "YG"  : ("ID", "Indonesia"),
    "YH"  : ("ID", "Indonesia"),
    "YI"  : ("IQ", "Iraq"),
    "YJ"  : ("VU", "Vanuatu"),
    "YK"  : ("SY", "Syria"),
    "YL"  : ("LV", "Latvia"),
    "YM"  : ("TR", "Turkey"),
    "YN"  : ("NI", "Nicaragua"),
    "YO"  : ("RO", "Romania"),
    "YP"  : ("RO", "Romania"),
    "YQ"  : ("RO", "Romania"),
    "YR"  : ("RO", "Romania"),
    "YS"  : ("SV", "El Salvador"),
    "YT"  : ("RS", "Serbia"),
    "YU"  : ("RS", "Serbia"),
    "YV"  : ("VE", "Venezuela"),
    "YW"  : ("VE", "Venezuela"),
    "YX"  : ("VE", "Venezuela"),
    "YY"  : ("VE", "Venezuela"),
    "Z2"  : ("ZW", "Zimbabwe"),
    "Z3"  : ("MK", "Macedonia"),
    "Z6"  : ("XK", "Kosovo"),
    "Z8"  : ("SS", "South Sudan"),
    "ZA"  : ("AL", "Albania"),
    "ZB"  : ("GB", "United Kingdom"),
    "ZC"  : ("GB", "United Kingdom"),
    "ZD"  : ("GB", "United Kingdom"),
    "ZE"  : ("GB", "United Kingdom"),
    "ZF"  : ("GB", "United Kingdom"),
    "ZG"  : ("GB", "United Kingdom"),
    "ZH"  : ("GB", "United Kingdom"),
    "ZI"  : ("GB", "United Kingdom"),
    "ZJ"  : ("GB", "United Kingdom"),
    "ZK"  : ("NZ", "New Zealand"),
    "ZL"  : ("NZ", "New Zealand"),
    "ZM"  : ("NZ", "New Zealand"),
    "ZN"  : ("GB", "United Kingdom"),
    "ZO"  : ("GB", "United Kingdom"),
    "ZP"  : ("PY", "Paraguay"),
    "ZQ"  : ("GB", "United Kingdom"),
    "ZR"  : ("ZA", "South Africa"),
    "ZS"  : ("ZA", "South Africa"),
    "ZT"  : ("ZA", "South Africa"),
    "ZU"  : ("ZA", "South Africa"),
    "ZV"  : ("BR", "Brazil"),
    "ZW"  : ("BR", "Brazil"),
    "ZX"  : ("BR", "Brazil"),
    "ZY"  : ("BR", "Brazil"),
    "ZZ"  : ("BR", "Brazil"),
}

#
# Mapping from AIS MID prefixes to countries
#

MID2COUNTRY = {
    "002" : (None, "Europe Coast Station"),
    "003" : (None, "North America Coast Station"),
    "004" : (None, "Asia Coast Station"),
    "005" : (None, "Oceania Coast Station"),
    "006" : (None, "Africa Coast Station"),
    "007" : (None, "South America Coast Station"),
    "501" : ("FR", "Adelie Land (France)"),
    "401" : ("AF", "Afghanistan"),
    "303" : ("US", "Alaska (USA)"),
    "201" : ("AL", "Albania"),
    "605" : ("DZ", "Algeria"),
    "559" : ("AS", "Samoa (USA)"),
    "202" : ("AD", "Andorra"),
    "603" : ("AO", "Angola"),
    "301" : ("AI", "Anguilla"),
    "304" : ("AG", "Antigua and Barbuda"),
    "305" : ("AG", "Antigua and Barbuda"),
    "701" : ("AR", "Argentina"),
    "216" : ("AM", "Armenia"),
    "307" : ("AW", "Aruba"),
    "608" : ("GB", "Ascension Island"),
    "503" : ("AU", "Australia"),
    "203" : ("AT", "Austria"),
    "423" : ("AZ", "Azerbaijan"),
    "204" : ("PT", "Azores (Portugal)"),
    "308" : ("BS", "Bahamas"),
    "309" : ("BS", "Bahamas"),
    "311" : ("BS", "Bahamas"),
    "408" : ("BH", "Bahrain"),
    "405" : ("BD", "Bangladesh"),
    "314" : ("BB", "Barbados"),
    "206" : ("BY", "Belarus"),
    "205" : ("BE", "Belgium"),
    "312" : ("BZ", "Belize"),
    "610" : ("BJ", "Benin"),
    "310" : ("BM", "Bermuda"),
    "410" : ("BT", "Bhutan"),
    "720" : ("BO", "Bolivia"),
    "306" : ("NL", "Bonaire, Sint Eustatius and Saba (Netherlands)"),
    "478" : ("BA", "Bosnia and Herzegovina"),
    "611" : ("BW", "Botswana"),
    "710" : ("BR", "Brazil"),
    "378" : ("VG", "Virgin Islands (UK)"),
    "508" : ("BN", "Brunei"),
    "207" : ("BG", "Bulgaria"),
    "633" : ("BF", "Burkina Faso"),
    "609" : ("BI", "Burundi"),
    "514" : ("KH", "Cambodia"),
    "515" : ("KH", "Cambodia"),
    "613" : ("CM", "Cameroon"),
    "316" : ("CA", "Canada"),
    "617" : ("CV", "Cape Verde"),
    "319" : ("KY", "Cayman Islands"),
    "612" : ("CF", "Central African Republic"),
    "670" : ("TD", "Chad"),
    "725" : ("CL", "Chile"),
    "412" : ("CN", "China"),
    "413" : ("CN", "China"),
    "414" : ("CN", "China"),
    "516" : ("CX", "Christmas Island"),
    "523" : ("CC", "Cocos (Keeling) Islands"),
    "730" : ("CO", "Colombia"),
    "616" : ("KM", "Comoros"),
    "620" : ("KM", "Comoros"),
    "615" : ("CG", "Republic of Congo"),
    "518" : ("CK", "Cook Islands"),
    "321" : ("CR", "Costa Rica"),
    "619" : ("CI", "Ivory Coast"),
    "238" : ("HR", "Croatia"),
    "618" : ("FR", "Crozet Archipelago"),
    "323" : ("CU", "Cuba"),
    "306" : ("CW", "Curacao (Netherlands)"),
    "209" : ("CY", "Cyprus"),
    "210" : ("CY", "Cyprus"),
    "212" : ("CY", "Cyprus"),
    "270" : ("CZ", "Czech Republic"),
    "445" : ("KP", "North Korea"),
    "676" : ("CD", "Republic of Congo"),
    "219" : ("DK", "Denmark"),
    "220" : ("DK", "Denmark"),
    "621" : ("DJ", "Djibouti"),
    "325" : ("DM", "Dominica"),
    "327" : ("DO", "Dominican Republic"),
    "735" : ("EC", "Ecuador"),
    "622" : ("EG", "Egypt"),
    "359" : ("SV", "El Salvador"),
    "631" : ("GQ", "Equatorial Guinea"),
    "625" : ("ER", "Eritrea"),
    "276" : ("EE", "Estonia"),
    "624" : ("ET", "Ethiopia"),
    "740" : ("FK", "Falkland Islands (Malvinas)"),
    "231" : ("FO", "Faroe Islands"),
    "520" : ("FJ", "Fiji"),
    "230" : ("FI", "Finland"),
    "226" : ("FR", "France"),
    "227" : ("FR", "France"),
    "228" : ("FR", "France"),
    "546" : ("PF", "French Polynesia"),
    "626" : ("GA", "Gabon"),
    "629" : ("GM", "Gambia"),
    "213" : ("GE", "Georgia"),
    "211" : ("DE", "Germany"),
    "218" : ("DE", "Germany (former DDR)"),
    "627" : ("GH", "Ghana"),
    "236" : ("GI", "Gibraltar"),
    "237" : ("GR", "Greece"),
    "239" : ("GR", "Greece"),
    "240" : ("GR", "Greece"),
    "241" : ("GR", "Greece"),
    "331" : ("GL", "Greenland"),
    "330" : ("GD", "Grenada"),
    "329" : ("FR", "Guadeloupe (France)"),
    "332" : ("GT", "Guatemala"),
    "745" : ("FR", "Guiana (France)"),
    "632" : ("GN", "Guinea"),
    "630" : ("GW", "Guinea-Bissau"),
    "750" : ("GY", "Guyana"),
    "336" : ("HT", "Haiti"),
    "334" : ("HN", "Honduras"),
    "477" : ("HK", "Hong Kong"),
    "243" : ("HU", "Hungary"),
    "251" : ("IS", "Iceland"),
    "419" : ("IN", "India"),
    "525" : ("ID", "Indonesia"),
    "422" : ("IR", "Iran"),
    "425" : ("IQ", "Iraq"),
    "250" : ("IE", "Ireland"),
    "428" : ("IL", "Israel"),
    "247" : ("IT", "Italy"),
    "339" : ("JM", "Jamaica"),
    "431" : ("JP", "Japan"),
    "432" : ("JP", "Japan"),
    "438" : ("JO", "Jordan"),
    "436" : ("KZ", "Kazakhstan"),
    "634" : ("KE", "Kenya"),
    "635" : ("FR", "Kerguelen Islands"),
    "529" : ("KI", "Kiribati"),
    "440" : ("KR", "South Korea"),
    "441" : ("KR", "South Korea"),
    "447" : ("KW", "Kuwait"),
    "451" : ("KG", "Kyrgyzstan"),
    "531" : ("LA", "Laos"),
    "275" : ("LV", "Latvia"),
    "450" : ("LB", "Lebanon"),
    "644" : ("LS", "Lesotho"),
    "636" : ("LR", "Liberia"),
    "637" : ("LR", "Liberia"),
    "642" : ("LY", "Libya"),
    "252" : ("LI", "Liechtenstein"),
    "277" : ("LT", "Lithuania"),
    "253" : ("LU", "Luxembourg"),
    "453" : ("MO", "Macao"),
    "274" : ("MK", "Macedonia"),
    "647" : ("MG", "Madagascar"),
    "255" : ("PT", "Madeira (Portugal)"),
    "655" : ("MW", "Malawi"),
    "533" : ("MY", "Malaysia"),
    "455" : ("MV", "Maldives"),
    "649" : ("ML", "Mali"),
    "215" : ("MT", "Malta"),
    "229" : ("MT", "Malta"),
    "248" : ("MT", "Malta"),
    "249" : ("MT", "Malta"),
    "256" : ("MT", "Malta"),
    "538" : ("MH", "Marshall Islands"),
    "347" : ("FR", "Martinique (France)"),
    "654" : ("MR", "Mauritania"),
    "645" : ("MU", "Mauritius"),
    "345" : ("MX", "Mexico"),
    "510" : ("FM", "Micronesia"),
    "214" : ("MD", "Moldova"),
    "254" : ("MC", "Monaco"),
    "457" : ("MN", "Mongolia"),
    "262" : ("ME", "Montenegro"),
    "348" : ("MS", "Montserrat"),
    "242" : ("MA", "Morocco"),
    "650" : ("MZ", "Mozambique"),
    "506" : ("MM", "Myanmar"),
    "659" : ("NA", "Namibia"),
    "544" : ("NR", "Nauru"),
    "459" : ("NP", "Nepal"),
    "244" : ("NL", "Netherlands"),
    "245" : ("NL", "Netherlands"),
    "246" : ("NL", "Netherlands"),
    "540" : ("NC", "New Caledonia"),
    "512" : ("NZ", "New Zealand"),
    "350" : ("NI", "Nicaragua"),
    "656" : ("NE", "Niger"),
    "657" : ("NG", "Nigeria"),
    "542" : ("NU", "Niue"),
    "536" : ("MP", "Northern Mariana Islands (USA)"),
    "257" : ("NO", "Norway"),
    "258" : ("NO", "Norway"),
    "259" : ("NO", "Norway"),
    "461" : ("OM", "Oman"),
    "463" : ("PK", "Pakistan"),
    "511" : ("PW", "Palau"),
    "443" : ("PS", "Palestine"),
    "351" : ("PA", "Panama"),
    "352" : ("PA", "Panama"),
    "353" : ("PA", "Panama"),
    "354" : ("PA", "Panama"),
    "355" : ("PA", "Panama"),
    "356" : ("PA", "Panama"),
    "357" : ("PA", "Panama"),
    "370" : ("PA", "Panama"),
    "371" : ("PA", "Panama"),
    "372" : ("PA", "Panama"),
    "373" : ("PA", "Panama"),
    "374" : ("PA", "Panama"),
    "553" : ("PG", "Papua New Guinea"),
    "755" : ("PY", "Paraguay"),
    "760" : ("PE", "Peru"),
    "548" : ("PH", "Philippines"),
    "555" : ("PN", "Pitcairn Islands"),
    "261" : ("PL", "Poland"),
    "263" : ("PT", "Portugal"),
    "358" : ("PR", "Puerto Rico (USA)"),
    "466" : ("QA", "Qatar"),
    "660" : ("RE", "Reunion (France)"),
    "264" : ("RO", "Romania"),
    "273" : ("RU", "Russia"),
    "661" : ("RW", "Rwanda"),
    "665" : ("SH", "Saint Helena"),
    "341" : ("KN", "Saint Kitts and Nevis"),
    "343" : ("LC", "Saint Lucia"),
    "607" : ("FR", "Saint Paul and Amsterdam Islands"),
    "361" : ("VC", "Saint Pierre and Miquelon"),
    "375" : ("PM", "Saint Vincent and Grenadines"),
    "376" : ("PM", "Saint Vincent and Grenadines"),
    "377" : ("PM", "Saint Vincent and Grenadines"),
    "561" : ("WS", "Samoa"),
    "268" : ("SM", "San Marino"),
    "668" : ("ST", "Sao Tome and Principe"),
    "403" : ("SA", "Saudi Arabia"),
    "663" : ("SN", "Senegal"),
    "279" : ("RS", "Serbia"),
    "664" : ("SC", "Seychelles"),
    "667" : ("SL", "Sierra Leone"),
    "563" : ("SG", "Singapore"),
    "564" : ("SG", "Singapore"),
    "565" : ("SG", "Singapore"),
    "566" : ("SG", "Singapore"),
    "306" : ("SX", "Sint Maarten (Netherlands)"),
    "267" : ("SK", "Slovakia"),
    "278" : ("SK", "Slovenia"),
    "557" : ("SB", "Solomon Islands"),
    "666" : ("SO", "Somalia"),
    "601" : ("ZA", "South Africa"),
    "224" : ("ES", "Spain"),
    "225" : ("ES", "Spain"),
    "417" : ("LK", "Sri Lanka"),
    "638" : ("SS", "South Sudan"),
    "662" : ("SD", "Sudan"),
    "765" : ("SR", "Suriname"),
    "669" : ("SZ", "Swaziland"),
    "265" : ("SE", "Sweden"),
    "266" : ("SE", "Sweden"),
    "269" : ("CH", "Switzerland"),
    "468" : ("SY", "Syria"),
    "416" : ("TW", "Taiwan"),
    "472" : ("TJ", "Tajikistan"),
    "674" : ("TZ", "Tanzania"),
    "677" : ("TZ", "Tanzania"),
    "567" : ("TH", "Thailand"),
    "671" : ("TG", "Togo"),
    "570" : ("TO", "Tonga"),
    "362" : ("TT", "Trinidad and Tobago"),
    "672" : ("TN", "Tunisia"),
    "271" : ("TR", "Turkey"),
    "434" : ("TM", "Turkmenistan"),
    "364" : ("TC", "Turks and Caicos Islands"),
    "572" : ("TV", "Tuvalu"),
    "675" : ("UG", "Uganda"),
    "272" : ("UA", "Ukraine"),
    "470" : ("AE", "United Arab Emirates"),
    "471" : ("AE", "United Arab Emirates"),
    "232" : ("GB", "United Kingdom"),
    "233" : ("GB", "United Kingdom"),
    "234" : ("GB", "United Kingdom"),
    "235" : ("GB", "United Kingdom"),
    "379" : ("VI", "Virgin Islands (USA)"),
    "338" : ("US", "United States"),
    "366" : ("US", "United States"),
    "367" : ("US", "United States"),
    "368" : ("US", "United States"),
    "369" : ("US", "United States"),
    "770" : ("UY", "Uruguay"),
    "437" : ("UZ", "Uzbekistan"),
    "576" : ("VU", "Vanuatu"),
    "577" : ("VU", "Vanuatu"),
    "208" : ("VA", "Vatican"),
    "775" : ("VE", "Venezuela"),
    "574" : ("VN", "Vietnam"),
    "578" : ("WF", "Wallis and Futuna Islands"),
    "473" : ("YE", "Yemen"),
    "475" : ("YE", "Yemen"),
    "678" : ("ZM", "Zambia"),
    "679" : ("ZW", "Zimbabwe"),
}

#
# Mapping from country codes to countries
#

CCODE2COUNTRY = {
    "ZW" : "Zimbabwe",              "ZWE" : "Zimbabwe",
    "ZM" : "Zambia",                "ZMB" : "Zambia",
    "YE" : "Yemen",                 "YEM" : "Yemen",
    "EH" : "Western Sahara",        "ESH" : "Western Sahara",
    "WF" : "Wallis and Futuna",     "WLF" : "Wallis and Futuna",
    "VI" : "U.S. Virgin Islands",   "VIR" : "U.S. Virgin Islands",
    "VN" : "Vietnam",               "VNM" : "Vietnam",
    "VE" : "Venezuela",             "VEN" : "Venezuela",
    "VA" : "Vatican",               "VAT" : "Vatican",
    "VU" : "Vanuatu",               "VUT" : "Vanuatu",
    "UZ" : "Uzbekistan",            "UZB" : "Uzbekistan",
    "US" : "United States",         "USA" : "United States",
    "UY" : "Uruguay",               "URY" : "Uruguay",
    "UA" : "Ukraine",               "UKR" : "Ukraine",
    "GB" : "United Kingdom",        "GBR" : "United Kingdom",
    "UG" : "Uganda",                "UGA" : "Uganda",
    "AE" : "United Arab Emirates",  "ARE" : "United Arab Emirates",
    "TV" : "Tuvalu",                "TUV" : "Tuvalu",
    "TC" : "Turks and Caicos Islands", "TCA" : "Turks and Caicos Islands",
    "TM" : "Turkmenistan",          "TKM" : "Turkmenistan",
    "TR" : "Turkey",                "TUR" : "Turkey",
    "TN" : "Tunisia",               "TUN" : "Tunisia",
    "TT" : "Trinidad and Tobago",   "TTO" : "Trinidad and Tobago",
    "TO" : "Tonga",                 "TON" : "Tonga",
    "TK" : "Tokelau",               "TKL" : "Tokelau",
    "TG" : "Togo",                  "TGO" : "Togo",
    "TH" : "Thailand",              "THA" : "Thailand",
    "TZ" : "Tanzania",              "TZA" : "Tanzania",
    "TJ" : "Tajikistan",            "TJK" : "Tajikistan",
    "TW" : "Taiwan",                "TWN" : "Taiwan",
    "SY" : "Syria",                 "SYR" : "Syria",
    "CH" : "Switzerland",           "CHE" : "Switzerland",
    "SE" : "Sweden",                "SWE" : "Sweden",
    "SZ" : "Swaziland",             "SWZ" : "Swaziland",
    "SJ" : "Svalbard and Jan Mayen", "SJM" : "Svalbard and Jan Mayen",
    "SR" : "Suriname",              "SUR" : "Suriname",
    "SD" : "Sudan",                 "SDN" : "Sudan",
    "VC" : "Saint Vincent and Grenadines", "VCT" : "Saint Vincent and Grenadines",
    "PM" : "Saint Pierre and Miquelon",    "SPM" : "Saint Pierre and Miquelon",
    "MF" : "Saint Martin",          "MAF" : "Saint Martin",
    "LC" : "Saint Lucia",           "LCA" : "Saint Lucia",
    "KN" : "Saint Kitts and Nevis", "KNA" : "Saint Kitts and Nevis",
    "SH" : "Saint Helena",          "SHN" : "Saint Helena",
    "LK" : "Sri Lanka",             "LKA" : "Sri Lanka",
    "ES" : "Spain",                 "ESP" : "Spain",
    "SS" : "South Sudan",           "SSD" : "South Sudan",
    "KR" : "South Korea",           "KOR" : "South Korea",
    "ZA" : "South Africa",          "ZAF" : "South Africa",
    "SO" : "Somalia",               "SOM" : "Somalia",
    "SB" : "Solomon Islands",       "SLB" : "Solomon Islands",
    "SI" : "Slovenia",              "SVN" : "Slovenia",
    "SK" : "Slovakia",              "SVK" : "Slovakia",
    "SX" : "Sint Maarten",          "SXM" : "Sint Maarten",
    "SG" : "Singapore",             "SGP" : "Singapore",
    "SL" : "Sierra Leone",          "SLE" : "Sierra Leone",
    "SC" : "Seychelles",            "SYC" : "Seychelles",
    "RS" : "Serbia",                "SRB" : "Serbia",
    "SN" : "Senegal",               "SEN" : "Senegal",
    "SA" : "Saudi Arabia",          "SAU" : "Saudi Arabia",
    "ST" : "Sao Tome and Principe", "STP" : "Sao Tome and Principe",
    "SM" : "San Marino",            "SMR" : "San Marino",
    "WS" : "Samoa",                 "WSM" : "Samoa",
    "BL" : "Saint Barthelemy",      "BLM" : "Saint Barthelemy",
    "RW" : "Rwanda",                "RWA" : "Rwanda",
    "RU" : "Russia",                "RUS" : "Russia",
    "RO" : "Romania",               "ROU" : "Romania",
    "RE" : "Reunion",               "REU" : "Reunion",
    "QA" : "Qatar",                 "QAT" : "Qatar",
    "PR" : "Puerto Rico",           "PRI" : "Puerto Rico",
    "PT" : "Portugal",              "PRT" : "Portugal",
    "PL" : "Poland",                "POL" : "Poland",
    "PN" : "Pitcairn",              "PCN" : "Pitcairn",
    "PH" : "Philippines",           "PHL" : "Philippines",
    "PE" : "Peru",                  "PER" : "Peru",
    "PY" : "Paraguay",              "PRY" : "Paraguay",
    "PG" : "Papua New Guinea",      "PNG" : "Papua New Guinea",
    "PA" : "Panama",                "PAN" : "Panama",
    "PS" : "Palestine",             "PSE" : "Palestine",
    "PW" : "Palau",                 "PLW" : "Palau",
    "PK" : "Pakistan",              "PAK" : "Pakistan",
    "OM" : "Oman",                  "OMN" : "Oman",
    "NO" : "Norway",                "NOR" : "Norway",
    "KP" : "North Korea",           "PRK" : "North Korea",
    "MP" : "Northern Mariana Islands", "MNP" : "Northern Mariana Islands",
    "NU" : "Niue",                  "NIU" : "Niue",
    "NG" : "Nigeria",               "NGA" : "Nigeria",
    "NE" : "Niger",                 "NER" : "Niger",
    "NI" : "Nicaragua",             "NIC" : "Nicaragua",
    "NZ" : "New Zealand",           "NZL" : "New Zealand",
    "NC" : "New Caledonia",         "NCL" : "New Caledonia",
    "AN" : "Antilles (Netherlands)", "ANT" : "Antilles (Netherlands)",
    "NL" : "Netherlands",           "NLD" : "Netherlands",
    "NP" : "Nepal",                 "NPL" : "Nepal",
    "NR" : "Nauru",                 "NRU" : "Nauru",
    "NA" : "Namibia",               "NAM" : "Namibia",
    "MZ" : "Mozambique",            "MOZ" : "Mozambique",
    "MA" : "Morocco",               "MAR" : "Morocco",
    "MS" : "Montserrat",            "MSR" : "Montserrat",
    "ME" : "Montenegro",            "MNE" : "Montenegro",
    "MN" : "Mongolia",              "MNG" : "Mongolia",
    "MC" : "Monaco",                "MCO" : "Monaco",
    "MD" : "Moldova",               "MDA" : "Moldova",
    "FM" : "Micronesia",            "FSM" : "Micronesia",
    "MX" : "Mexico",                "MEX" : "Mexico",
    "YT" : "Mayotte",               "MYT" : "Mayotte",
    "MU" : "Mauritius",             "MUS" : "Mauritius",
    "MR" : "Mauritania",            "MRT" : "Mauritania",
    "MH" : "Marshall Islands",      "MHL" : "Marshall Islands",
    "MT" : "Malta",                 "MLT" : "Malta",
    "ML" : "Mali",                  "MLI" : "Mali",
    "MV" : "Maldives",              "MDV" : "Maldives",
    "MY" : "Malaysia",              "MYS" : "Malaysia",
    "MW" : "Malawi",                "MWI" : "Malawi",
    "MG" : "Madagascar",            "MDG" : "Madagascar",
    "MK" : "Macedonia",             "MKD" : "Macedonia",
    "MO" : "Macao",                 "MAC" : "Macao",
    "LU" : "Luxembourg",            "LUX" : "Luxembourg",
    "LT" : "Lithuania",             "LTU" : "Lithuania",
    "LI" : "Liechtenstein",         "LIE" : "Liechtenstein",
    "LY" : "Libya",                 "LBY" : "Libya",
    "LR" : "Liberia",               "LBR" : "Liberia",
    "LS" : "Lesotho",               "LSO" : "Lesotho",
    "LB" : "Lebanon",               "LBN" : "Lebanon",
    "LV" : "Latvia",                "LVA" : "Latvia",
    "LA" : "Laos",                  "LAO" : "Laos",
    "KG" : "Kyrgyzstan",            "KGZ" : "Kyrgyzstan",
    "KW" : "Kuwait",                "KWT" : "Kuwait",
    "XK" : "Kosovo",                "XKX" : "Kosovo",
    "KI" : "Kiribati",              "KIR" : "Kiribati",
    "KE" : "Kenya",                 "KEN" : "Kenya",
    "KZ" : "Kazakhstan",            "KAZ" : "Kazakhstan",
    "JO" : "Jordan",                "JOR" : "Jordan",
    "JE" : "Jersey",                "JEY" : "Jersey",
    "JP" : "Japan",                 "JPN" : "Japan",
    "JM" : "Jamaica",               "JAM" : "Jamaica",
    "CI" : "Ivory Coast",           "CIV" : "Ivory Coast",
    "IT" : "Italy",                 "ITA" : "Italy",
    "IL" : "Israel",                "ISR" : "Israel",
    "IM" : "Isle of Man",           "IMN" : "Isle of Man",
    "IE" : "Ireland",               "IRL" : "Ireland",
    "IQ" : "Iraq",                  "IRQ" : "Iraq",
    "IR" : "Iran",                  "IRN" : "Iran",
    "ID" : "Indonesia",             "IDN" : "Indonesia",
    "IN" : "India",                 "IND" : "India",
    "IS" : "Iceland",               "ISL" : "Iceland",
    "HU" : "Hungary",               "HUN" : "Hungary",
    "HK" : "Hong Kong",             "HKG" : "Hong Kong",
    "HN" : "Honduras",              "HND" : "Honduras",
    "HT" : "Haiti",                 "HTI" : "Haiti",
    "GY" : "Guyana",                "GUY" : "Guyana",
    "GW" : "Guinea-Bissau",         "GNB" : "Guinea-Bissau",
    "GN" : "Guinea",                "GIN" : "Guinea",
    "GG" : "Guernsey",              "GGY" : "Guernsey",
    "GT" : "Guatemala",             "GTM" : "Guatemala",
    "GU" : "Guam",                  "GUM" : "Guam",
    "GD" : "Grenada",               "GRD" : "Grenada",
    "GL" : "Greenland",             "GRL" : "Greenland",
    "GR" : "Greece",                "GRC" : "Greece",
    "GI" : "Gibraltar",             "GIB" : "Gibraltar",
    "GH" : "Ghana",                 "GHA" : "Ghana",
    "DE" : "Germany",               "DEU" : "Germany",
    "GE" : "Georgia",               "GEO" : "Georgia",
    "GM" : "Gambia",                "GMB" : "Gambia",
    "GA" : "Gabon",                 "GAB" : "Gabon",
    "PF" : "French Polynesia",      "PYF" : "French Polynesia",
    "FR" : "France",                "FRA" : "France",
    "FI" : "Finland",               "FIN" : "Finland",
    "FJ" : "Fiji",                  "FJI" : "Fiji",
    "FO" : "Faroe Islands",         "FRO" : "Faroe Islands",
    "FK" : "Falkland Islands",      "FLK" : "Falkland Islands",
    "ET" : "Ethiopia",              "ETH" : "Ethiopia",
    "EE" : "Estonia",               "EST" : "Estonia",
    "ER" : "Eritrea",               "ERI" : "Eritrea",
    "GQ" : "Equatorial Guinea",     "GNQ" : "Equatorial Guinea",
    "SV" : "El Salvador",           "SLV" : "El Salvador",
    "EG" : "Egypt",                 "EGY" : "Egypt",
    "EC" : "Ecuador",               "ECU" : "Ecuador",
    "TL" : "East Timor",            "TLS" : "East Timor",
    "DO" : "Dominican Republic",    "DOM" : "Dominican Republic",
    "DM" : "Dominica",              "DMA" : "Dominica",
    "DJ" : "Djibouti",              "DJI" : "Djibouti",
    "DK" : "Denmark",               "DNK" : "Denmark",
    "CZ" : "Czech Republic",        "CZE" : "Czech Republic",
    "CY" : "Cyprus",                "CYP" : "Cyprus",
    "CW" : "Curacao",               "CUW" : "Curacao",
    "CU" : "Cuba",                  "CUB" : "Cuba",
    "HR" : "Croatia",               "HRV" : "Croatia",
    "CR" : "Costa Rica",            "CRI" : "Costa Rica",
    "CK" : "Cook Islands",          "COK" : "Cook Islands",
    "CD" : "Democratic Republic of Congo", "COD" : "Democratic Republic of Congo",
    "CG" : "Republic of Congo",     "COG" : "Republic of Congo",
    "KM" : "Comoros",               "COM" : "Comoros",
    "CO" : "Colombia",              "COL" : "Colombia",
    "CC" : "Cocos Islands",         "CCK" : "Cocos Islands",
    "CX" : "Christmas Island",      "CXR" : "Christmas Island",
    "CN" : "China",                 "CHN" : "China",
    "CL" : "Chile",                 "CHL" : "Chile",
    "TD" : "Chad",                  "TCD" : "Chad",
    "CF" : "Central African Republic", "CAF" : "Central African Republic",
    "KY" : "Cayman Islands",        "CYM" : "Cayman Islands",
    "CV" : "Cape Verde",            "CPV" : "Cape Verde",
    "CA" : "Canada",                "CAN" : "Canada",
    "CM" : "Cameroon",              "CMR" : "Cameroon",
    "KH" : "Cambodia",              "KHM" : "Cambodia",
    "BI" : "Burundi",               "BDI" : "Burundi",
    "MM" : "Myanmar",               "MMR" : "Myanmar",
    "BF" : "Burkina Faso",          "BFA" : "Burkina Faso",
    "BG" : "Bulgaria",              "BGR" : "Bulgaria",
    "BN" : "Brunei",                "BRN" : "Brunei",
    "VG" : "British Virgin Islands", "VGB" : "British Virgin Islands",
    "IO" : "British Indian Ocean Territory", "IOT" : "British Indian Ocean Territory",
    "BR" : "Brazil",                "BRA" : "Brazil",
    "BW" : "Botswana",              "BWA" : "Botswana",
    "BA" : "Bosnia and Herzegovina", "BIH" : "Bosnia and Herzegovina",
    "BO" : "Bolivia",               "BOL" : "Bolivia",
    "BT" : "Bhutan",                "BTN" : "Bhutan",
    "BM" : "Bermuda",               "BMU" : "Bermuda",
    "BJ" : "Benin",                 "BEN" : "Benin",
    "BZ" : "Belize",                "BLZ" : "Belize",
    "BE" : "Belgium",               "BEL" : "Belgium",
    "BY" : "Belarus",               "BLR" : "Belarus",
    "BB" : "Barbados",              "BRB" : "Barbados",
    "BD" : "Bangladesh",            "BGD" : "Bangladesh",
    "BH" : "Bahrain",               "BHR" : "Bahrain",
    "BS" : "Bahamas",               "BHS" : "Bahamas",
    "AZ" : "Azerbaijan",            "AZE" : "Azerbaijan",
    "AT" : "Austria",               "AUT" : "Austria",
    "AU" : "Australia",             "AUS" : "Australia",
    "AW" : "Aruba",                 "ABW" : "Aruba",
    "AM" : "Armenia",               "ARM" : "Armenia",
    "AR" : "Argentina",             "ARG" : "Argentina",
    "AG" : "Antigua and Barbuda",   "ATG" : "Antigua and Barbuda",
    "AQ" : "Antarctica",            "ATA" : "Antarctica",
    "AI" : "Anguilla",              "AIA" : "Anguilla",
    "AO" : "Angola",                "AGO" : "Angola",
    "AD" : "Andorra",               "AND" : "Andorra",
    "AS" : "American Samoa",        "ASM" : "American Samoa",
    "DZ" : "Algeria",               "DZA" : "Algeria",
    "AL" : "Albania",               "ALB" : "Albania",
    "AF" : "Afghanistan",           "AFG" : "Afghanistan",
}
