from owrx.config import Config
from owrx.bookmarks import Bookmark
from owrx.web import WebAgent
from datetime import datetime

import threading
import logging
import re
import math

logger = logging.getLogger(__name__)

#
# Maximal distance on Earth surface (kilometers)
#
MAX_DISTANCE = 25000


class EIBI(WebAgent):
    sharedInstance = None
    creationLock = threading.Lock()

    @staticmethod
    def getSharedInstance():
        with EIBI.creationLock:
            if EIBI.sharedInstance is None:
                EIBI.sharedInstance = EIBI("eibi.json")
        return EIBI.sharedInstance

    @staticmethod
    def start():
        EIBI.getSharedInstance().startThread()

    @staticmethod
    def stop():
        EIBI.getSharedInstance().stopThread()

    # Offset frequency for proper tuning
    @staticmethod
    def correctFreq(freq: int, mode: str) -> int:
        if mode == "cw":
            # This is now done on the client side
            #return freq - 800
            return freq
        elif mode == "fax":
            return freq - 1900
        elif mode in ["cwdecoder", "rtty450", "rtty170", "rtty85", "sitorb", "navtex", "dsc", "bpsk31", "bpsk63"]:
            return freq - 1000
        else:
            return freq

    # Compute distance, in kilometers, between two latlons.
    @staticmethod
    def distKm(p1, p2):
        # Earth radius in km
        earthR = 6371
        # Convert degrees to radians
        rlat1 = p1[0] * (math.pi/180)
        rlat2 = p2[0] * (math.pi/180)
        # Compute difference in radians
        difflat = rlat2 - rlat1
        difflon = (p2[1] - p1[1]) * (math.pi/180)
        # Compute distance
        return round(2 * earthR * math.asin(math.sqrt(
            math.sin(difflat/2) * math.sin(difflat/2) +
            math.cos(rlat1) * math.cos(rlat2) * math.sin(difflon/2) * math.sin(difflon/2)
        )))

    @staticmethod
    def getDescription(b):
        description = []
        # Describe target region
        if "tgt" in b:
            target = b["tgt"]
            # Add country name if ITU code present
            if "itu" in b and b["itu"] in EIBI_Countries:
                source = EIBI_Countries[b["itu"]]
            else:
                source = target
            description += [(
                "Broadcasting to " + target +
                ((" from " + source) if source != target else "") +
                "."
            )]
        # Describe transmitter locations
        if "src" in b and b["src"] in EIBI_Locations:
            pm = Config.get()
            rxPos = (pm["receiver_gps"]["lat"], pm["receiver_gps"]["lon"])
            description += [(
                "Transmitting from " +
                ", ".join([
                    "{0} ({1}km)".format(x["name"], EIBI.distKm(rxPos, (x["lat"], x["lon"])))
                    for x in EIBI_Locations[b["src"]]
                ]) + "."
            )]
        # Done
        return " ".join(description)

    def __init__(self, dataName: str):
        super().__init__(dataName)
        self.patternCSV = re.compile(r"^([\d\.]+);(\d\d\d\d)-(\d\d\d\d);(\S*);(\S+);(.*);(.*);(.*);(.*);(\d+);(.*);(.*)$")
        self.patternDays = re.compile(r"^(.*)(Mo|Tu|We|Th|Fr|Sa|Su)-(Mo|Tu|We|Th|Fr|Sa|Su)(.*)$")

    # Find all current broadcasts for a given source
    def findBySource(self, src: str):
        # Get entries active at the current time
        now = datetime.utcnow()
        now = now.hour * 100 + now.minute
        result = []
        # Search for entries originating from given source at current time
        with self.lock:
            for entry in self.data:
                if entry["time1"] <= now and entry["time2"] > now:
                    if entry["src"] == src:
                        result.append(entry)
        # Done
        return result

    # Find all current broadcasts for a given frequency range
    def findCurrent(self, freq1: int, freq2: int):
        # Get entries active at the current time
        now = datetime.utcnow()
        now = now.hour * 100 + now.minute
        return self.find(freq1, freq2, now, now)

    # Find all broadcasts for given frequency and time ranges
    def find(self, freq1: int, freq2: int, time1: int, time2: int):
        result = []
        # Search for entries within given frequency and time ranges
        with self.lock:
            for entry in self.data:
                f = entry["freq"]
                if f >= freq1 and f <= freq2:
                    if entry["time1"] <= time2 and entry["time2"] > time1:
                        result.append(entry)
        # Done
        return result

    # Create list of currently broadcasting locations
    def currentTransmitters(self, hours: int = 1):
        # Get entries active at the current time + 1 hour
        ts   = datetime.now().timestamp()
        now  = datetime.utcnow()
        day  = now.weekday()
        date = now.year * 10000 + now.month * 100 + now.day
        t1   = now.hour * 100 + now.minute
        t2   = t1 + hours * 100
        result = {}
        # Search for current entries
        with self.lock:
            for entry in self.data:
                try:
                    # Check if entry is currently active
                    entryActive = (
                        entry["days"][day] != "."
                    and (entry["date1"] == 0 or entry["date1"] <= date)
                    and (entry["date2"] == 0 or entry["date2"] >= date)
                    )
                    # Check the hours, rolling over to the next day
                    if entryActive:
                        e1 = entry["time1"]
                        e2 = entry["time2"]
                        e2 = e2 if e2 > e1 else e2 + 2400
                        entryActive = e1 < t2 and e2 > t1
                    # For every currently active schedule entry...
                    if entryActive:
                        src = entry["src"]
                        if src not in EIBI_Locations:
                            # Warn if location not found
                            # @@@ Too much output here
                            #logger.debug("Location '{0}' for '{1}' not found!".format(src, entry["name"]))
                            pass
                        else:
                            # Compute TTL for the entry
                            ttl = ts + (
                                ((e2 // 100) - (t1 // 100)) * 3600 +
                                ((e2 % 100)  - (t1 % 100)) * 60
                            )
                            # Find all matching transmitter locations
                            for loc in EIBI_Locations[src]:
                                name = loc["name"]
                                if name not in result:
                                    # Add location to the result
                                    result[name] = loc.copy()
                                    result[name]["schedule"] = [ entry ]
                                    result[name]["ttl"] = ttl
                                else:
                                    # Add schedule entry, update TTL
                                    result[name]["schedule"].append(entry)
                                    result[name]["ttl"] = max(ttl, result[name]["ttl"]);

                except Exception as e:
                    logger.error("currentTransmitters() exception: {0}".format(e))

        # Done
        return result

    # Create list of current bookmarks for a frequency range
    def currentBookmarks(self, frequencyRange, hours: int = 0, rangeKm: int = MAX_DISTANCE):
        # Make sure freq2>freq1
        (f1, f2) = frequencyRange
        if f1>f2:
            f = f1
            f1 = f2
            f2 = f

        # Get entries active at the current time + 1 hour
        now  = datetime.utcnow()
        day  = now.weekday()
        date = now.year * 10000 + now.month * 100 + now.day
        t1   = now.hour * 100 + now.minute
        t2   = t1 + hours * 100

        # Get receiver location for computing distance
        pm = Config.get()
        rxPos = (pm["receiver_gps"]["lat"], pm["receiver_gps"]["lon"])

        # No result yet
        result = {}

        logger.info("Creating bookmarks for {0}-{1}kHz within {2}km...".format(f1//1000, f2//1000, rangeKm))

        # Search for current entries
        with self.lock:
            for entry in self.data:
                try:
                    # No distance or duration yet
                    dist = MAX_DISTANCE
                    duration = 10000

                    # Check if entry active and within frequency range
                    f = entry["freq"]
                    entryActive = (
                        f1 <= f <= f2 and entry["days"][day] != "."
                    and (entry["date1"] == 0 or entry["date1"] <= date)
                    and (entry["date2"] == 0 or entry["date2"] >= date)
                    )

                    # Check the hours, rolling over to the next day
                    if entryActive:
                        e1 = entry["time1"]
                        e2 = entry["time2"]
                        e2 = e2 if e2 > e1 else e2 + 2400
                        entryActive = e1 < t2 and e2 > t1
                        duration = e2 - e1

                    # Find closest transmitter for this entry
                    if entryActive:
                        src = entry["src"]
                        if src not in EIBI_Locations:
                            # Warn if location not found
                            # @@@ Too much output here
                            #logger.debug("Location '{0}' for '{1}' not found!".format(src, entry["name"]))
                            pass
                        else:
                            # Find closest source
                            for loc in EIBI_Locations[src]:
                                txPos = (loc["lat"], loc["lon"])
                                dist  = min(dist, EIBI.distKm(rxPos, txPos))
                            # Prefer closer transmitters, apply range
                            entryActive = ((dist <= rangeKm) and (
                                (f not in result) or (dist < result[f][1])
                            ))

                    # Add entry to the result
                    if entryActive:
                        #if f in result:
                        #    logger.debug("Replacing '{0}' ({1}km) with '{2}' ({3}km)".format(
                        #        result[f][0]["name"], result[f][1], entry["name"], dist
                        #    ))
                        result[f] = ( entry, dist, duration )

                except Exception as e:
                    logger.error("currentBookmarks() exception: {0}".format(e))

        logger.info("Created {0} bookmarks for {1}-{2}kHz within {3}km.".format(len(result), f1//1000, f2//1000, rangeKm))

        # Return bookmarks for all found entries
        return [ Bookmark({
            "name"        : result[f][0]["name"],
            "modulation"  : result[f][0]["mode"],
            "frequency"   : EIBI.correctFreq(f, result[f][0]["mode"]),
            "description" : EIBI.getDescription(result[f][0])
        }, srcFile = "EIBI") for f in result.keys() ]

    def convertDate(self, date: str):
        # No-date is a common case
        if date == "":
            return 0
        # Remove last-seen data
        date = re.sub(r"\[.*\]", "", date)
        # Match day/month
        m = re.match(r"^(\d\d)(\d\d)$", date)
        if m is None:
            return 0
        else:
            now   = datetime.utcnow()
            month = int(m.group(2))
            day   = int(m.group(1))
            year  = (
                now.year + 1 if (now.month >= 11) and (month < now.month) else
                now.year - 1 if (now.month <= 3)  and (month > now.month) else
                now.year
            )
            return year * 10000 + month * 100 + day

    def convertDays(self, days: str):
        # Look up and process special cases
        if days in EIBI_SpecialDays:
            return EIBI_SpecialDays[days]
        # Start with empty result
        result = [ ".", ".", ".", ".", ".", ".", "."]
        # Extract day spans
        m = self.patternDays.match(days)
        if m is not None:
            x = EIBI_Days[m.group(2)]
            y = EIBI_Days[m.group(3)]
            result[y - 1] = str(y)
            while x != y:
                result[x - 1] = str(x)
                x = x + 1 if x < 7 else 1
            # Remove extracted span
            days = m.group(1) + m.group(4)
        # Extract singular days
        for day in EIBI_Days.keys():
            if day in days:
                x = EIBI_Days[day]
                result[x - 1] = str(x)
        # Done
        return "".join(result)

    def _loadFromWeb(self):
        return self.loadFromWeb("http://www.eibispace.de/dx/sked-{0}.csv")

    def loadFromWeb(self, url: str):
        # Figure out CSV file name based on the current date
        # SUMMER: Apr - Oct - sked-aNN.csv
        # WINTER: Nov - Mar - sked-bNN.csv
        now = datetime.utcnow()
        url = url.format(
            ("a" if now.month >= 4 and now.month <= 10 else "b") +
            str((now.year if now.month >= 4 else now.year - 1) % 100)
        )

        # Fetch and parse CSV file
        result = []
        try:
            logger.info("Scraping '{0}'...".format(url))
            for line in self._openUrl(url).readlines():
                # Convert read bytes to a string
                line = line.decode('cp1252').rstrip()

                # When we encounter a location...
                m = self.patternCSV.match(line)
                if m is not None:
                    freq = int(float(m.group(1)) * 1000)
                    days = m.group(4)
                    name = m.group(6).lower()
                    lang = m.group(7)
                    trgt = m.group(8)

                    # Guess modulation, default to AM
                    mode = (
                        "hfdl"    if lang == "-HF"     else
                        "rtty450" if lang == "-TY"     else
                        "cw"      if lang == "-CW"     else
                        "usb"     if days == "USB"     else
                        "lsb"     if days == "LSB"     else
                        "hfdl"    if "hfdl"    in name else # HFDL
                        "drm"     if "digital" in name else # DRM
                        "fax"     if " fax"    in name else # Weather FAX
                        "rtty450" if "rtty"    in name else # Weather RTTY
                        "usb"     if "volmet"  in name else # Weather
                        "usb"     if "cross "  in name else # Weather
                        "usb"     if " ldoc"   in name else # Aircraft
                        "usb"     if " car-"   in name else # Aircraft
                        "usb"     if " nat-"   in name else # Aircraft
                        "usb"     if " usb"    in name else
                        "usb"     if "fsk"     in name else
                        "usb"     if freq < 4800000    else # Services
                        "am")

                    # Convert language code to language
                    if lang in EIBI_Languages:
                        lang = EIBI_Languages[lang]["name"]

                    # Convert target country code to target country
                    if trgt in EIBI_Countries:
                        trgt = EIBI_Countries[trgt]
                    elif len(trgt)==3 and trgt[0] in EIBI_Dirs and trgt[1:] in EIBI_Countries:
                        trgt = EIBI_Dirs[trgt[0]] + " " + EIBI_Countries[trgt[1:]]

                    # Convert ITU/SRC pair to a complete location code
                    itu = m.group(5)
                    src = m.group(9)
                    if len(src) == 0:
                        # Primary site
                        src = itu
                    elif src.startswith("/"):
                        # Transmitter located elsewhere
                        src = src[1:]
                    else:
                        # Country ITU -dash- site code
                        src = itu + "-" + src

                    # Append a new entry to the result
                    result.append({
                        "freq"  : freq,
                        "mode"  : mode,
                        "time1" : int(m.group(2)),
                        "time2" : int(m.group(3)),
                        "days"  : self.convertDays(days),
                        "itu"   : itu,
                        "name"  : m.group(6),
                        "lang"  : lang,
                        "tgt"   : trgt,
                        "src"   : src,
                        "pers"  : int(m.group(10)),
                        "date1" : self.convertDate(m.group(11)),
                        "date2" : self.convertDate(m.group(12)),
                    })

        except Exception as e:
            logger.error("loadFromWeb() exception: {0}".format(e))
            return None

        # Done
        return result


#
# Normal days of the week
#
EIBI_Days = {
  "Mo" : 1,
  "Tu" : 2,
  "We" : 3,
  "Th" : 4,
  "Fr" : 5,
  "Sa" : 6,
  "Su" : 7,
}

#
# Special Codes for the days field
#
EIBI_SpecialDays = {
  ""     : "1234567", # Empty field means whole week
  "LSB"  : "1234567", # Upper side band transmission
  "USB"  : "1234567", # Upper side band transmission
  "alt"  : "xxxxxxx", # Alternative frequency, usually not in use
  "irr"  : "xxxxxxx", # Irregular operation
  "Haj"  : "xxxxxxx", # Special Haj broadcast
  "Ram"  : "xxxxxxx", # Special Ramadan schedule
  "tent" : "xxxxxxx", # Tentatively, check and report your observations
  "test" : "xxxxxxx", # Test operation, may cease at any time
  "harm" : ".......", # Harmonic signal (multiples of fundamental frequency)
  "imod" : ".......", # Intermodulation signal
}

#
# Region and Country Codes
#
EIBI_Dirs = {
  "C" : "Central",
  "W" : "Western",
  "E" : "Eastern",
  "S" : "South",
  "N" : "North",
}

EIBI_Countries = {
  # Regions
  "Af" : "Africa",
  "Am" : "America",
  "AO" : "Atlantic Ocean",
  "As" : "Asia",
  "C..": "Central ..",
  "Car": "Caribbean, Gulf of Mexico, Florida Waters",
  "Cau": "Caucasus",
  "CIS": "Commonwealth of Independent States (former Soviet Union)",
  "E..": "East ..",
  "EA" : "East Asia",
  "EE" : "East Europe",
  "Eu" : "Europe, incl. North Africa / Middle East",
  "FE" : "Far East",
  "Glo": "World",
  "In" : "Indian Subcontinent",
  "IO" : "Indian Ocean",
  "LAm": "Latin America",
  "ME" : "Middle East",
  "N..": "North ..",
  "NA" : "North America",
  "NE" : "Northeast",
  "NW" : "Northwest",
  "Oc" : "Oceania (Australia, New Zealand, Pacific Ocean)",
  "S..": "South ..",
  "SA" : "South America",
  "SE" : "Southeast",
  "Sib": "Siberia",
  "SW" : "Southwest",
  "Tib": "Tibet",
  "W..": "West ..",
  "WE" : "West Europe",
  # ITU codes start here
  "ABW": "Aruba",
  "AFG": "Afghanistan",
  "AFS": "South Africa",
  "AGL": "Angola",
  "AIA": "Anguilla",
  "ALB": "Albania",
  "ALG": "Algeria",
  "ALS": "Alaska",
  "AMS": "Saint Paul & Amsterdam Is.",
  "AND": "Andorra",
  "AOE": "Western Sahara",
  "ARG": "Argentina",
  "ARM": "Armenia",
  "ARS": "Saudi Arabia",
  "ASC": "Ascension Island",
  "ATA": "Antarctica",
  "ATG": "Antigua and Barbuda",
  "ATN": "Netherlands Leeward Antilles (dissolved in 2010)",
  "AUS": "Australia",
  "AUT": "Austria",
  "AZE": "Azerbaijan",
  "AZR": "Azores",
  "B":   "Brasil",
  "BAH": "Bahamas",
  "BDI": "Burundi",
  "BEL": "Belgium",
  "BEN": "Benin",
  "BER": "Bermuda",
  "BES": "Bonaire, St Eustatius, Saba (Dutch islands in the Caribbean)",
  "BFA": "Burkina Faso",
  "BGD": "Bangla Desh",
  "BHR": "Bahrain",
  "BIH": "Bosnia-Herzegovina",
  "BIO": "Chagos Is. (Diego Garcia) (British Indian Ocean Territory)",
  "BLM": "Saint-Barthelemy",
  "BLR": "Belarus",
  "BLZ": "Belize",
  "BOL": "Bolivia",
  "BOT": "Botswana",
  "BRB": "Barbados",
  "BRM": "Burma (Myanmar)",
  "BRU": "Brunei Darussalam",
  "BTN": "Bhutan",
  "BUL": "Bulgaria",
  "BVT": "Bouvet",
  "CAB": "Cabinda",
  "CAF": "Central African Republic",
  "CAN": "Canada",
  "CBG": "Cambodia",
  "CEU": "Ceuta",
  "CG7": "Guantanamo Bay",
  "CHL": "Chile",
  "CHN": "People's Republic of China",
  "CHR": "Christmas Island in Indian Ocean",
  "CKH": "Cook Island",
  "CLA": "Clandestine stations",
  "CLM": "Colombia",
  "CLN": "Sri Lanka",
  "CME": "Cameroon",
  "CNR": "Canary Islands",
  "COD": "Democratic Republic of Congo (capital Kinshasa)",
  "COG": "Republic of Congo (capital Brazzaville)",
  "COM": "Comores",
  "CPT": "Clipperton",
  "CPV": "Cape Verde Islands",
  "CRO": "Crozet Archipelago",
  "CTI": "Ivory Coast (Côte d'Ivoire)",
  "CTR": "Costa Rica",
  "CUB": "Cuba",
  "CUW": "Curacao",
  "CVA": "Vatican State",
  "CYM": "Cayman Islands",
  "CYP": "Cyprus",
  "CZE": "Czech Republic",
  "D":   "Germany",
  "DJI": "Djibouti",
  "DMA": "Dominica",
  "DNK": "Denmark",
  "DOM": "Dominican Republic",
  "E":   "Spain",
  "EGY": "Egypt",
  "EQA": "Ecuador",
  "ERI": "Eritrea",
  "EST": "Estonia",
  "ETH": "Ethiopia",
  "EUR": "Iles Europe & Bassas da India",
  "F":   "France",
  "FIN": "Finland",
  "FJI": "Fiji",
  "FLK": "Falkland Islands",
  "FRO": "Faroe Islands",
  "FSM": "Federated States of Micronesia",
  "G":   "United Kingdom of Great Britain and Northern Ireland",
  "GAB": "Gabon",
  "GEO": "Georgia",
  "GHA": "Ghana",
  "GIB": "Gibraltar",
  "GLP": "Guadeloupe",
  "GMB": "Gambia",
  "GNB": "Guinea-Bissau",
  "GNE": "Equatorial Guinea",
  "GPG": "Galapagos",
  "GRC": "Greece",
  "GRD": "Grenada",
  "GRL": "Greenland",
  "GTM": "Guatemala",
  "GUF": "French Guyana",
  "GUI": "Guinea",
  "GUM": "Guam / Guahan",
  "GUY": "Guyana",
  "HKG": "Hong Kong",
  "HMD": "Heard & McDonald Islands",
  "HND": "Honduras",
  "HNG": "Hungary",
  "HOL": "Netherlands",
  "HRV": "Croatia",
  "HTI": "Haiti",
  "HWA": "Hawaii",
  "HWL": "Howland & Baker",
  "I":   "Italy",
  "ICO": "Cocos (Keeling) Island",
  "IND": "India",
  "INS": "Indonesia",
  "IRL": "Ireland",
  "IRN": "Iran",
  "IRQ": "Iraq",
  "ISL": "Iceland",
  "ISR": "Israel",
  "IW":  "International Waters",
  "IWA": "Ogasawara (Bonin, Iwo Jima)",
  "J":   "Japan",
  "JAR": "Jarvis Island",
  "JDN": "Juan de Nova",
  "JMC": "Jamaica",
  "JMY": "Jan Mayen",
  "JON": "Johnston Island",
  "JOR": "Jordan",
  "JUF": "Juan Fernandez Island",
  "KAL": "Kaliningrad",
  "KAZ": "Kazakhstan",
  "KEN": "Kenya",
  "KER": "Kerguelen",
  "KGZ": "Kyrgyzstan",
  "KIR": "Kiribati",
  "KNA": "St Kitts and Nevis",
  "KOR": "Korea, South (Republic)",
  "KOS": "Kosovo",
  "KRE": "Korea, North (Democratic People's Republic)",
  "KWT": "Kuwait",
  "LAO": "Laos",
  "LBN": "Lebanon",
  "LBR": "Liberia",
  "LBY": "Libya",
  "LCA": "Saint Lucia",
  "LIE": "Liechtenstein",
  "LSO": "Lesotho",
  "LTU": "Lithuania",
  "LUX": "Luxembourg",
  "LVA": "Latvia",
  "MAC": "Macao",
  "MAF": "St Martin",
  "MAU": "Mauritius",
  "MCO": "Monaco",
  "MDA": "Moldova",
  "MDG": "Madagascar",
  "MDR": "Madeira",
  "MDW": "Midway Islands",
  "MEL": "Melilla",
  "MEX": "Mexico",
  "MHL": "Marshall Islands",
  "MKD": "Macedonia (F.Y.R.)",
  "MLA": "Malaysia",
  "MLD": "Maldives",
  "MLI": "Mali",
  "MLT": "Malta",
  "MNE": "Montenegro",
  "MNG": "Mongolia",
  "MOZ": "Mozambique",
  "MRA": "Northern Mariana Islands",
  "MRC": "Morocco",
  "MRN": "Marion & Prince Edward Islands",
  "MRT": "Martinique",
  "MSR": "Montserrat",
  "MTN": "Mauritania",
  "MWI": "Malawi",
  "MYA": "Myanmar (Burma)",
  "MYT": "Mayotte",
  "NCG": "Nicaragua",
  "NCL": "New Caledonia",
  "NFK": "Norfolk Island",
  "NGR": "Niger",
  "NIG": "Nigeria",
  "NIU": "Niue",
  "NMB": "Namibia",
  "NOR": "Norway",
  "NPL": "Nepal",
  "NRU": "Nauru",
  "NZL": "New Zealand",
  "OCE": "French Polynesia",
  "OMA": "Oman",
  "PAK": "Pakistan",
  "PAQ": "Easter Island",
  "PHL": "Philippines",
  "PHX": "Phoenix Is.",
  "PLM": "Palmyra Island",
  "PLW": "Palau",
  "PNG": "Papua New Guinea",
  "PNR": "Panama",
  "POL": "Poland",
  "POR": "Portugal",
  "PRG": "Paraguay",
  "PRU": "Peru",
  "PRV": "Okino-Tori-Shima (Parece Vela)",
  "PSE": "Palestine",
  "PTC": "Pitcairn",
  "PTR": "Puerto Rico",
  "QAT": "Qatar",
  "REU": "La Réunion",
  "ROD": "Rodrigues",
  "ROU": "Romania",
  "RRW": "Rwanda",
  "RUS": "Russian Federation",
  "S":   "Sweden",
  "SAP": "San Andres & Providencia",
  "SDN": "Sudan",
  "SEN": "Senegal",
  "SEY": "Seychelles",
  "SGA": "South Georgia Islands",
  "SHN": "Saint Helena",
  "SLM": "Solomon Islands",
  "SLV": "El Salvador",
  "SMA": "Samoa (American)",
  "SMO": "Samoa",
  "SMR": "San Marino",
  "SNG": "Singapore",
  "SOK": "South Orkney Islands",
  "SOM": "Somalia",
  "SPM": "Saint Pierre et Miquelon",
  "SRB": "Serbia",
  "SRL": "Sierra Leone",
  "SSD": "South Sudan",
  "SSI": "South Sandwich Islands",
  "STP": "Sao Tome & Principe",
  "SUI": "Switzerland",
  "SUR": "Suriname",
  "SVB": "Svalbard",
  "SVK": "Slovakia",
  "SVN": "Slovenia",
  "SWZ": "Swaziland",
  "SXM": "Sint Maarten",
  "SYR": "Syria",
  "TCA": "Turks and Caicos Islands",
  "TCD": "Tchad",
  "TGO": "Togo",
  "THA": "Thailand",
  "TJK": "Tajikistan",
  "TKL": "Tokelau",
  "TKM": "Turkmenistan",
  "TLS": "Timor-Leste",
  "TON": "Tonga",
  "TRC": "Tristan da Cunha",
  "TRD": "Trinidad and Tobago",
  "TUN": "Tunisia",
  "TUR": "Turkey",
  "TUV": "Tuvalu",
  "TWN": "Taiwan",
  "TZA": "Tanzania",
  "UAE": "United Arab Emirates",
  "UGA": "Uganda",
  "UKR": "Ukraine",
  "UN":  "United Nations",
  "URG": "Uruguay",
  "USA": "United States of America",
  "UZB": "Uzbekistan",
  "VCT": "Saint Vincent and the Grenadines",
  "VEN": "Venezuela",
  "VIR": "American Virgin Islands",
  "VRG": "British Virgin Islands",
  "VTN": "Vietnam",
  "VUT": "Vanuatu",
  "WAK": "Wake Island",
  "WAL": "Wallis and Futuna",
  "XBY": "Abyei area",
  "XGZ": "Gaza strip",
  "XSP": "Spratly Islands",
  "XUU": "Unidentified",
  "XWB": "West Bank",
  "YEM": "Yemen",
  "ZMB": "Zambia",
  "ZWE": "Zimbabwe"
}

#
# Languages
#
EIBI_Languages = {
  "-CW": { "name": "Morse Station" },
  "-EC": { "name": "Empty Carrier" },
  "-HF": { "name": "HFDL Squitter (Aircraft comms station)" },
  "-MX": { "name": "Music" },
  "-TS": { "name": "Time Signal Station" },
  "-TY": { "name": "Teletype or other digital Station (RTTY/SITOR/..)" },
  "A":   { "name": "Arabic (300m)", "code": "arb" },
  "AB":  { "name": "Abkhaz: Georgia-Abkhazia (0.1m)", "code": "abk" },
  "AC":  { "name": "Aceh: Indonesia-Sumatera (3m)", "code": "ace" },
  "ACH": { "name": "Achang / Ngac'ang: Myanmar, South China (60,000)", "code": "acn" },
  "AD":  { "name": "Adygea / Adyghe / Circassian: Russia-Caucasus (0.5m)", "code": "ady" },
  "ADI": { "name": "Adi: India-Assam,Arunachal Pr. (0.1m)", "code": "adi" },
  "AF":  { "name": "Afrikaans: South Africa, Namibia (5m)", "code": "afr" },
  "AFA": { "name": "Afar: Djibouti (0.3m), Ethiopia (0.45m), Eritrea (0.3m)", "code": "aar" },
  "AFG": { "name": "Pashto and Dari (main Afghan languages, see there)" },
  "AH":  { "name": "Amharic: Ethiopia (22m)", "code": "amh" },
  "AJ":  { "name": "Adja / Aja-Gbe: Benin, Togo (0.5m)", "code": "ajg" },
  "AK":  { "name": "Akha: Burma (0.2m), China-Yunnan (0.13m)", "code": "ahk" },
  "AKL": { "name": "Aklanon: Philippines-Visayas (0.5m)", "code": "akl" },
  "AL":  { "name": "Albanian: Albania (Tosk)(3m), Macedonia / Yugoslavia (Gheg)(2m)", "code": "sqi" },
  "ALG": { "name": "Algerian (Arabic): Algeria (28m)", "code": "arq" },
  "AM":  { "name": "Amoy: S China (25m), Taiwan (15m), SoEaAsia (5m); dialect of Minnan", "code": "nan" },
  "AMD": { "name": "Tibetan Amdo (Tibet, Qinghai, Gansu, Sichuan: 2m)", "code": "adx" },
  "Ang": { "name": "Angelus programme of Vatican Radio" },
  "AR":  { "name": "Armenian: Armenia (3m), USA (1m), RUS,GEO,SYR,LBN,IRN,EGY", "code": "hye" },
  "ARO": { "name": "Aromanian / Vlach: Greece, Albania, Macedonia (0.1m)", "code": "rup" },
  "ARU": { "name": "Languages of Arunachal, India (collectively)" },
  "ASS": { "name": "Assamese: India-Assam (13m)", "code": "asm" },
  "ASY": { "name": "Assyrian / Syriac / Neo-Aramaic: Iraq, Iran, Syria (0.2m)", "code": "aii" },
  "ATS": { "name": "Atsi / Zaiwa: Myanmar (13,000), China-Yunnan (70,000)", "code": "atb" },
  "Aud": { "name": "Papal Audience (Vatican Radio)" },
  "AV":  { "name": "Avar: Dagestan, S Russia (0.7m)", "code": "ava" },
  "AW":  { "name": "Awadhi: N&Ce India (3m)", "code": "awa" },
  "AY":  { "name": "Aymara: Bolivia (2m)", "code": "ayr" },
  "AZ":  { "name": "Azeri / Azerbaijani: Azerbaijan (6m)", "code": "azj" },
  "BAD": { "name": "Badaga: India-Tamil Nadu (0.13m)", "code": "bfq" },
  "BAG": { "name": "Bagri: India-Punjab (0.6m), Pakistan (0.2m)", "code": "bgq" },
  "BAI": { "name": "Bai: China-Yunnan (1.2m)", "code": "bca" },
  "BAJ": { "name": "Bajau: Malaysia-Sabah (50,000)", "code": "bdr" },
  "BAL": { "name": "Balinese: Indonesia-Bali (3m)", "code": "ban" },
  "BAN": { "name": "Banjar / Banjarese: Indonesia-Kalimantan (3.5m)", "code": "bjn" },
  "BAO": { "name": "Baoulé: Cote d'Ivoire (2m)", "code": "bci" },
  "BAR": { "name": "Bari: South Sudan (0.4m)", "code": "bfa" },
  "BAS": { "name": "Bashkir / Bashkort: Russia-Bashkortostan (1m)", "code": "bak" },
  "BAY": { "name": "Bayash / Boyash (gypsy dialect of Romanian): Serbia, Croatia (10,000)" },
  "BB":  { "name": "Braj Bhasa / Braj Bhasha / Brij: India-Rajasthan (0.6m)", "code": "bra" },
  "BC":  { "name": "Baluchi: Pakistan (5m)", "code": "bal" },
  "BE":  { "name": "Bengali / Bangla: Bangladesh (110m), India (82m)", "code": "ben" },
  "BED": { "name": "Bedawiyet / Bedawi / Beja: Sudan (1m)", "code": "bej" },
  "BEM": { "name": "Bemba: Zambia (3m)", "code": "bem" },
  "BET": { "name": "Bete / Bété (Guiberoua): Ivory Coast (0.2m)", "code": "bet" },
  "BGL": { "name": "Bagheli: N India (3m)", "code": "bfy" },
  "BH":  { "name": "Bhili: India-Madhya Pradesh, Gujarat (3.3m)", "code": "bhb" },
  "BHN": { "name": "Bahnar: Vietnam (160,000)", "code": "bdq" },
  "BHT": { "name": "Bhatri: India-Chhattisgarh,Maharashtra (0.2m)", "code": "bgw" },
  "BI":  { "name": "Bilen / Bile: Eritrea-Keren (90,000)", "code": "byn" },
  "BID": { "name": "Bidayuh languages: Malaysia-Sarawak (70,000)", "code": "sdo" },
  "BIS": { "name": "Bisaya: Malaysia-Sarawak,Sabah (20,000), Brunei (40,000)", "code": "bsb" },
  "BJ":  { "name": "Bhojpuri / Bihari: India (38m), Nepal (1.7m)", "code": "bho" },
  "BK":  { "name": "Balkarian: Russia-Caucasus (0.3m)", "code": "krc" },
  "BLK": { "name": "Balkan Romani: Bulgaria (0.3m), Serbia (0.1m), Macedonia (0.1m)", "code": "rmn" },
  "BLT": { "name": "Balti: NE Pakistan (0.3m)", "code": "bft" },
  "BM":  { "name": "Bambara / Bamanankan / Mandenkan: Mali (4m)", "code": "bam" },
  "BNA": { "name": "Borana Oromo / Afan Oromo: Ethiopia (4m)", "code": "gax" },
  "BNG": { "name": "Bangala / Mbangala: Central Angola (0.4m)", "code": "mxg" },
  "BNI": { "name": "Baniua / Baniwa: Brazil-Amazonas (6,000)", "code": "bwi" },
  "BNJ": { "name": "Banjari / Banjara / Gormati / Lambadi: India (4m)", "code": "lmn" },
  "BNT": { "name": "Bantawa: Nepal (400,000)", "code": "bap" },
  "BNY": { "name": "Banyumasan dialect of Javanese: western Central Java" },
  "BON": { "name": "Bondo: India-Odisha (9000)", "code": "bfw" },
  "BOR": { "name": "Boro / Bodo: India-Assam,W Bengal (1.3m)", "code": "brx" },
  "BOS": { "name": "Bosnian (derived from Serbocroat): Bosnia-Hercegovina (2m)", "code": "bos" },
  "BR":  { "name": "Burmese / Barma / Myanmar: Myanmar (32m)", "code": "mya" },
  "BRA": { "name": "Brahui: Pakistan (4m), Afghanistan (0.2m)", "code": "brh" },
  "BRB": { "name": "Bariba / Baatonum: Benin (0.5m), Nigeria (0.1m)", "code": "bba" },
  "BRU": { "name": "Bru: Laos (30,000), Vietnam (55,000)", "code": "bru" },
  "BSL": { "name": "Bislama: Vanuatu (10,000)", "code": "bis" },
  "BT":  { "name": "Black Tai / Tai Dam: Vietnam (0.7m)", "code": "blt" },
  "BTK": { "name": "Batak-Toba: Indonesia-Sumatra (2m)", "code": "bbc" },
  "BU":  { "name": "Bulgarian: Bulgaria (6m)", "code": "bul" },
  "BUG": { "name": "Bugis / Buginese: Indonesia-Sulawesi (5m)", "code": "bug" },
  "BUK": { "name": "Bukharian / Bukhori: Israel (50,000), Uzbekistan (10,000)", "code": "bhh" },
  "BUN": { "name": "Bundeli / Bundelkhandi / Bundelkandi: India-Uttar,Madhya Pr. (3m)", "code": "bns" },
  "BUR": { "name": "Buryat: Russia-Buryatia, Lake Baikal (0.4m)", "code": "bxr" },
  "BUY": { "name": "Bouyei / Buyi / Yay: China-Guizhou (2.6m), No.Vietnam. Close to ZH.", "code": "pcc" },
  "BY":  { "name": "Byelorussian / Belarusian: Belarus, Poland, Ukraine (8m)", "code": "bel" },
  "C":   { "name": "Chinese (not further specified)" },
  "CA":  { "name": "Cantonese / Yue: China-Guangdong (50m),Hongkong (6m),Malaysia (1m)", "code": "yue" },
  "CC":  { "name": "Chaochow (dialect of Min-Nan): China-Guangdong (10m), Thailand (1m)", "code": "nan" },
  "CD":  { "name": "Chowdary / Chaudhry / Chodri: India-Gujarat (0.2m)", "code": "cdi" },
  "CEB": { "name": "Cebuano: Philippines (16m)", "code": "ceb" },
  "CH":  { "name": "Chin (not further specified): Myanmar; includes those below a.o." },
  "C-A": { "name": "Chin-Asho: Myanmar-Ayeyarwady,Rakhine (30,000)", "code": "csh" },
  "C-D": { "name": "Chin-Daai: Myanmar-Chin (37,000)", "code": "dao" },
  "C-F": { "name": "Chin-Falam / Halam: Myanmar-Chin, Bangladesh, India (0.1m)", "code": "cfm" },
  "C-H": { "name": "Chin-Haka: Myanmar-Chin (100,000)", "code": "cnh" },
  "CHA": { "name": "Cha'palaa / Chachi: Ecuador-Esmeraldas (10,000)", "code": "cbi" },
  "CHE": { "name": "Chechen: Russia-Chechnya (1.4m)", "code": "che" },
  "CHG": { "name": "Chhattisgarhi: India-Chhattisgarh, Odisha, Bihar (13m)", "code": "hne" },
  "CHI": { "name": "Chitrali / Khowar: NW Pakistan (0.2m)", "code": "khw" },
  "C-K": { "name": "Chin-Khumi: Myanmar-Chin,Rakhine (0.6m)", "code": "cnk" },
  "C-M": { "name": "Chin-Mro: Myanmar-Rakhine,Chin (75,000)", "code": "cmr" },
  "C-O": { "name": "Chin-Thado / Thadou-Kuki: India-Assam, Manipur (0.2m)", "code": "tcz" },
  "CHR": { "name": "Chrau: Vietnam (7,000)", "code": "crw" },
  "CHU": { "name": "Chuwabu: Mozambique (1m)", "code": "chw" },
  "C-T": { "name": "Chin-Tidim: Myanmar-Chin (0.2m), India-Mizoram,Manipur (0.15m)", "code": "ctd" },
  "C-Z": { "name": "Chin-Zomin / Zomi-Chin: Myanmar (60,000), India-Manipur (20,000)", "code": "zom" },
  "CKM": { "name": "Chakma: India-Mizoram,Tripura,Assam (0.2m), Bangladesh (0.15m)", "code": "ccp" },
  "CKW": { "name": "Chokwe: Angola (0.5m), DR Congo (0.5m)", "code": "cjk" },
  "COF": { "name": "Cofan / Cofán: Ecuador-Napo, Colombia (2000)", "code": "con" },
  "COK": { "name": "Cook Islands Maori / Rarotongan: Cook Islands (13,000)", "code": "rar" },
  "CR":  { "name": "Creole / Haitian: Haiti (7m)", "code": "hat" },
  "CRU": { "name": "Chru: Vietnam (19,000)", "code": "cje" },
  "CT":  { "name": "Catalan: Spain (7m), Andorra (31,000)", "code": "cat" },
  "CV":  { "name": "Chuvash: Russia-Chuvashia (1m)", "code": "chv" },
  "CVC": { "name": "Chavacano / Chabacano: Spanish creole in PHL-Mindanao (4m)", "code": "cbk" },
  "CW":  { "name": "Chewa / Chichewa / Nyanja / Chinyanja: Malawi (7m), MOZ (0.6m),ZMB (0.8m)", "code": "nya" },
  "CZ":  { "name": "Czech: Czech Republic (9m)", "code": "ces" },
  "D":   { "name": "German: Germany (80m), Austria, Switzerland, Belgium", "code": "deu" },
  "D-P": { "name": "Lower German (varieties in N.Germany, USA:Pennsylvania Dutch)", "code": "pdc" },
  "DA":  { "name": "Danish: Denmark (5.5m)", "code": "dan" },
  "DAH": { "name": "Dahayia: India" },
  "DAO": { "name": "Dao: Vietnam ethnic group speaking MIE and Kim Mun (0.7m)" },
  "DAR": { "name": "Dargwa / Dargin: Russia-Dagestan (0.5m)", "code": "dar" },
  "DD":  { "name": "Dhodiya / Dhodia: India-Gujarat (150,000)", "code": "dho" },
  "DEC": { "name": "Deccan / Deccani / Desi: India-Maharashtra (13m)", "code": "dcc" },
  "DEG": { "name": "Degar / Montagnard (Vietnam): comprises JR, RAD, BHN, KOH, MNO, STI" },
  "DEN": { "name": "Dendi: Benin (30,000)", "code": "ddn" },
  "DEO": { "name": "Deori: India-Assam (27,000)", "code": "der" },
  "DES": { "name": "Desiya / Deshiya: India-Odisha (50,000)", "code": "dso" },
  "DH":  { "name": "Dhivehi: Maldives (0.3m)", "code": "div" },
  "DI":  { "name": "Dinka: South Sudan (1.4m)", "code": "dip,diw,dik,dib,dks" },
  "DIM": { "name": "Dimasa / Dhimasa: India-Assam: (0.1m)", "code": "dis" },
  "DIT": { "name": "Ditamari: Benin (0.1m)", "code": "tbz" },
  "DO":  { "name": "Dogri (sometimes includes Kangri dialect): N India (4m)", "code": "doi,dgo,him" },
  "DR":  { "name": "Dari / Eastern Farsi: Afghanistan (7m), Pakistan (2m)", "code": "prs" },
  "DU":  { "name": "Dusun: Malaysia-Sabah (0.1m)", "code": "dtp" },
  "DUN": { "name": "Dungan: Kyrgyzstan (40,000)", "code": "dng" },
  "DY":  { "name": "Dyula / Jula: Burkina Faso (1m), Ivory Coast (1.5m), Mali (50,000)", "code": "dyu" },
  "DZ":  { "name": "Dzongkha: Bhutan (0.2m)", "code": "dzo" },
  "E":   { "name": "English: UK (60m), USA (225m), India (200m), others", "code": "eng" },
  "E,F": { "name": "English, French" },
  "E,S": { "name": "English, Spanish" },
  "EC":  { "name": "Eastern Cham: Vietnam (70,000)", "code": "cjm" },
  "EGY": { "name": "Egyptian Arabic: Egypt (52m)", "code": "arz" },
  "EO":  { "name": "Esperanto: Constructed language (2m)", "code": "epo" },
  "ES":  { "name": "Estonian: Estonia (1m)", "code": "ekk" },
  "EWE": { "name": "Ewe / Éwé: Ghana (2m), Togo (1m)", "code": "ewe" },
  "F":   { "name": "French: France (53m), Canada (7m), Belgium (4m), Switzerland (1m)", "code": "fra" },
  "FA":  { "name": "Faroese: Faroe Islands (66,000)", "code": "fao" },
  "FI":  { "name": "Finnish: Finland (5m)", "code": "fin" },
  "FJ":  { "name": "Fijian: Fiji (0.3m)", "code": "fij" },
  "FON": { "name": "Fon / Fongbe: Benin (1.4m)", "code": "fon" },
  "FP":  { "name": "Filipino (based on Tagalog): Philippines (25m)", "code": "fil" },
  "FS":  { "name": "Farsi / Iranian Persian: Iran (45m)", "code": "pes" },
  "FT":  { "name": "Fiote / Vili: Rep. Congo (7000), Gabon (4000)", "code": "vif" },
  "FU":  { "name": "Fulani / Fulfulde: Nigeria (8m), Niger (1m),Burkina Faso (1m)", "code": "fub,fuh,fuq" },
  "FUJ": { "name": "FutaJalon / Pular: Guinea (3m)", "code": "fuf" },
  "Fujian:": { "name": "see TW-Taiwanese" },
  "FUR": { "name": "Fur: Sudan-Darfur (0.5m)", "code": "fvr" },
  "GA":  { "name": "Garhwali: India-Uttarakhand,Himachal Pr. (3m)", "code": "gbm" },
  "GAG": { "name": "Gagauz: Moldova (0.1m)", "code": "gag" },
  "GAR": { "name": "Garo: India-Meghalaya,Assam,Nagaland,Tripura (1m)", "code": "grt" },
  "GD":  { "name": "Greenlandic Inuktikut: Greenland (50,000)", "code": "kal" },
  "GE":  { "name": "Georgian: Georgia (4m)", "code": "kat" },
  "GI":  { "name": "Gilaki: Iran (3m)", "code": "glk" },
  "GJ":  { "name": "Gujari / Gojri: NW India (0.7m), Pakistan (0.3m)", "code": "gju" },
  "GL":  { "name": "Galicic / Gallego: Spain (3m)", "code": "glg" },
  "GM":  { "name": "Gamit: India-Gujarat (0.3m)", "code": "gbl" },
  "GNG": { "name": "Gurung (Eastern and Western): Nepal (0.4m)", "code": "ggn,gvr" },
  "GO":  { "name": "Gorontalo: Indonesia-Sulawesi (1m)", "code": "gor" },
  "GON": { "name": "Gondi: India-Madhya Pr.,Maharashtra (2m)", "code": "gno" },
  "GR":  { "name": "Greek: Greece (10m), Cyprus (0.7m)", "code": "ell" },
  "GU":  { "name": "Gujarati: India-Gujarat,Maharashtra,Rajasthan (46m)", "code": "guj" },
  "GUA": { "name": "Guaraní: Paraguay (5m)", "code": "grn" },
  "GUN": { "name": "Gungbe / Gongbe / Goun: Benin, Nigeria (0.7m)", "code": "guw" },
  "GUR": { "name": "Gurage / Guragena: Ethiopia (0.4m)", "code": "sgw" },
  "GZ":  { "name": "Ge'ez / Geez (liturgic language of Ethiopia)", "code": "gez" },
  "HA":  { "name": "Haussa: Nigeria (19m), Niger (5m), Benin (1m)", "code": "hau" },
  "HAD": { "name": "Hadiya: Ethiopia (1.2m)", "code": "hdy" },
  "HAR": { "name": "Haryanvi /  Bangri / Harayanvi / Hariyanvi: India-Haryana (8m)", "code": "bgc" },
  "HAS": { "name": "Hassinya / Hassaniya: Mauritania (3m)", "code": "mey" },
  "HB":  { "name": "Hebrew: Israel (5m)", "code": "heb" },
  "HD":  { "name": "Hindko (Northern and Southern): Pakistan (3m)", "code": "hnd,hno" },
  "HI":  { "name": "Hindi: India (260m)", "code": "hin" },
  "HIM": { "name": "Himachali languages: India-Himachal Pradesh", "code": "him" },
  "HK":  { "name": "Hakka: South China (26m), Taiwan (3m), Malaysia (1m)", "code": "hak" },
  "Hokkien:": { "name": "see TW-Taiwanese" },
  "HM":  { "name": "Hmong / Miao languages: S China, N Vietnam, N Laos, USA (3m)", "code": "hmn" },
  "HMA": { "name": "Hmar: India-Assam,Manipur,Mizoram (80,000)", "code": "hmr" },
  "HMB": { "name": "Hmong-Blue / Njua: Laos (0.1m), Thailand (60,000)", "code": "hnj" },
  "HMQ": { "name": "Hmong / Miao, Northern Qiandong / Black Hmong: S China (1m)", "code": "hea" },
  "HMW": { "name": "Hmong-White / Daw: Vietnam (1m), Laos (0.2m), S China (0.2m)", "code": "mww" },
  "HN":  { "name": "Hani: China-Yunnan (0.7m)", "code": "hni" },
  "HO":  { "name": "Ho: India-Jharkand,Odisha,W Bengal (1m)", "code": "hoc" },
  "HR":  { "name": "Croatian / Hrvatski: Croatia (4m), BIH (0.5m), Serbia (0.1m)", "code": "hrv" },
  "HRE": { "name": "Hre: Vietnam (0.1m)", "code": "hre" },
  "HU":  { "name": "Hungarian: Hungary (10m), Romania (1.5m), SVK (0.5m), SRB (0.3m)", "code": "hun" },
  "HUI": { "name": "Hui / Huizhou: China-Anhui,Zhejiang (5m)", "code": "czh" },
  "HZ":  { "name": "Hazaragi: Afghanistan (1.8m), Iran (0.3m)", "code": "haz" },
  "I":   { "name": "Italian: Italy (55m), Switzerland (0.5m), San Marino (25,000)", "code": "ita" },
  "IB":  { "name": "Iban: Malaysia-Sarawak (0.7m)", "code": "iba" },
  "IBN": { "name": "Ibanag: Philippines-Luzon (0.5m)", "code": "ibg" },
  "IF":  { "name": "Ifè / Ife: Togo (0.1m), Benin (80,000)", "code": "ife" },
  "IG":  { "name": "Igbo / Ibo: Nigeria (18m)", "code": "ibo" },
  "ILC": { "name": "Ilocano: Philippines (7m)", "code": "ilo" },
  "ILG": { "name": "Ilonggo / Hiligaynon: Philippines-Visayas / Mindanao (9m)", "code": "hil" },
  "IN":  { "name": "Indonesian / Bahasa Indonesia: Indonesia (140m)", "code": "ind" },
  "INU": { "name": "Inuktikut: Canada-Nunavut,N Quebec,Labrador (30,000)", "code": "ike" },
  "IRQ": { "name": "Iraqi Arabic: Iraq (12m), Iran (1m), Syria (2m)", "code": "acm" },
  "IS":  { "name": "Icelandic: Iceland (0.2m)", "code": "isl" },
  "ISA": { "name": "Isan / Northeastern Thai: Thailand (15m)", "code": "tts" },
  "ITA": { "name": "Itawis / Tawit: Philippines-Luzon (0.1m)", "code": "itv" },
  "J":   { "name": "Japanese: Japan (122m)", "code": "jpn" },
  "JAI": { "name": "Jaintia / Pnar / Synteng: India-Meghalaya, Bangladesh (250,000)", "code": "pbv" },
  "JEH": { "name": "Jeh: Vietnam (15,000), Laos (8,000)", "code": "jeh" },
  "JG":  { "name": "Jingpho: see KC-Kachin" },
  "JOR": { "name": "Jordanian Arabic: Jordan (3.5m), Israel / Palestine (2.5m)", "code": "ajp" },
  "JR":  { "name": "Jarai / Giarai / Jra: Vietnam (0.3m)", "code": "jra" },
  "JU":  { "name": "Juba Arabic: South Sudan (60,000)", "code": "pga" },
  "JV":  { "name": "Javanese: Indonesia-Java,Bali (84m)", "code": "jav" },
  "K":   { "name": "Korean: Korea (62m), China-Jilin,Heilongjiang,Liaoning (2m)", "code": "kor" },
  "KA":  { "name": "Karen (unspecified): Myanmar (3m)" },
  "K-G": { "name": "Karen-Geba: Myanmar (40,000)", "code": "kvq" },
  "K-K": { "name": "Karen-Geko / Gekho: Myanmar (17,000)", "code": "ghk" },
  "K-M": { "name": "Manumanaw Karen / Kawyaw / Kayah: Myanmar (10,000)", "code": "kxf" },
  "K-P": { "name": "Karen-Pao / Black Karen / Pa'o: Myanmar (0.5m)", "code": "blk" },
  "K-S": { "name": "Karen-Sgaw / S'gaw: Myanmar (1.3m), Thailand (0.2m)", "code": "ksw" },
  "K-W": { "name": "Karen-Pwo: Myanmar (1m); Northern variant: Thailand (60,000)", "code": "kjp,pww" },
  "KAD": { "name": "Kadazan: Malaysia-Sabah (80,000)", "code": "kzj,dtb" },
  "KAL": { "name": "Kalderash Romani (Dialect of Vlax): Romania (0.2m)", "code": "rmy" },
  "KAB": { "name": "Kabardian: Russia-Caucasus (0.5m), Turkey (1m)", "code": "kbd" },
  "KAM": { "name": "Kambaata: Ethiopia (0.6m)", "code": "ktb" },
  "KAN": { "name": "Kannada: India-Karnataka,Andhra Pr.,Tamil Nadu (40m)", "code": "kan" },
  "KAO": { "name": "Kaonde: Zambia (0.2m)", "code": "kqn" },
  "KAR": { "name": "Karelian: Russia-Karelia (25,000), Finland (10,000)", "code": "krl" },
  "KAT": { "name": "Katu: Vietnam (50,000)", "code": "ktv" },
  "KAU": { "name": "Kau Bru / Kaubru/ Riang: India-Tripura,Mizoram,Assam (77,000)", "code": "ria" },
  "KAY": { "name": "Kayan: Myanmar (0.1m)", "code": "pdu" },
  "KB":  { "name": "Kabyle: Algeria (5m)", "code": "kab" },
  "KBO": { "name": "Kok Borok / Tripuri: India (0.8m)", "code": "trp" },
  "KC":  { "name": "Kachin / Jingpho: Myanmar (0.9m)", "code": "kac" },
  "KG":  { "name": "Kyrgyz /Kirghiz: Kyrgystan (2.5m), China (0.1m)", "code": "kir" },
  "KGU": { "name": "Kalanguya / Kallahan: Philippines-Luzon (0.1m)", "code": "kak" },
  "KH":  { "name": "Khmer: Cambodia (13m), Vietnam (1m)", "code": "khm" },
  "KHA": { "name": "Kham / Khams, Eastern: China-NE Tibet (1.4m)", "code": "khg" },
  "KHM": { "name": "Khmu: Laos (0.6m)", "code": "kjg" },
  "KHR": { "name": "Kharia / Khariya: India-Jharkand (0.2m)", "code": "khr" },
  "KHS": { "name": "Khasi / Kahasi: India-Meghalaya,Assam (0.8m)", "code": "kha" },
  "KHT": { "name": "Khota (India)" },
  "KIM": { "name": "Kimwani: Mozambique (0.1m)", "code": "wmw" },
  "KIN": { "name": "Kinnauri / Kinori: India-Himachal Pr. (65,000)", "code": "kfk" },
  "KiR": { "name": "KiRundi: Burundi (9m)", "code": "run" },
  "KIS": { "name": "Kisili: West Africa (ask TWR)" },
  "KK":  { "name": "KiKongo / Kongo: DR Congo, Angola (8m)", "code": "kng" },
  "KKA": { "name": "Kankana-ey: Philippines-Luzon (0.3m)", "code": "kne" },
  "KKN": { "name": "Kukna: India-Gujarat (0.1m)", "code": "kex" },
  "KKU": { "name": "Korku: India-Madhya Pr.,Maharashtra (1m)", "code": "kfq" },
  "KMB": { "name": "Kimbundu / Mbundu / Luanda: Angola (4m)", "code": "kmb" },
  "KMY": { "name": "Kumyk: Russia-Dagestan (0.4m)", "code": "kum" },
  "KND": { "name": "Khandesi: India-Maharashtra (22,000)", "code": "khn" },
  "KNG": { "name": "Kangri: close to Dogri; India-Himachal Pradesh, Punjab (1m)", "code": "xnr" },
  "KNK": { "name": "KinyaRwanda-KiRundi service of the Voice of America / BBC" },
  "KNU": { "name": "Kanuri: Nigeria (3.2m), Chad (0.1m), Niger (0.4m)", "code": "kau" },
  "KNY": { "name": "Konyak Naga: India-Assam,Nagaland (0.25m)", "code": "nbe" },
  "KOH": { "name": "Koho / Kohor: Vietnam (0.2m)", "code": "kpm" },
  "KOK": { "name": "Kokang Shan: Myanmar (dialect of Shan)" },
  "KOM": { "name": "Komering: Indonesia-Sumatera (0.5m)", "code": "kge" },
  "KON": { "name": "Konkani: India-Maharashtra,Karnataka,Kerala (2.4m)", "code": "knn" },
  "KOR": { "name": "Korambar / Kurumba Kannada: India-Tamil Nadu (0.2m)", "code": "kfi" },
  "KOT": { "name": "Kotokoli / Tem: Togo (0.2m), Benin (0.05m), Ghana (0.05m)", "code": "kdh" },
  "KOY": { "name": "Koya: India-Andhra Pr.,Odisha (0.4m)", "code": "kff" },
  "KPK": { "name": "Karakalpak: W Uzbekistan (0.4m)", "code": "kaa" },
  "KRB": { "name": "Karbi / Mikir / Manchati: India-Assam,Arunachal Pr. (0.4m)", "code": "mjw" },
  "KRI": { "name": "Krio: Sierra Leone (0.5m)", "code": "kri" },
  "KRW": { "name": "KinyaRwanda: Rwanda (7m), Uganda (0.4m), DR Congo (0.2m)", "code": "kin" },
  "KRY": { "name": "Karay-a: Philippines-Visayas (0.4m)", "code": "krj" },
  "KS":  { "name": "Kashmiri: India (5m), Pakistan (0.1m)", "code": "kas" },
  "KT":  { "name": "Kituba (simplified Kikongo): Rep. Congo (1m), DR Congo (4m)", "code": "ktu" },
  "KTW": { "name": "Kotwali (dialect of Bhili): India-Gujarat,Maharshtra", "code": "bhb" },
  "KU":  { "name": "Kurdish: Turkey (15m), Iraq (6.3m), Iran (6.5m), Syria (1m)", "code": "ckb,kmr,sdh" },
  "KuA": { "name": "Kurdish and Arabic" },
  "KuF": { "name": "Kurdish and Farsi" },
  "KUI": { "name": "Kui: India-Odisha,Ganjam,Andhra Pr. (1m)", "code": "kxu" },
  "KUL": { "name": "Kulina: Brazil-Acre (3500)", "code": "cul" },
  "KUM": { "name": "Kumaoni / Kumauni: India-Uttarakhand (2m)", "code": "kfy" },
  "KUN": { "name": "Kunama: Eritrea (0.2m)", "code": "kun" },
  "KUP": { "name": "Kupia / Kupiya: India-Andhra Pr. (6,000)", "code": "key" },
  "KUR": { "name": "Kurukh / Kurux: India-Chhatisgarh,Jharkhand,W.Bengal (2m)", "code": "kru" },
  "KUs": { "name": "Sorani (Central) Kurdish: Iraq (3.5m), Iran (3.3m)", "code": "ckb" },
  "KUT": { "name": "Kutchi: India-Gujarat (0.4m), Pakistan-Sindh (0.2m)", "code": "gjk" },
  "KUV": { "name": "Kuvi: India-Odisha (0.16m)", "code": "kxv" },
  "KVI": { "name": "Kulluvi / Kullu: India-Himachal Pr. (0.1m)", "code": "kfx" },
  "KWA": { "name": "Kwanyama / Kuanyama (dialect of OW): Angola (0.4m), Namibia (0.2m)", "code": "kua" },
  "KYH": { "name": "Kayah: Myanmar (0.15m)", "code": "kyu" },
  "KZ":  { "name": "Kazakh: Kazakhstan (7m), China (1m), Mongolia (0.1m)", "code": "kaz" },
  "L":   { "name": "Latin: Official language of Catholic church", "code": "lat" },
  "LA":  { "name": "Ladino: see SEF" },
  "LAD": { "name": "Ladakhi / Ladak: India-Jammu and Kashmir (0.1m)", "code": "lbj" },
  "LAH": { "name": "Lahu: China (0.3m), Myanmar (0.2m)", "code": "lhu" },
  "LAK": { "name": "Lak: Russia-Dagestan (0.15m)", "code": "lbe" },
  "LAM": { "name": "Lampung: Indonesia-Sumatera (1m)", "code": "abl,ljp" },
  "LAO": { "name": "Lao: Laos (3m)", "code": "lao" },
  "LB":  { "name": "Lun Bawang / Murut: Malaysia-Sarawak (24,000), Indonesia (23,000)", "code": "lnd" },
  "LBN": { "name": "Lebanon Arabic (North Levantine): Lebanon (4m), Syria (9m)", "code": "apc" },
  "LBO": { "name": "Limboo /Limbu: Nepal (0.3m), India-Sikkim,W.Bengal,Assam (40,000)", "code": "lif" },
  "LEP": { "name": "Lepcha: India-Sikkim,W.Bengal (50,000)", "code": "lep" },
  "LEZ": { "name": "Lezgi: Russia-Dagestan (0.4m), Azerbaijan (0.4m)", "code": "lez" },
  "LIM": { "name": "Limba: Sierra Leone (0.3m)", "code": "lia" },
  "LIN": { "name": "Lingala: DR Congo (2m), Rep. Congo (0.1m)", "code": "lin" },
  "LIS": { "name": "Lisu: China-West Yunnan (0.6m), Burma (0.3m)", "code": "lis" },
  "LND": { "name": "Lunda (see LU), in particular its dialect Ndembo: Angola (0.2m)", "code": "lun" },
  "LNG": { "name": "Lungeli Magar (possibly same as MGA?)" },
  "LO":  { "name": "Lomwe / Ngulu: Mocambique (1.5m)", "code": "ngl" },
  "LOK": { "name": "Lokpa / Lukpa / Lupka: Benin (50,000), Togo (14,000)", "code": "dop" },
  "LOZ": { "name": "Lozi / Silozi: Zambia (0.6m), ZWE (70,000), NMB-E Caprivi (30,000)", "code": "loz" },
  "LT":  { "name": "Lithuanian: Lithuania (3m)", "code": "lit" },
  "LTO": { "name": "Oriental Liturgy of Vatican Radio" },
  "LU":  { "name": "Lunda: Zambia (0.5m)", "code": "lun" },
  "LUB": { "name": "Luba: DR Congo-Kasai (6m)", "code": "lua" },
  "LUC": { "name": "Luchazi: Angola (0.4m), Zambia (0.05m)", "code": "lch" },
  "LUG": { "name": "Luganda: Uganda (4m)", "code": "lug" },
  "LUN": { "name": "Lunyaneka / Nyaneka: Angola (0.3m)", "code": "nyk" },
  "LUR": { "name": "Luri, Northern and Southern: Iran (1.5m and 0.9m)", "code": "lrc,luz" },
  "LUV": { "name": "Luvale: Angola (0.5m), Zambia (0.2m)", "code": "lue" },
  "LV":  { "name": "Latvian: Latvia (1.2m)", "code": "lvs" },
  "M":   { "name": "Mandarin (Standard Chinese / Beijing dialect): China (840m)", "code": "cmn" },
  "MA":  { "name": "Maltese: Malta (0.3m)", "code": "mlt" },
  "MAD": { "name": "Madurese / Madura: Indonesia-Java (7m)", "code": "mad" },
  "MAG": { "name": "Maghi / Magahi / Maghai: India-Bihar,Jharkhand (14m)", "code": "mag" },
  "MAI": { "name": "Maithili / Maithali: India-Bihar (30m), Nepal (3m)", "code": "mai" },
  "MAK": { "name": "Makonde: Tanzania (1m), Mozambique (0.4m)", "code": "kde" },
  "MAL": { "name": "Malayalam: India-Kerala (33m)", "code": "mal" },
  "MAM": { "name": "Maay / Mamay / Rahanweyn: Somalia (2m)", "code": "ymm" },
  "MAN": { "name": "Mandenkan (dialect continuum of BM, DY, MLK): Mali, BFA, CTI, GUI", "code": "man" },
  "MAO": { "name": "Maori: New Zealand (60,000)", "code": "mri" },
  "MAR": { "name": "Marathi: India-Maharashtra (72m)", "code": "mar" },
  "MAS": { "name": "Maasai / Massai / Masai: Kenya (0.8m), Tanzania (0.5m)", "code": "mas" },
  "MC":  { "name": "Macedonian: Macedonia (1.4m), Albania (0.1m)", "code": "mkd" },
  "MCH": { "name": "Mavchi / Mouchi / Mauchi / Mawchi: India-Gujarat,Maharashtra (0.1m)", "code": "mke" },
  "MEI": { "name": "Meithei / Manipuri / Meitei: India-Manipur,Assam (1.5m)", "code": "mni" },
  "MEN": { "name": "Mende: Sierra Leone (1.5m)", "code": "men" },
  "MEW": { "name": "Mewari / Mewadi (a Rajasthani variety): India-Rajasthan (5m)", "code": "mtr" },
  "MGA": { "name": "Magar (Western and Eastern): Nepal (0.8m)", "code": "mrd,mgp" },
  "MIE": { "name": "Mien / Iu Mien: S China (0.4m), Vietnam (0.4m)", "code": "ium" },
  "MIS": { "name": "Mising: India-Assam,Arunachal Pr. (0.5m)", "code": "mrg" },
  "MKB": { "name": "Minangkabau: Indonesia-West Sumatra (5.5m)", "code": "min" },
  "MKS": { "name": "Makassar / Makasar: Indonesia-South Sulawesi (2m)", "code": "mak" },
  "MKU": { "name": "Makua / Makhuwa: Mocambique (3m)", "code": "vmw" },
  "ML":  { "name": "Malay / Baku: Malaysia (10m), Singapore (0.4m), Indonesia (5m)", "code": "zsm,zlm" },
  "MLK": { "name": "Malinke / Maninka (We/Ea): Guinea (3m), SEN (0.4m), Mali (0.8m)", "code": "emk,mlq" },
  "MLT": { "name": "Malto / Kumarbhag Paharia: India-Jharkhand (12,000)", "code": "kmj" },
  "MNA": { "name": "Mina / Gen: Togo (0.2m), Benin (0.1m)", "code": "gej" },
  "MNB": { "name": "Manobo / T'duray: Philippines-Mindanao (0.1m)", "code": "mno" },
  "MNE": { "name": "Montenegrin (quite the same as SR): Montenegro (0.2m)", "code": "srp" },
  "MNO": { "name": "Mnong (Ea,Ce,So): Vietnam (90,000), Cambodia (40,000)", "code": "mng,cmo,mnn" },
  "MO":  { "name": "Mongolian: Mongolia (Halh; 2m), China (Peripheral; 3m)", "code": "khk,mvf" },
  "MON": { "name": "Mon: Myanmar-Mon,Kayin (0.7m), Thailand (0.1m)", "code": "mnw" },
  "MOO": { "name": "Moore / Mòoré / Mossi: Burkina Faso (5m)", "code": "mos" },
  "MOR": { "name": "Moro / Moru / Muro: Sudan-S Korodofan (30,000)", "code": "mor" },
  "MR":  { "name": "Maronite / Cypriot Arabic: Cyprus (1300)", "code": "acy" },
  "MRC": { "name": "Moroccan / Mugrabian Arabic: Morocco (20m)", "code": "ary" },
  "MRI": { "name": "Mari: Russia-Mari (0.8m)", "code": "chm" },
  "MRU": { "name": "Maru / Lhao Vo: Burma-Kachin,Shan (0.1m)", "code": "mhx" },
  "MSY": { "name": "Malagasy: Madagaskar (16m)", "code": "mlg" },
  "MUN": { "name": "Mundari: India-Jharkhand,Odisha (1.1m)", "code": "unr" },
  "MUO": { "name": "Muong: Vietnam (1m)", "code": "mtq" },
  "MUR": { "name": "Murut: Malaysia-Sarawak,Sabah (4500)", "code": "kxi,mvv,tih" },
  "MV":  { "name": "Malvi: India-Madhya Pradesh, Rajasthan (6m)", "code": "mup" },
  "MW":  { "name": "Marwari (a Rajasthani variety): India-Rajasthan,Gujarat (6m)", "code": "mwr,rwr" },
  "MX":  { "name": "Macuxi / Macushi: Brazil (16,000), Guyana (1,000)", "code": "mbc" },
  "MY":  { "name": "Maya (Yucatec): Mexico (0.7m), Belize (6000)", "code": "yua" },
  "MZ":  { "name": "Mizo / Lushai: India-Mizoram (0.7m)", "code": "lus" },
  "NAG": { "name": "Naga (var.incl. Ao,Makware): India-Nagaland, Assam (2m)", "code": "njh,njo,nmf,nph" },
  "NAP": { "name": "Naga Pidgin / Bodo / Nagamese: India-Nagaland (30,000)", "code": "nag" },
  "NDA": { "name": "Ndau: Mocambique (1.6m), Zimbabwe (0.8m)", "code": "ndc" },
  "NDE": { "name": "Ndebele: Zimbabwe (1.5m), South Africa-Limpopo (0.6m)", "code": "nde,nbl" },
  "NE":  { "name": "Nepali / Lhotshampa: Nepal (11m), India (3m), Bhutan (0.1m)", "code": "npi" },
  "NG":  { "name": "Nagpuri / Sadani / Sadari / Sadri: India-Jharkhand,W.Bengal (3m)", "code": "sck" },
  "NGA": { "name": "Ngangela / Nyemba: Angola (0.2m)", "code": "nba" },
  "NIC": { "name": "Nicobari: India-Nicobar Islands (40,000)", "code": "caq" },
  "NIG": { "name": "Nigerian Pidgin: Nigeria (30m)", "code": "pcm" },
  "NIS": { "name": "Nishi / Nyishi: India-Arunachal Pradesh (0.2m)", "code": "njz" },
  "NIU": { "name": "Niuean: Niue (2,000)", "code": "niu" },
  "NJ":  { "name": "Ngaju Dayak: Indonesia-Borneo (0.9m)", "code": "nij" },
  "NL":  { "name": "Dutch: Netherlands (16m), Belgium (6m), Suriname (0.2m)", "code": "nld" },
  "NLA": { "name": "Nga La / Matu Chin: Myanmar-Chin (30,000), India-Mizoram (10,000)", "code": "hlt" },
  "NO":  { "name": "Norwegian: Norway (5m)", "code": "nor" },
  "NOC": { "name": "Nocte / Nockte: India-Assam,Arunachal Pr. (33,000)", "code": "njb" },
  "NP":  { "name": "Nupe: Nigeria (0.8m)", "code": "nup" },
  "NTK": { "name": "Natakani / Netakani / Varhadi-Nagpuri: India-Maharashtra,M.Pr. (7m)", "code": "vah" },
  "NU":  { "name": "Nuer: Sudan (0.8m), Ethiopia (0.2m)", "code": "nus" },
  "NUN": { "name": "Nung: Vietnam (1m)", "code": "nut" },
  "NW":  { "name": "Newar / Newari: Nepal (0.8m)", "code": "new" },
  "NY":  { "name": "Nyanja: see CW-Chichewa which is the same" },
  "OG":  { "name": "Ogan: Indonesia-Sumatera (less than 0.5m)", "code": "pse" },
  "OH":  { "name": "Otjiherero service in Namibia (Languages: Herero, SeTswana)" },
  "OO":  { "name": "Oromo: Ethiopia (26m)", "code": "orm" },
  "OR":  { "name": "Odia / Oriya / Orissa: India-Odisha,Chhattisgarh (32m)", "code": "ory" },
  "OS":  { "name": "Ossetic: Russia (0.5m), Georgia (0.1m)", "code": "oss" },
  "OW":  { "name": "Oshiwambo service in Angola and Namibia (Languages: Ovambo, Kwanyama)" },
  "P":   { "name": "Portuguese: Brazil (187m), Angola (14m), Portugal (10m)", "code": "por" },
  "PAL": { "name": "Palaung - Pale: Myanmar (0.3m)", "code": "pce" },
  "PAS": { "name": "Pasemah: Indonesia-Sumatera (less than 0.5m)", "code": "pse" },
  "PED": { "name": "Pedi: S Africa (4m)", "code": "nso" },
  "PJ":  { "name": "Punjabi: Pakistan (60m), India-Punjab,Rajasthan (28m)", "code": "pnb,pan" },
  "PO":  { "name": "Polish: Poland (37m)", "code": "pol" },
  "POR": { "name": "Po: Myanmar-Rakhine (identical to K-W?)" },
  "POT": { "name": "Pothwari: Pakistan (2.5m)", "code": "phr" },
  "PS":  { "name": "Pashto / Pushtu: Afghanistan (6m), Pakistan (1m)", "code": "pbt" },
  "PU":  { "name": "Pulaar: Senegal (3m), Gambia (0.3m)", "code": "fuc" },
  "Q":   { "name": "Quechua: Peru, Bolivia, Ecuador (various varieties; 9m)", "code": "que,qvi" },
  "QQ":  { "name": "Qashqai: Iran (1.5m)", "code": "qxq" },
  "R":   { "name": "Russian: Russia (137m), Ukraine (8m), Kazakhstan (6m), Belarus (1m)", "code": "rus" },
  "RAD": { "name": "Rade / Ede: Vietnam (0.2m)", "code": "rad" },
  "RAJ": { "name": "Rajasthani: common lingua franca in Rajasthan (18m)", "code": "raj" },
  "REN": { "name": "Rengao: Vietnam (18,000)", "code": "ren" },
  "RGM": { "name": "Rengma Naga: India-Nagaland (34,000)", "code": "nre,nnl" },
  "RO":  { "name": "Romanian: Romania (20m), Moldova (3m), Serbia-Vojvodina (0.2m)", "code": "ron" },
  "ROG": { "name": "Roglai (Northern, Southern): Vietnam (0.1m)", "code": "rog,rgs" },
  "ROH": { "name": "Rohingya (rjj): Myanmar-Rakhine (2m)", "code": "rhg" },
  "RON": { "name": "Rongmei Naga: India-Manipur,Nagaland,Assam (60,000)", "code": "nbu" },
  "Ros": { "name": "Rosary session of Vatican Radio" },
  "RU":  { "name": "Rusyn / Ruthenian: Ukraine (0.5m), Serbia-Vojvodina (30,000)", "code": "rue" },
  "RWG": { "name": "Rawang: Myanmar-Kachin (60,000)", "code": "raw" },
  "S":   { "name": "Spanish / Castellano: Spain (30m), Latin America (336m), USA (34m)", "code": "spa" },
  "SAH": { "name": "Saho: Eritrea (0.2m)", "code": "ssy" },
  "SAN": { "name": "Sango: Central African Rep. (0.4m)", "code": "sag" },
  "SAR": { "name": "Sara / Sar: Chad (0.2m)", "code": "mwm" },
  "SAS": { "name": "Sasak: Indonesia-Lombok (2m)", "code": "sas" },
  "SC":  { "name": "Serbocroat (Yugoslav language up to national / linguistic separation)", "code": "hbs" },
  "SCA": { "name": "Scandinavian languages (Norwegian, Swedish, Finnish)" },
  "SD":  { "name": "Sindhi: Pakistan (19m), India (2m)", "code": "snd" },
  "SED": { "name": "Sedang: Vietnam (0.1m)", "code": "sed" },
  "SEF": { "name": "Sefardi / Judeo Spanish / Ladino: Israel (0.1m), Turkey (10,000)", "code": "lad" },
  "SEN": { "name": "Sena: Mocambique (1m)", "code": "seh" },
  "SFO": { "name": "Senoufo / Sénoufo-Syenara: Mali (0.15m)", "code": "shz" },
  "SGA": { "name": "Shangaan / Tsonga: Mocambique (2m), South Africa (2m)", "code": "tso" },
  "SGM": { "name": "Sara Gambai / Sara Ngambai: Chad (0.9m)", "code": "sba" },
  "SGO": { "name": "Songo: Angola (50,000)", "code": "nsx" },
  "SGT": { "name": "Sangtam: India-Nagaland (84,000)", "code": "nsa" },
  "SHA": { "name": "Shan: Myanmar (3m)", "code": "shn" },
  "SHk": { "name": "Shan-Khamti: Myanmar (8,000), India-Assam (5,000)", "code": "kht" },
  "SHC": { "name": "Sharchogpa / Sarchopa / Tshangla: E Bhutan (0.14m)", "code": "tsj" },
  "SHE": { "name": "Sheena / Shina: Pakistan (0.6m)", "code": "scl,plk" },
  "SHK": { "name": "Shiluk / Shilluk: South Sudan (0.2m)", "code": "shk" },
  "SHO": { "name": "Shona: Zimbabwe (11m)", "code": "sna" },
  "SHP": { "name": "Sherpa: Nepal (0.1m)", "code": "xsr" },
  "SHU": { "name": "Shuwa Arabic: Chad (1m), Nigeria (0.1m), N Cameroon (0.1m)", "code": "shu" },
  "SI":  { "name": "Sinhalese / Sinhala: Sri Lanka (16m)", "code": "sin" },
  "SID": { "name": "Sidamo / Sidama: Ethiopia (3m)", "code": "sid" },
  "SIK": { "name": "Sikkimese / Bhutia: India-Sikkim,W.Bengal (70,000)", "code": "sip" },
  "SIR": { "name": "Siraiki / Seraiki: Pakistan (14m)", "code": "skr" },
  "SK":  { "name": "Slovak: Slovakia (5m), Czech Republic (0.2m), Serbia (80,000)", "code": "slk" },
  "SLM": { "name": "Pijin / Solomon Islands Pidgin: Solomon Islands (0.3m)", "code": "pis" },
  "SLT": { "name": "Silte / East Gurage / xst: Ethiopia (1m)", "code": "stv" },
  "SM":  { "name": "Samoan: Samoa (0.2m), American Samoa (0.05m)", "code": "smo" },
  "SMP": { "name": "Sambalpuri / Sambealpuri: India-Odisha,Chhattisgarh (18m)", "code": "spv" },
  "SNK": { "name": "Sanskrit: India (0.2m)", "code": "san" },
  "SNT": { "name": "Santhali: India-Bihar,Jharkhand,Odisha (6m), Bangladesh (0.2m)", "code": "sat" },
  "SO":  { "name": "Somali: Somalia (8m), Ethiopia (5m), Kenya (2m), Djibouti (0.3m)", "code": "som" },
  "SON": { "name": "Songhai: Mali (0.6m)", "code": "ses,khq" },
  "SOT": { "name": "SeSotho: South Africa (4m), Lesotho (2m)", "code": "sot" },
  "SR":  { "name": "Serbian: Serbia (7m), Bosnia-Hercegovina (1.5m)", "code": "srp" },
  "SRA": { "name": "Soura / Sora: India-Odisha,Andhra Pr. (0.3m)", "code": "srb" },
  "STI": { "name": "Stieng: Vietnam (85,000)", "code": "sti,stt" },
  "SUA": { "name": "Shuar: Ecuador (35,000)", "code": "jiv" },
  "SUD": { "name": "Sudanese Arabic: Sudan and South Sudan (15m)", "code": "apd" },
  "SUM": { "name": "Sumi Naga: India-Nagaland (0.1m)", "code": "nsm" },
  "SUN": { "name": "Sunda / Sundanese: Indonesia-West Java (34m)", "code": "sun" },
  "SUR": { "name": "Surgujia: India-Chhattisgarh (1.5m)", "code": "sgj" },
  "SUS": { "name": "Sudan Service of IBRA, in SUD, BED, FUR, and TGR" },
  "SV":  { "name": "Slovenian: Slovenia (1.7m), Italy (0.1m), Austria (18,000)", "code": "slv" },
  "SWA": { "name": "Swahili / Kisuaheli: Tanzania (15m), Kenya, Ea.DR Congo (9m)", "code": "swc,swh" },
  "SWE": { "name": "Swedish: Sweden (8m), Finland (0.3m)", "code": "swe" },
  "SWZ": { "name": "SiSwati: Swaziland (1m), South Africa (1m)", "code": "ssw" },
  "T":   { "name": "Thai: Thailand (20m)", "code": "tha" },
  "TAG": { "name": "Tagalog: Philippines (22m)", "code": "tgl" },
  "TAH": { "name": "Tachelhit / Sous: Morocco, southern (4m), Algeria", "code": "shi" },
  "TAL": { "name": "Talysh: Azerbaijan, Iran (1m)", "code": "tly" },
  "TAM": { "name": "Tamil: S.India (60m), Malaysia (4m), Sri Lanka (4m)", "code": "tam" },
  "TAU": { "name": "Tausug: Philippines-Sulu, n.Borneo (1m)", "code": "tsg" },
  "TB":  { "name": "Tibetan / Lhasa Tibetan: Tibet (1m), India (0.1m)", "code": "bod" },
  "TBL": { "name": "Tboli / T'boli / Tagabili: Philippines-Mindanao (0.1m)", "code": "tbl" },
  "TBS": { "name": "Tabasaran: Russia-Dagestan (0.1m)", "code": "tab" },
  "TEL": { "name": "Telugu: India-Andhra Pr. (74m)", "code": "tel" },
  "TEM": { "name": "Temme / Temne: Sierra Leone (1.5m)", "code": "tem" },
  "TFT": { "name": "Tarifit: Morocco, northern (1.3m), Algeria", "code": "rif" },
  "TGB": { "name": "Tagabawa / Bagobo: Philippines-Mindanao (43,000)", "code": "bgs" },
  "TGK": { "name": "Tangkhul / Tangkul Naga: India-Manipur,Nagaland (0.15m)", "code": "nmf" },
  "TGR": { "name": "Tigre / Tigré / Tigrawit: Eritrea (1m)", "code": "tig" },
  "TGS": { "name": "Tangsa / Naga-Tase: Myanmar (60,000), India-Arunachal Pr. (40,000)", "code": "nst" },
  "THA": { "name": "Tharu Buksa: India-Uttarakhand (43,000)", "code": "tkb" },
  "TIG": { "name": "Tigrinya / Tigray: Ethiopia (4m), Eritrea (3m)", "code": "tir" },
  "TJ":  { "name": "Tajik: Tajikistan (3m), Uzbekistan (1m)", "code": "tgk" },
  "TK":  { "name": "Turkmen: Turkmenistan (3m), Iran (2m), Afghanistan (1.5m)", "code": "tuk" },
  "TKL": { "name": "Tagakaulo (dialect of Kalagan): Philippines-Mindanao (0.2m)", "code": "kqe,klg" },
  "TL":  { "name": "Tai-Lu / Lu: China-Yunnan (0.3m), Myanmar (0.2m), Laos (0.1m)", "code": "khb" },
  "TM":  { "name": "Tamazight: Morocco, central (3m)", "code": "zgh" },
  "TMG": { "name": "Tamang: Nepal (1.5m)", "code": "taj,tdg,tmk,tsf" },
  "TMJ": { "name": "Tamajeq: Niger (0.8m), Mali (0.44m), Algeria (40,000)", "code": "taq,thv,thz,ttq" },
  "TN":  { "name": "Tai-Nua / Chinese Shan: China-Yunnan (0.5m), LAO/MYA/VTN (0.2m)", "code": "tdd" },
  "TNG": { "name": "Tonga: Zambia (1m), Zimbabwe (0.1m)", "code": "toi" },
  "TO":  { "name": "Tongan: Tonga (0.1m)", "code": "ton" },
  "TOK": { "name": "Tokelau: Tokelau (1000)", "code": "tkl" },
  "TOR": { "name": "Torajanese / Toraja: Indonesia-Sulawesi (0.8m)", "code": "sda" },
  "TP":  { "name": "Tok Pisin: Papua New Guinea (4m)", "code": "tpi" },
  "TS":  { "name": "Tswana / SeTswana: Botswana (1m), South Africa (3m)", "code": "tsn" },
  "TSA": { "name": "Tsangla: see SHC" },
  "TSH": { "name": "Tshwa: Mocambique (1m)", "code": "tsc" },
  "TT":  { "name": "Tatar: Russia-Tatarstan,Bashkortostan (5m)", "code": "tat" },
  "TTB": { "name": "Tatar-Bashkir service of Radio Liberty" },
  "TU":  { "name": "Turkish: Turkey (46m), Bulgaria (0.6m), N Cyprus (0.2m)", "code": "tur" },
  "TUL": { "name": "Tulu: India-Karnataka,Kerala (2m)", "code": "tcy" },
  "TUM": { "name": "Tumbuka: Malawi (2m), Zambia (0.5m)", "code": "tum" },
  "TUN": { "name": "Tunisian Arabic: Tunisia (9m)", "code": "aeb" },
  "TUR": { "name": "Turkana: NW Kenya (1m)", "code": "tuv" },
  "TV":  { "name": "Tuva / Tuvinic: Russia-Tannu Tuva (0.25m)", "code": "tyv" },
  "TW":  { "name": "Taiwanese / Fujian / Hokkien / Min Nan (CHN 25m, TWN 15m, others 9m)", "code": "nan" },
  "TWI": { "name": "Twi / Akan: Ghana (8m)", "code": "aka" },
  "TWT": { "name": "Tachawit / Shawiya / Chaouia: Algeria (1.4m)", "code": "shy" },
  "TZ":  { "name": "Tamazight / Berber: Morocco (2m)", "code": "zgh,tzm" },
  "UA":  { "name": "Uab Meto / Dawan / Baikenu: West Timor (1m)", "code": "aoz" },
  "UD":  { "name": "Udmurt: Russia-Udmurtia (0.3m)", "code": "udm" },
  "UI":  { "name": "Uighur: China-Xinjiang (8m), Kazakhstan (0.3m)", "code": "uig" },
  "UK":  { "name": "Ukrainian: Ukraine (32m), Kazakhstan (0.9m), Moldova (0.6m)", "code": "ukr" },
  "UM":  { "name": "Umbundu: Angola (6m)", "code": "umb" },
  "UR":  { "name": "Urdu: Pakistan (104m), India (51m)", "code": "urd" },
  "UZ":  { "name": "Uzbek: Uzbekistan (16m)", "code": "uzn" },
  "V":   { "name": "Vasco / Basque / Euskera: Spain (0.6m), France (76,000)", "code": "eus" },
  "VAD": { "name": "Vadari / Waddar / Od: India-Andhra Pr. (0.2m)", "code": "wbq" },
  "VAR": { "name": "Varli / Warli: India-Maharashtra (0.6m)", "code": "vav" },
  "Ves": { "name": "Vespers (Vatican Radio)" },
  "Vn":  { "name": "Vernacular = local language(s)" },
  "VN":  { "name": "Vietnamese: Vietnam (66m)", "code": "vie" },
  "VV":  { "name": "Vasavi: India-Maharashtra,Gujarat (1m)", "code": "vas" },
  "VX":  { "name": "Vlax Romani / Romanes / Gypsy: Romania (0.2m), Russia (0.1m)", "code": "rmy" },
  "W":   { "name": "Wolof: Senegal (4m)", "code": "wol" },
  "WA":  { "name": "Wa / Parauk: South China (0.4m), Myanmar (0.4m)", "code": "prk" },
  "WAO": { "name": "Waodani / Waorani: Ecuador (2000)", "code": "auc" },
  "WE":  { "name": "Wenzhou: dialect of WU" },
  "WT":  { "name": "White Tai / Tai Don: Vietnam (0.3m), Laos (0.2m)", "code": "twh" },
  "WU":  { "name": "Wu: China-Jiangsu,Zhejiang (80m)", "code": "wuu" },
  "XH":  { "name": "Xhosa: South Africa (8m)", "code": "xho" },
  "YAO": { "name": "Yao / Yawo: Malawi (2m), Mocambique (0.5m), Tanzania (0.4m)", "code": "yao" },
  "YER": { "name": "Yerukula: India-Andhra Pr. (70,000)", "code": "yeu" },
  "YI":  { "name": "Yi / Nosu: China-Sichuan (2m)", "code": "iii" },
  "YK":  { "name": "Yakutian / Sakha: Russia-Sakha (0.5m)", "code": "sah" },
  "YO":  { "name": "Yoruba: Nigeria (20m), Benin (0.5m)", "code": "yor" },
  "YOL": { "name": "Yolngu / Yuulngu: Australia-Northern Territory (4000)", "code": "djr" },
  "YUN": { "name": "Dialects / languages of Yunnan (China)" },
  "YZ":  { "name": "Yezidi program (Kurdish-Kurmanji language)" },
  "Z":   { "name": "Zulu: South Africa (10m), Lesotho (0.3m)", "code": "zul" },
  "ZA":  { "name": "Zarma / Zama: Niger (2m)", "code": "dje" },
  "ZD":  { "name": "Zande: DR Congo (0.7m), South Sudan (0.35m)", "code": "zne" },
  "ZG":  { "name": "Zaghawa: Chad (87,000), Sudan (75,000)", "code": "zag" },
  "ZH":  { "name": "Zhuang: Southern China, 16 varieties (15m)", "code": "zha" },
  "ZWE": { "name": "Languages of Zimbabwe" },
  "Zomi-Chin:": { "name": "see Chin-Zomi (C-Z)" }
}

#
# Transmitter Locations
#
EIBI_Locations = {
  "AFG-k":  [{"name": "Kabul / Pol-e-Charkhi", "code": "AFG-k", "lat": 34.53333, "lon": 69.33333 }],
  "AFG-x":  [{"name": "Khost", "code": "AFG-x", "lat": 33.23333, "lon": 69.81667 }],
  "AFG-y":  [{"name": "Kabul-Yakatut", "code": "AFG-y", "lat": 34.53333, "lon": 69.21667 }],
  "AFS":    [{"name": "Meyerton", "code": "AFS", "lat": -26.58333, "lon": 28.13333 }],
  "AFS-ct": [{"name": "Cape Town", "code": "AFS-ct", "lat": -33.68333, "lon": 18.70000 }],
  "AFS-j":  [{"name": "Johannesburg", "code": "AFS-j", "lat": -26.12778, "lon": 28.20556 }],
  "AGL-L":  [{"name": "Luena (Moxico)", "code": "AGL-L", "lat": -11.78333, "lon": 19.92194 }],
  "AGL-lu": [{"name": "Luanda Radio", "code": "AGL-lu", "lat": -8.80000, "lon": 13.26667 }],
  "AGL-m":  [{"name": "Luanda - Mulenvos", "code": "AGL-m", "lat": -8.85000, "lon": 13.31667 }],
  "AIA":    [{"name": "The Valley", "code": "AIA", "lat": 18.21667, "lon": -63.01667 }],
  "ALB-c":  [{"name": "Cerrik (CRI)", "code": "ALB-c", "lat": 41.00000, "lon": 20.00000 }],
  "ALB-f":  [{"name": "Fllake (Durres, 500kW)", "code": "ALB-f", "lat": 41.36667, "lon": 19.50000 }],
  "ALB-s":  [{"name": "Shijiak (Radio Tirana) (1x100kW = 2x50kW)", "code": "ALB-s", "lat": 41.33333, "lon": 19.55000 }],
  "ALG-ad": [{"name": "Adrar", "code": "ALG-ad", "lat": 27.86667, "lon": -0.28333 }],
  "ALG-al": [{"name": "Algiers", "code": "ALG-al", "lat": 36.76667, "lon": 3.05000 }],
  "ALG-an": [{"name": "Annaba", "code": "ALG-an", "lat": 36.90000, "lon": 7.76667 }],
  "ALG-b":  [{"name": "Béchar", "code": "ALG-b", "lat": 31.56667, "lon": -2.35000 }],
  "ALG-fk": [{"name": "F'Kirina (Aïn Beïda)", "code": "ALG-fk", "lat": 35.73333, "lon": 7.35000 }],
  "ALG-o":  [{"name": "Ouargla / Ourgla", "code": "ALG-o", "lat": 31.91667, "lon": 5.06667 }],
  "ALG-of": [{"name": "Ouled Fayet", "code": "ALG-of", "lat": 36.71667, "lon": 2.95000 }],
  "ALG-or": [{"name": "Oran 7TO", "code": "ALG-or", "lat": 35.76667, "lon": -0.55000 }],
  "ALG-r":  [{"name": "Reggane", "code": "ALG-r", "lat": 26.70000, "lon": 0.16667 }],
  "ALG-s":  [{"name": "In Salah (Ain Salih)", "code": "ALG-s", "lat": 27.25000, "lon": 2.51667 }],
  "ALG-t":  [{"name": "Tindouf (Rabbouni)", "code": "ALG-t", "lat": 27.55000, "lon": -8.10000 }],
  "ALG-tm": [{"name": "Timimoun", "code": "ALG-tm", "lat": 29.26667, "lon": 0.23333 }],
  "ALS-an": [{"name": "Annette", "code": "ALS-an", "lat": 55.05000, "lon": -131.56667 }],
  "ALS-ap": [{"name": "Anchor Point", "code": "ALS-ap", "lat": 59.74944, "lon": -151.73222 }],
  "ALS-ba": [{"name": "Barrow", "code": "ALS-ba", "lat": 71.25833, "lon": -156.57750 }],
  "ALS-cb": [{"name": "Cold Bay", "code": "ALS-cb", "lat": 55.21667, "lon": -162.71667 }],
  "ALS-e":  [{"name": "Elmendorf AFB", "code": "ALS-e", "lat": 61.25111, "lon": -149.80639 }],
  "ALS-g":  [{"name": "Gakona", "code": "ALS-g", "lat": 62.39167, "lon": -145.14667 }],
  "ALS-k":  [{"name": "Kodiak", "code": "ALS-k", "lat": 57.77500, "lon": -152.53333 }],
  "ALS-ks": [{"name": "King Salmon", "code": "ALS-ks", "lat": 58.68333, "lon": -156.66667 }],
  "ALS-no": [{"name": "Nome", "code": "ALS-no", "lat": 64.51667, "lon": -165.41667 }],
  "ARG-b":  [{"name": "Buenos Aires", "code": "ARG-b", "lat": -34.62194, "lon": -58.35500 }],
  "ARG-co": [{"name": "Córdoba", "code": "ARG-co", "lat": -31.30917, "lon": -64.22611 }],
  "ARG-cr": [{"name": "Comodoro Rivadavia (Navy)", "code": "ARG-cr", "lat": -45.88361, "lon": -67.50917 }],
  "ARG-cv": [{"name": "Comodoro Rivadavia (Air)", "code": "ARG-cv", "lat": -45.79139, "lon": -67.47944 }],
  "ARG-e":  [{"name": "Ezeiza, Prov. Buenos Aires", "code": "ARG-e", "lat": -34.83278, "lon": -58.53194 }],
  "ARG-g":  [{"name": "General Pacheco", "code": "ARG-g", "lat": -34.60000, "lon": -58.36667 }],
  "ARG-mp": [{"name": "Mar del Plata, Prov. Buenos Aires", "code": "ARG-mp", "lat": -38.05000, "lon": -57.53333 }],
  "ARG-r":  [{"name": "Resistencia, Chaco", "code": "ARG-r", "lat": -27.46417, "lon": -59.07056 }],
  "ARM":    [{"name": "Gavar (formerly Kamo)", "code": "ARM", "lat": 40.41667, "lon": 45.20000 }],
  "ARM-y":  [{"name": "Yerevan", "code": "ARM-y", "lat": 40.16667, "lon": 44.50000 }],
  "ARS-j":  [{"name": "Jeddah/Jiddah", "code": "ARS-j", "lat": 21.25000, "lon": 39.16667 }],
  "ARS-jr": [{"name": "Jiddah Radio", "code": "ARS-jr", "lat": 21.38333, "lon": 39.16667 }],
  "ARS-jz": [{"name": "Jazan", "code": "ARS-jz", "lat": 16.88333, "lon": 42.56667 }],
  "ARS-nj": [{"name": "Najran", "code": "ARS-nj", "lat": 17.50000, "lon": 44.13333 }],
  "ARS-r":  [{"name": "Riyadh", "code": "ARS-r", "lat": 24.50000, "lon": 46.38333 }],
  "ASC":    [{"name": "Ascension Island,", "code": "ASC", "lat": -7.90000, "lon": -14.38333 }],
  "ATA-e":  [{"name": "Base Esperanza", "code": "ATA-e", "lat": -63.40000, "lon": -57.00000 }],
  "ATA-f":  [{"name": "Bahia Fildes, King George Island", "code": "ATA-f", "lat": -62.20000, "lon": -58.96667 }],
  "ATA-ma": [{"name": "Maramio Base, Seymour Island", "code": "ATA-ma", "lat": -64.23333, "lon": -56.63333 }],
  "AUS-a":  [{"name": "Alice Springs NT", "code": "AUS-a", "lat": -23.81667, "lon": 133.85000 }],
  "AUS-ae": [
      {"name": "Cape Pallarenda (aero site)", "code": "AUS-ae", "lat": -19.201389, "lon": 146.768056 },
      {"name": "Broken Hill (aero site)", "code": "AUS-ae", "lat": -31.927222, "lon": 141.482500 },
      {"name": "Knuckeys Lagoon (aero site)", "code": "AUS-ae", "lat": -12.43111, "lon": 130.96417 }
  ],
  "AUS-al": [{"name": "VKS737 Alice Springs NT", "code": "AUS-al", "lat": -23.683333, "lon": 133.866667 }],
  "AUS-as": [{"name": "Alice Springs NT", "code": "AUS-as", "lat": -23.79667, "lon": 133.87444 }],
  "AUS-at": [{"name": "Alice Springs NT", "code": "AUS-at", "lat": -23.77917, "lon": 133.87361 }],
  "AUS-av": [{"name": "Alice Springs Velodrome NT", "code": "AUS-av", "lat": -23.670556, "lon": 133.865 }],
  "AUS-b":  [{"name": "Brandon QL", "code": "AUS-b", "lat": -19.51667, "lon": 147.33333 }],
  "AUS-be": [{"name": "Bendigo VIC", "code": "AUS-be", "lat": -36.59028, "lon": 144.24417 }],
  "AUS-bm": [{"name": "Broadmeadows VIC", "code": "AUS-bm", "lat": -37.69194, "lon": 144.94556 }],
  "AUS-c":  [{"name": "Charleville QL", "code": "AUS-c", "lat": -26.41667, "lon": 146.25000 }],
  "AUS-ca": [{"name": "Casino", "code": "AUS-ca", "lat": -28.875278, "lon": 153.051111 }],
  "AUS-ch": [{"name": "VKS737 Charter Towers QLD", "code": "AUS-ch", "lat": -20.15, "lon": 146.3 }],
  "AUS-cl": [{"name": "RFDS Charleville QLD", "code": "AUS-cl", "lat": -26.415278, "lon": 146.226389 }],
  "AUS-ct": [{"name": "VKS737 Charter Towers QLD", "code": "AUS-ct", "lat": -20.085, "lon": 146.259444 }],
  "AUS-cv": [{"name": "RFDS Carnarvon WA", "code": "AUS-cv", "lat": -24.888889, "lon": 113.673333 }],
  "AUS-ee": [{"name": "VKS737 Adelaide/Elizabeth East SA", "code": "AUS-ee", "lat": -34.722222, "lon": 138.683056 }],
  "AUS-ex": [{"name": "Exmouth WA", "code": "AUS-ex", "lat": -21.81667, "lon": 114.16667 }],
  "AUS-g":  [{"name": "Gunnedah NSW", "code": "AUS-g", "lat": -30.98333, "lon": 150.25000 }],
  "AUS-h":  [{"name": "Humpty Doo NT", "code": "AUS-h", "lat": -12.56667, "lon": 131.08333 }],
  "AUS-hc": [{"name": "Halls Creek NSW", "code": "AUS-hc", "lat": -18.23, "lon": 127.67 }],
  "AUS-hp": [{"name": "Hurlstone Park, Sydney NSW", "code": "AUS-hp", "lat": -33.90556, "lon": 151.13222 }],
  "AUS-il": [{"name": "VKS737 Alice Springs-Ilparpa NT", "code": "AUS-il", "lat": -23.755556, "lon": 133.823889 }],
  "AUS-in": [{"name": "Innisfail QL", "code": "AUS-in", "lat": -17.53333, "lon": 146.05000 }],
  "AUS-ka": [{"name": "Katherine NT", "code": "AUS-ka", "lat": -14.40000, "lon": 132.18333 }],
  "AUS-kd": [{"name": "RFDS Kuranda QLD", "code": "AUS-kd", "lat": -16.836667, "lon": 145.6125 }],
  "AUS-ku": [{"name": "Kununurra WA Lot 3000", "code": "AUS-ku", "lat": -15.816667, "lon": 128.666667 }],
  "AUS-kw": [{"name": "Kununurra WA", "code": "AUS-kw", "lat": -15.771389, "lon": 128.729444 }],
  "AUS-L":  [{"name": "Sydney-Leppington NSW", "code": "AUS-L", "lat": -33.96667, "lon": 150.80000 }],
  "AUS-m":  [{"name": "Macleay Island QL", "code": "AUS-m", "lat": -27.61667, "lon": 153.35000 }],
  "AUS-ma": [{"name": "Manilla NSW", "code": "AUS-ma", "lat": -30.73972, "lon": 150.71556 }],
  "AUS-md": [{"name": "Mareeba-Dimbulah QL", "code": "AUS-md", "lat": -17.166667, "lon": 145.083333 }],
  "AUS-mi": [{"name": "RFDS Mount Isa QLD 20S", "code": "AUS-mi", "lat": 43.000000, "lon": 139.487222 }],
  "AUS-mk": [{"name": "RFDS Meekatharra WA", "code": "AUS-mk", "lat": -26.586667, "lon": 118.500833 }],
  "AUS-n":  [{"name": "Ningi QL", "code": "AUS-n", "lat": -27.06667, "lon": 153.05556 }],
  "AUS-nc": [{"name": "VKS737 Newcastle/Edgeworth NSW", "code": "AUS-nc", "lat": -32.922222, "lon": 151.607778 }],
  "AUS-pc": [{"name": "Perth / Chittering WA", "code": "AUS-pc", "lat": -31.494444, "lon": 116.081111 }],
  "AUS-pe": [{"name": "Penong SA", "code": "AUS-pe", "lat": -31.924444, "lon": 132.991111 }],
  "AUS-pw": [{"name": "VKS737 Perth/Wanneroo WA", "code": "AUS-pw", "lat": -31.766944, "lon": 115.804167 }],
  "AUS-ri": [{"name": "Russell Island QL", "code": "AUS-ri", "lat": -27.66667, "lon": 153.35000 }],
  "AUS-rm": [{"name": "Roma QL", "code": "AUS-rm", "lat": -26.55000, "lon": 148.80000 }],
  "AUS-rz": [{"name": "Razorback NSW", "code": "AUS-rz", "lat": -34.15000, "lon": 150.66667 }],
  "AUS-s":  [{"name": "Shepparton VIC", "code": "AUS-s", "lat": -36.33333, "lon": 145.41667 }],
  "AUS-sa": [{"name": "Shepparton-Ardmona VIC", "code": "AUS-sa", "lat": -36.360833, "lon": 145.293889 }],
  "AUS-sb": [{"name": "VKS737 Stawell-Black Range VIC", "code": "AUS-sb", "lat": -37.100278, "lon": 142.753889 }],
  "AUS-sf": [{"name": "Schofields, western Sydney", "code": "AUS-sf", "lat": -33.70000, "lon": 150.86667 }],
  "AUS-sm": [{"name": "St Mary's, Sydney", "code": "AUS-sm", "lat": -33.75000, "lon": 150.76667 }],
  "AUS-st": [{"name": "VKS737 St Marys TAS", "code": "AUS-st", "lat": -41.585556, "lon": 148.214722 }],
  "AUS-t":  [{"name": "Tennant Creek NT", "code": "AUS-t", "lat": -19.66667, "lon": 134.26667 }],
  "AUS-va": [{"name": "VKS737 Adelaide/Virginia SA", "code": "AUS-va", "lat": -34.681111, "lon": 138.593056 }],
  "AUS-w":  [{"name": "Wiluna WA", "code": "AUS-w", "lat": -26.33333, "lon": 120.56667 }],
  "AUS-ww": [{"name": "Wee Waa NSW", "code": "AUS-ww", "lat": -30.21528, "lon": 149.45722 }],
  "AUT":    [{"name": "Moosbrunn", "code": "AUT", "lat": 48.00000, "lon": 16.46667 }],
  "AZE-b":  [{"name": "Baku", "code": "AZE-b", "lat": 40.46667, "lon": 50.05000 }],
  "AZE-g":  [{"name": "Gäncä", "code": "AZE-g", "lat": 40.60000, "lon": 46.33333 }],
  "AZE-s":  [{"name": "Stepanakert", "code": "AZE-s", "lat": 39.82639, "lon": 46.73972 }],
  "AZR-ho": [{"name": "Horta", "code": "AZR-ho", "lat": 38.53333, "lon": -28.63333 }],
  "AZR-lj": [{"name": "Lajes Field", "code": "AZR-lj", "lat": 38.76667, "lon": 27.08333 }],
  "AZR-sm": [{"name": "Santa Maria", "code": "AZR-sm", "lat": 36.94722, "lon": -25.15833 }],
  "B-a":    [{"name": "Porto Alegre, RS", "code": "B-a", "lat": -30.02361, "lon": -51.25528 }],
  "B-ag":   [{"name": "Araguaína, TO", "code": "B-ag", "lat": -7.20000, "lon": -48.20000 }],
  "B-am":   [{"name": "Amparo, SP", "code": "B-am", "lat": -22.70000, "lon": -46.76667 }],
  "B-an":   [{"name": "Anápolis, GO", "code": "B-an", "lat": -16.25694, "lon": -49.01889 }],
  "B-ap":   [{"name": "Aparecida, SP", "code": "B-ap", "lat": -22.84639, "lon": -45.22028 }],
  "B-ar":   [{"name": "Araraquara, SP", "code": "B-ar", "lat": -21.80000, "lon": -48.18333 }],
  "B-b":    [{"name": "Brasilia, Parque do Rodeador, DF", "code": "B-b", "lat": -15.61111, "lon": -48.13139 }],
  "B-be":   [{"name": "Belém, PA", "code": "B-be", "lat": -1.45000, "lon": -48.48333 }],
  "B-bh":   [{"name": "Belo Horizonte, Minas Gerais", "code": "B-bh", "lat": -19.91667, "lon": -43.93333 }],
  "B-br":   [{"name": "Braganca, PA", "code": "B-br", "lat": -1.06333, "lon": -46.77333 }],
  "B-bt":   [{"name": "Belém, PA (Ondas Tropicais 5045)", "code": "B-bt", "lat": -1.36667, "lon": -48.35000 }],
  "B-bv":   [{"name": "Boa Vista, RR", "code": "B-bv", "lat": 2.92194, "lon": -60.71056 }],
  "B-c":    [{"name": "Contagem/Belo Horizonte, MG", "code": "B-c", "lat": -19.89972, "lon": -44.05444 }],
  "B-ca":   [{"name": "Campo Largo (Curitiba), PR", "code": "B-ca", "lat": -25.43000, "lon": -49.39694 }],
  "B-cb":   [{"name": "Camboriú, SC", "code": "B-cb", "lat": -27.04028, "lon": -48.65472 }],
  "B-cc":   [{"name": "Cáceres, MT", "code": "B-cc", "lat": -16.07667, "lon": -57.64083 }],
  "B-cg":   [{"name": "Campo Grande, MS", "code": "B-cg", "lat": -20.52000, "lon": -54.58333 }],
  "B-Cg":   [{"name": "Campo Grande, MS", "code": "B-Cg", "lat": -20.45000, "lon": -54.61667 }],
  "B-cm":   [{"name": "Campinas, SP", "code": "B-cm", "lat": -22.94778, "lon": -47.01806 }],
  "B-cn":   [{"name": "Congonhas, MG", "code": "B-cn", "lat": -20.50000, "lon": -43.86667 }],
  "B-co":   [{"name": "Coari, AM", "code": "B-co", "lat": -4.11639, "lon": -63.12528 }],
  "B-cp":   [{"name": "Cachoeira Paulista, SP", "code": "B-cp", "lat": -22.64583, "lon": -45.07833 }],
  "B-Cp":   [{"name": "Cachoeira Paulista, SP", "code": "B-Cp", "lat": -22.64417, "lon": -45.07722 }],
  "B-cs":   [{"name": "Cruzeiro do Sul, Estrada do Aeroporto, AC", "code": "B-cs", "lat": -7.63333, "lon": -72.66667 }],
  "B-cu":   [{"name": "Curitiba, PR", "code": "B-cu", "lat": -25.45222, "lon": -49.11389 }],
  "B-cv":   [{"name": "Cuiabá, MT", "code": "B-cv", "lat": -15.61861, "lon": -56.09778 }],
  "B-c2":   [{"name": "Curitiba, PR RB2", "code": "B-c2", "lat": -25.39278, "lon": -49.16778 }],
  "B-E":    [{"name": "Esteio (Porto Alegre), RS", "code": "B-E", "lat": -29.82806, "lon": -51.16500 }],
  "B-e":    [{"name": "Esteio (Porto Alegre), RS", "code": "B-e", "lat": -29.86639, "lon": -51.10306 }],
  "B-f":    [{"name": "Foz do Iguacu, PR", "code": "B-f", "lat": -25.51750, "lon": -54.50833 }],
  "B-fl":   [{"name": "Florianópolis, SC", "code": "B-fl", "lat": -27.60250, "lon": -48.53083 }],
  "B-fp":   [{"name": "Florianópolis - Comboriú, SC", "code": "B-fp", "lat": -27.04000, "lon": -48.65472 }],
  "B-g":    [{"name": "Guarujá, SP", "code": "B-g", "lat": -23.99306, "lon": -46.25639 }],
  "B-gb":   [{"name": "Guaíba (Porto Alegre), RS", "code": "B-gb", "lat": -29.99722, "lon": -51.28556 }],
  "B-gc":   [{"name": "Sao Gabriel de Cachoeira, AM", "code": "B-gc", "lat": -0.13333, "lon": -67.08333 }],
  "B-gm":   [{"name": "Guajará-Mirim, RO", "code": "B-gm", "lat": -10.78333, "lon": -65.33333 }],
  "B-go":   [{"name": "Goiânia,", "code": "B-go", "lat": -16.65833, "lon": -49.22722 }],
  "B-gu":   [{"name": "Guarulhos, SP", "code": "B-gu", "lat": -23.43333, "lon": -46.41667 }],
  "B-h":    [{"name": "Belo Horizonte, MG", "code": "B-h", "lat": -19.97611, "lon": -43.93333 }],
  "B-ib":   [{"name": "Ibitinga, SP", "code": "B-ib", "lat": -21.77222, "lon": -48.83611 }],
  "B-it":   [{"name": "Itapevi, SP", "code": "B-it", "lat": -23.51083, "lon": -46.67611 }],
  "B-ld":   [{"name": "Londrina, PR", "code": "B-ld", "lat": -23.33778, "lon": -51.22167 }],
  "B-li":   [{"name": "Limeira, SP", "code": "B-li", "lat": -22.56083, "lon": -47.41889 }],
  "B-lj":   [{"name": "Lajeado, RS", "code": "B-lj", "lat": -29.46667, "lon": -51.96667 }],
  "B-lo":   [{"name": "Londrina, PR", "code": "B-lo", "lat": -23.40472, "lon": -51.15528 }],
  "B-m":    [{"name": "Manaus AM", "code": "B-m", "lat": -3.10000, "lon": -60.03333 }],
  "B-ma":   [{"name": "Manaus - Radiodif.Amazonas, AM", "code": "B-ma", "lat": -3.13778, "lon": -59.98139 }],
  "B-mc":   [{"name": "Macapá, AP", "code": "B-mc", "lat": 0.06389, "lon": -51.03889 }],
  "B-mg":   [{"name": "Manaus - Radio Globo, AM", "code": "B-mg", "lat": -3.13444, "lon": -59.97750 }],
  "B-mi":   [{"name": "Marília, SP", "code": "B-mi", "lat": -22.22583, "lon": -49.96278 }],
  "B-mm":   [{"name": "São Mateus do Maranhão, Maranhão", "code": "B-mm", "lat": -4.03333, "lon": -44.46667 }],
  "B-mo":   [{"name": "Mogi das Cruces, SP", "code": "B-mo", "lat": -23.51528, "lon": -46.20222 }],
  "B-mr":   [{"name": "Manaus - Radio Rio Mar, AM", "code": "B-mr", "lat": -3.12167, "lon": -60.04167 }],
  "B-ob":   [{"name": "Óbidos, PA", "code": "B-ob", "lat": -1.91667, "lon": -55.51667 }],
  "B-os":   [{"name": "Osasco, SP", "code": "B-os", "lat": -23.51417, "lon": -46.59417 }],
  "B-pa":   [{"name": "Parintins, AM", "code": "B-pa", "lat": -2.61667, "lon": -56.75000 }],
  "B-pc":   [{"name": "Pocos da Caldas, MG", "code": "B-pc", "lat": -21.79778, "lon": -46.54056 }],
  "B-pe":   [{"name": "Petrolina, PE", "code": "B-pe", "lat": -9.40000, "lon": -40.50000 }],
  "B-pi":   [{"name": "Piraquara (Curitiba), PR", "code": "B-pi", "lat": -25.39278, "lon": -49.16778 }],
  "B-r":    [{"name": "Rio de Janeiro (Radio Globo), RJ", "code": "B-r", "lat": -22.82333, "lon": -43.09694 }],
  "B-rb":   [{"name": "Rio Branco, AC", "code": "B-rb", "lat": -9.96667, "lon": -67.81667 }],
  "B-rc":   [{"name": "Rio de Janeiro (Radio Capital), RJ", "code": "B-rc", "lat": -22.77861, "lon": -43.01556 }],
  "B-re":   [{"name": "Recife, PE", "code": "B-re", "lat": -8.06667, "lon": -34.96667 }],
  "B-rj":   [{"name": "Rio de Janeiro (Radio Relogio), RJ", "code": "B-rj", "lat": -22.77806, "lon": -42.98389 }],
  "B-ro":   [{"name": "Rio de Janeiro, Observatório Nacional,", "code": "B-ro", "lat": -22.89583, "lon": -43.22417 }],
  "B-rp":   [{"name": "Ribeirão Preto, SP", "code": "B-rp", "lat": -21.18333, "lon": -47.80000 }],
  "B-rs":   [{"name": "Rio de Janeiro (Super Radio), RJ", "code": "B-rs", "lat": -22.82278, "lon": -43.08917 }],
  "B-rw":   [{"name": "Rio de Janeiro PWZ", "code": "B-rw", "lat": -22.95000, "lon": -42.91667 }],
  "B-sa":   [{"name": "Santarém, PA", "code": "B-sa", "lat": -2.44861, "lon": -54.73278 }],
  "B-sb":   [{"name": "Sao Paulo - Radio Bandeirantes, SP", "code": "B-sb", "lat": -23.64833, "lon": -46.60056 }],
  "B-sc":   [{"name": "Sao Paulo - Radio Cultura, SP", "code": "B-sc", "lat": -23.51167, "lon": -46.56139 }],
  "B-se":   [{"name": "Senador Guiomard, AC", "code": "B-se", "lat": -10.05000, "lon": -67.61667 }],
  "B-sg":   [{"name": "Sao Paulo - Radio Globo, SP", "code": "B-sg", "lat": -23.60722, "lon": -46.43667 }],
  "B-sj":   [{"name": "Sao Paulo - Radio 9 de Julho, SP", "code": "B-sj", "lat": -23.54750, "lon": -46.63611 }],
  "B-sm":   [{"name": "Santa Maria, RS", "code": "B-sm", "lat": -29.73833, "lon": -53.55528 }],
  "B-so":   [{"name": "Sorocaba, SP / Votorantim,", "code": "B-so", "lat": -23.55000, "lon": -47.43333 }],
  "B-sr":   [{"name": "Sao Paulo - Radio Record, SP", "code": "B-sr", "lat": -23.68389, "lon": -46.74306 }],
  "B-sy":   [{"name": "Sao Paulo PYB45", "code": "B-sy", "lat": -23.55000, "lon": -46.63333 }],
  "B-sz":   [{"name": "Sao Paulo - Radio Gazeta, SP", "code": "B-sz", "lat": -23.66944, "lon": -46.75000 }],
  "B-ta":   [{"name": "Taubaté, SP", "code": "B-ta", "lat": -23.01667, "lon": -45.56667 }],
  "B-te":   [{"name": "Teresina, PI", "code": "B-te", "lat": -5.08694, "lon": -42.76083 }],
  "B-tf":   [{"name": "Tefé, AM", "code": "B-tf", "lat": -3.35417, "lon": -64.71139 }],
  "B-vi":   [{"name": "Vitória, ES", "code": "B-vi", "lat": -20.31667, "lon": -40.31667 }],
  "B-x":    [{"name": "Xapuri, AC", "code": "B-x", "lat": -10.65000, "lon": -68.50000 }],
  "B-xm":   [{"name": "Unknown location in Maranhão,", "code": "B-xm", "lat": -2.50000, "lon": -44.25000 }],
  "B-xn":   [{"name": "Unknown location in Paraná,", "code": "B-xn", "lat": -25.00000, "lon": -52.00000 }],
  "B-xp":   [{"name": "Unknown location in Paraíba,", "code": "B-xp", "lat": -7.16667, "lon": -36.83333 }],
  "BEL-o":  [{"name": "Oostende", "code": "BEL-o", "lat": 51.18333, "lon": 2.80000 }],
  "BEL-w":  [{"name": "Wingene", "code": "BEL-w", "lat": 51.18333, "lon": 2.81667 }],
  "BEN-c":  [{"name": "Cotonou", "code": "BEN-c", "lat": 6.46667, "lon": 2.35000 }],
  "BEN-p":  [{"name": "Parakou", "code": "BEN-p", "lat": 9.33333, "lon": 2.63333 }],
  "BER-h":  [{"name": "Bermuda Harbour", "code": "BER-h", "lat": 32.38333, "lon": -64.68333 }],
  "BES":    [{"name": "Bonaire", "code": "BES", "lat": 12.20000, "lon": -68.30000 }],
  "BFA":    [{"name": "Ouagadougou", "code": "BFA", "lat": 12.43333, "lon": -1.55000 }],
  "BGD-d":  [{"name": "Dhaka-Dhamrai", "code": "BGD-d", "lat": 23.90000, "lon": 90.20000 }],
  "BGD-k":  [{"name": "Dhaka-Khabirpur", "code": "BGD-k", "lat": 24.00000, "lon": 90.25000 }],
  "BGD-s":  [{"name": "Dhaka-Savar", "code": "BGD-s", "lat": 23.86667, "lon": 90.26667 }],
  "BHR-a":  [{"name": "Abu Hayan", "code": "BHR-a", "lat": 26.03333, "lon": 50.61667 }],
  "BHR-m":  [{"name": "Al Muharraq", "code": "BHR-m", "lat": 26.26667, "lon": 50.65000 }],
  "BIH-b":  [{"name": "Bijeljina", "code": "BIH-b", "lat": 44.70000, "lon": 19.16667 }],
  "BIH-z":  [{"name": "Zavidovici", "code": "BIH-z", "lat": 44.43333, "lon": 18.15000 }],
  "BIO":    [{"name": "Diego Garcia", "code": "BIO", "lat": -7.43333, "lon": 72.43333 }],
  "BLR":    [{"name": "Minsk-Sasnovy/Kalodziscy", "code": "BLR", "lat": 53.96667, "lon": 27.78333 }],
  "BLR-b":  [{"name": "Brest", "code": "BLR-b", "lat": 52.30000, "lon": 23.90000 }],
  "BLR-g":  [{"name": "Hrodna/Grodno", "code": "BLR-g", "lat": 53.66667, "lon": 23.83333 }],
  "BLR-m":  [{"name": "Mahiliou/Mogilev ('Orsha')", "code": "BLR-m", "lat": 53.61667, "lon": 30.33333 }],
  "BLR-mo": [{"name": "Molodechno/Vileyka (43 Comm Center Russian Navy)", "code": "BLR-mo", "lat": 54.46667, "lon": 26.78333 }],
  "BLR-vi": [{"name": "Vitebsk", "code": "BLR-vi", "lat": 55.13333, "lon": 30.35000 }],
  "BOL-ay": [{"name": "Santa Ana del Yacuma", "code": "BOL-ay", "lat": -13.75000, "lon": -65.53333 }],
  "BOL-cb": [{"name": "Cochabamba", "code": "BOL-cb", "lat": -17.38333, "lon": -66.18333 }],
  "BOL-p":  [{"name": "La Paz", "code": "BOL-p", "lat": -16.50000, "lon": -68.13333 }],
  "BOL-ri": [{"name": "Riberalta", "code": "BOL-ri", "lat": -10.99694, "lon": -66.06694 }],
  "BOL-sc": [{"name": "Santa Cruz", "code": "BOL-sc", "lat": -17.80000, "lon": -63.16667 }],
  "BOL-sz": [{"name": "Santa Cruz Airport", "code": "BOL-sz", "lat": -17.66667, "lon": -63.13333 }],
  "BOL-uy": [{"name": "Uyuni", "code": "BOL-uy", "lat": -20.46667, "lon": -66.81667 }],
  "BOL-yu": [{"name": "Yura", "code": "BOL-yu", "lat": -20.06667, "lon": -66.13333 }],
  "BOT":    [{"name": "Mopeng Hill", "code": "BOT", "lat": -21.95000, "lon": 27.65000 }],
  "BTN":    [{"name": "Thimphu", "code": "BTN", "lat": 27.46667, "lon": 89.65000 }],
  "BUL-bg": [{"name": "Blagoevgrad (864)", "code": "BUL-bg", "lat": 42.05000, "lon": 23.05000 }],
  "BUL-bk": [{"name": "Bankya", "code": "BUL-bk", "lat": 42.72667, "lon": 23.15917 }],
  "BUL-do": [{"name": "Doulovo (1161)", "code": "BUL-do", "lat": 43.81667, "lon": 27.15000 }],
  "BUL-kj": [{"name": "Kardjali (963)", "code": "BUL-kj", "lat": 41.60000, "lon": 25.36667 }],
  "BUL-p":  [{"name": "Plovdiv-Padarsko", "code": "BUL-p", "lat": 42.38333, "lon": 24.86667 }],
  "BUL-pe": [{"name": "Petrich", "code": "BUL-pe", "lat": 41.46667, "lon": 23.33333 }],
  "BUL-s":  [{"name": "Sofia-Kostinbrod", "code": "BUL-s", "lat": 42.81667, "lon": 23.21667 }],
  "BUL-sa": [{"name": "Samuil (864)", "code": "BUL-sa", "lat": 43.53500, "lon": 26.73694 }],
  "BUL-sl": [{"name": "Salmanovo (747)", "code": "BUL-sl", "lat": 43.18333, "lon": 26.96667 }],
  "BUL-tv": [{"name": "Targovishte (1161)", "code": "BUL-tv", "lat": 43.25000, "lon": 26.51667 }],
  "BUL-va": [{"name": "Varna", "code": "BUL-va", "lat": 43.06667, "lon": 27.78333 }],
  "BUL-vi": [{"name": "Vidin", "code": "BUL-vi", "lat": 43.83333, "lon": 22.71667 }],
  "BUL-vk": [{"name": "Vakarel (261)", "code": "BUL-vk", "lat": 42.56667, "lon": 23.70000 }],
  "CAF-ba": [{"name": "Bangui", "code": "CAF-ba", "lat": 4.35000, "lon": 18.58333 }],
  "CAF-bo": [{"name": "Boali", "code": "CAF-bo", "lat": 4.65000, "lon": 18.20000 }],
  "CAN-al": [{"name": "Aldergrove BC, Matsqui Tx site", "code": "CAN-al", "lat": 49.10833, "lon": -122.24444 }],
  "CAN-ap": [{"name": "Amphitrite Point BC", "code": "CAN-ap", "lat": 48.92528, "lon": -125.54028 }],
  "CAN-c":  [{"name": "Calgary AB", "code": "CAN-c", "lat": 50.90056, "lon": -113.87583 }],
  "CAN-cb": [{"name": "Cambridge Bay, Victoria Island NU", "code": "CAN-cb", "lat": 69.11472, "lon": -105.01972 }],
  "CAN-cc": [{"name": "Churchill MB", "code": "CAN-cc", "lat": 58.76167, "lon": -93.94417 }],
  "CAN-ch": [{"name": "Coral Harbour NU", "code": "CAN-ch", "lat": 64.15028, "lon": -83.37278 }],
  "CAN-cr": [{"name": "Cap des Rosiers", "code": "CAN-cr", "lat": 48.86111, "lon": -64.21472 }],
  "CAN-cw": [{"name": "Cartwright NL", "code": "CAN-cw", "lat": 53.70833, "lon": -57.02139 }],
  "CAN-di": [{"name": "Digby Island BC", "code": "CAN-di", "lat": 54.29750, "lon": -130.41833 }],
  "CAN-ex": [{"name": "Essex County (Harrow), near Detroit, ON", "code": "CAN-ex", "lat": 42.04167, "lon": -82.97417 }],
  "CAN-fg": [{"name": "CFG8525 ON", "code": "CAN-fg", "lat": 43.86667, "lon": -79.31667 }],
  "CAN-g":  [{"name": "Gander NL", "code": "CAN-g", "lat": 48.96806, "lon": -54.67389 }],
  "CAN-h":  [{"name": "Halifax NS", "code": "CAN-h", "lat": 44.68417, "lon": -63.60972 }],
  "CAN-hd": [{"name": "Hopedale NL", "code": "CAN-hd", "lat": 55.45667, "lon": -60.20833 }],
  "CAN-hp": [{"name": "Hunter Point BC", "code": "CAN-hp", "lat": 53.25861, "lon": -132.71472 }],
  "CAN-hr": [{"name": "Hay River", "code": "CAN-hr", "lat": 60.84083, "lon": -115.77000 }],
  "CAN-hx": [{"name": "Halifax CFH NS", "code": "CAN-hx", "lat": 44.96389, "lon": -63.98194 }],
  "CAN-i":  [{"name": "Iqaluit NU", "code": "CAN-i", "lat": 63.73111, "lon": -68.54222 }],
  "CAN-in": [{"name": "Inuvik NWT", "code": "CAN-in", "lat": 68.32583, "lon": -133.59806 }],
  "CAN-j":  [{"name": "St John's NL", "code": "CAN-j", "lat": 47.56944, "lon": -52.81444 }],
  "CAN-k":  [{"name": "Killiniq/Killinek NU", "code": "CAN-k", "lat": 60.42417, "lon": -64.84167 }],
  "CAN-ki": [{"name": "Kingsburg NS", "code": "CAN-ki", "lat": 44.27556, "lon": -64.28750 }],
  "CAN-lp": [{"name": "Lockeport NS", "code": "CAN-lp", "lat": 43.66361, "lon": -65.12972 }],
  "CAN-lv": [{"name": "La Vernière, Îles-de-la-Madeleine QC", "code": "CAN-lv", "lat": 47.35722, "lon": -61.92667 }],
  "CAN-na": [{"name": "Natashquan QC", "code": "CAN-na", "lat": 50.15167, "lon": -61.79500 }],
  "CAN-o":  [{"name": "Ottawa ON", "code": "CAN-o", "lat": 45.29472, "lon": -75.75806 }],
  "CAN-pc": [{"name": "Port Caledonia NS", "code": "CAN-pc", "lat": 46.18722, "lon": -59.89972 }],
  "CAN-r":  [{"name": "Resolute, Cornwallis Island NU", "code": "CAN-r", "lat": 74.74639, "lon": -95.00306 }],
  "CAN-sa": [{"name": "St Anthony NL", "code": "CAN-sa", "lat": 51.50000, "lon": -55.82389 }],
  "CAN-sj": [{"name": "St John's NL", "code": "CAN-sj", "lat": 47.61111, "lon": -52.66694 }],
  "CAN-sl": [{"name": "St Lawrence NL", "code": "CAN-sl", "lat": 46.91917, "lon": -55.37917 }],
  "CAN-sm": [{"name": "Sambro NS", "code": "CAN-sm", "lat": 44.47250, "lon": -63.62028 }],
  "CAN-sv": [{"name": "Stephenville NL", "code": "CAN-sv", "lat": 48.55472, "lon": -58.75889 }],
  "CAN-t":  [{"name": "Toronto (Mississauga/Clarkson) ON", "code": "CAN-t", "lat": 43.50639, "lon": -79.63361 }],
  "CAN-tr": [{"name": "Trenton (Pointe Petre, Lake Ontario)", "code": "CAN-tr", "lat": 43.84417, "lon": -77.14639 }],
  "CAN-tr2": [{"name": "Trenton Receiver Site ON", "code": "CAN-tr2", "lat": 44.03222, "lon": -77.55056 }],
  "CAN-v":  [{"name": "Vancouver BC", "code": "CAN-v", "lat": 49.13917, "lon": -123.19556 }],
  "CAN-ym": [{"name": "Yarmouth/Chebogue NS", "code": "CAN-ym", "lat": 43.74417, "lon": -66.12250 }],
  "CBG-ka": [{"name": "Kandal", "code": "CBG-ka", "lat": 11.41667, "lon": 104.83333 }],
  "CHL-a":  [{"name": "Antofagasta", "code": "CHL-a", "lat": -23.66667, "lon": -70.40000 }],
  "CHL-e":  [{"name": "Radio Esperanza", "code": "CHL-e", "lat": -38.68333, "lon": -72.58333 }],
  "CHL-fx": [{"name": "Bahia Felix", "code": "CHL-fx", "lat": -52.96194, "lon": -74.08083 }],
  "CHL-jf": [{"name": "Juan Fernández", "code": "CHL-jf", "lat": -33.63333, "lon": -78.83333 }],
  "CHL-pa": [{"name": "Punta Arenas", "code": "CHL-pa", "lat": -53.16667, "lon": -70.90000 }],
  "CHL-pm": [{"name": "Puerto Montt", "code": "CHL-pm", "lat": -41.65556, "lon": -73.17333 }],
  "CHL-s":  [{"name": "Santiago (Calera de Tango)", "code": "CHL-s", "lat": -33.64333, "lon": -70.85056 }],
  "CHL-t":  [{"name": "Talagante", "code": "CHL-t", "lat": -33.66667, "lon": -70.93333 }],
  "CHL-tq": [{"name": "Talahuano, Quiriquina Island", "code": "CHL-tq", "lat": -36.61667, "lon": -73.06667 }],
  "CHL-v":  [{"name": "Valparaiso 32S48-71W29 or", "code": "CHL-v", "lat": -33.02028, "lon": -71.64222 }],
  "CHL-w":  [{"name": "Wollaston Island", "code": "CHL-w", "lat": -55.61667, "lon": -67.43333 }],
  "CHN-a":  [{"name": "Baoji-Xinjie (Shaanxi; CRI 150 kW; CNR2 9820) '722'", "code": "CHN-a", "lat": 34.50000, "lon": 107.16667 }],
  "CHN-as": [{"name": "Baoji-Sifangshan (Shaanxi; CNR1,8) '724'", "code": "CHN-as", "lat": 37.45000, "lon": 107.68333 }],
  "CHN-b":  [{"name": "Beijing-Matoucun '572' (100 kW CNR1)", "code": "CHN-b", "lat": 39.75000, "lon": 116.81667 }],
  "CHN-B":  [{"name": "Beijing-Chaoyang/Gaobeidian/Shuangqiao '491' (CNR2-8)", "code": "CHN-B", "lat": 39.88333, "lon": 116.56667 }],
  "CHN-bd": [{"name": "Beijing-Doudian (150/500 kW CRI) '564'", "code": "CHN-bd", "lat": 39.63333, "lon": 116.08333 }],
  "CHN-bm": [{"name": "Beijing BAF", "code": "CHN-bm", "lat": 39.90000, "lon": 116.46667 }],
  "CHN-bs": [{"name": "Beijing 3SD", "code": "CHN-bs", "lat": 39.70000, "lon": 115.91667 }],
  "CHN-b0": [{"name": "Basuo, Hainan", "code": "CHN-b0", "lat": 19.09611, "lon": 108.63444 }],
  "CHN-c":  [{"name": "Chengdu (Sichuan)", "code": "CHN-c", "lat": 30.90000, "lon": 104.11667 }],
  "CHN-cc": [{"name": "Changchun '523' (Jilin)", "code": "CHN-cc", "lat": 44.02889, "lon": 125.41889 }],
  "CHN-ch": [{"name": "Changzhou Henglinchen '623' (Jiangsu)", "code": "CHN-ch", "lat": 31.70917, "lon": 120.11222 }],
  "CHN-d":  [{"name": "Dongfang (Hainan)", "code": "CHN-d", "lat": 18.88333, "lon": 108.65000 }],
  "CHN-da": [{"name": "Dalian", "code": "CHN-da", "lat": 38.91667, "lon": 121.65000 }],
  "CHN-db": [{"name": "Dongfang-Basuo", "code": "CHN-db", "lat": 19.10000, "lon": 108.61667 }],
  "CHN-e":  [{"name": "Gejiu (Yunnan)", "code": "CHN-e", "lat": 23.35000, "lon": 103.13333 }],
  "CHN-eb": [{"name": "Beijing, Posolstvo", "code": "CHN-eb", "lat": 39.91667, "lon": 116.45000 }],
  "CHN-f":  [{"name": "Fuzhou (Fujian)", "code": "CHN-f", "lat": 26.10000, "lon": 119.40000 }],
  "CHN-fz": [{"name": "Fuzhou-Mawei XSL (Fujian)", "code": "CHN-fz", "lat": 26.016667, "lon": 119.450000 }],
  "CHN-fz": [{"name": "Tailu", "code": "CHN-fz", "lat": 26.36861, "lon": 119.93944 }],
  "CHN-g":  [{"name": "Gannan (Hezuo)", "code": "CHN-g", "lat": 34.97500, "lon": 102.91667 }],
  "CHN-gu": [{"name": "Gutian-Xincheng", "code": "CHN-gu", "lat": 26.56667, "lon": 118.73333 }],
  "CHN-gx": [{"name": "Guangzhou XSQ", "code": "CHN-gx", "lat": 23.15750, "lon": 113.51417 }],
  "CHN-gy": [{"name": "Guiyang", "code": "CHN-gy", "lat": 26.41667, "lon": 106.60000 }],
  "CHN-gz": [{"name": "Guangzhou-Huadu (Guangdong)", "code": "CHN-gz", "lat": 23.40000, "lon": 113.23333 }],
  "CHN-h":  [{"name": "Hohhot '694' (Nei Menggu, CRI)", "code": "CHN-h", "lat": 40.80000, "lon": 111.78333 }],
  "CHN-ha": [{"name": "Hailar (Nei Menggu)", "code": "CHN-ha", "lat": 49.18333, "lon": 119.71667 }],
  "CHN-hd": [{"name": "Huadian '763' (Jilin)", "code": "CHN-hd", "lat": 43.11667, "lon": 126.51667 }],
  "CHN-he": [{"name": "Hezuo", "code": "CHN-he", "lat": 34.97056, "lon": 102.90889 }],
  "CHN-hh": [{"name": "Hohhot-Yijianfang (Nei Menggu, PBS NM)", "code": "CHN-hh", "lat": 40.71667, "lon": 111.55000 }],
  "CHN-hk": [{"name": "Haikou (Hainan) XSR", "code": "CHN-hk", "lat": 20.06667, "lon": 110.70000 }],
  "CHN-hu": [{"name": "Hutubi (Xinjiang)", "code": "CHN-hu", "lat": 44.16667, "lon": 86.90000 }],
  "CHN-j":  [{"name": "Jinhua", "code": "CHN-j", "lat": 29.11667, "lon": 119.31667 }],
  "CHN-k":  [{"name": "Kunming-Anning CRI (Yunnan)", "code": "CHN-k", "lat": 24.88333, "lon": 102.50000 }],
  "CHN-ka": [{"name": "Kashi (Kashgar) (Xinjiang)", "code": "CHN-ka", "lat": 39.35000, "lon": 75.76667 }],
  "CHN-kl": [{"name": "Kunming-Lantao PBS (Yunnan)", "code": "CHN-kl", "lat": 25.16667, "lon": 102.83333 }],
  "CHN-L":  [{"name": "Lingshi '725' (Shanxi)", "code": "CHN-L", "lat": 36.86667, "lon": 111.66667 }],
  "CHN-ly": [{"name": "Lianyungang, Jiangsu", "code": "CHN-ly", "lat": 34.70111, "lon": 119.31250 }],
  "CHN-n":  [{"name": "Nanning (Guangxi) '954'", "code": "CHN-n", "lat": 22.78333, "lon": 108.18333 }],
  "CHN-nj": [{"name": "Nanjing (Jiangsu)", "code": "CHN-nj", "lat": 32.03333, "lon": 118.73333 }],
  "CHN-p":  [{"name": "Pucheng (Shaanxi)", "code": "CHN-p", "lat": 35.00000, "lon": 109.51667 }],
  "CHN-pt": [{"name": "Putian (Fujian)", "code": "CHN-pt", "lat": 25.46667, "lon": 119.16667 }],
  "CHN-q":  [{"name": "Ge'ermu/Golmud '916' (Qinghai)", "code": "CHN-q", "lat": 36.43333, "lon": 95.00000 }],
  "CHN-qq": [{"name": "Qiqihar", "code": "CHN-qq", "lat": 47.033333, "lon": 124.5 }],
  "CHN-qz": [{"name": "Quanzhou '641' (Fujian)", "code": "CHN-qz", "lat": 24.88333, "lon": 118.80000 }],
  "CHN-s":  [{"name": "Shijiazhuang '723' (Hebei; Nanpozhuang CRI 500 kW; Huikou CNR 100 kW)", "code": "CHN-s", "lat": 38.21667, "lon": 114.10000 }],
  "CHN-sg": [{"name": "Shanghai-Taopuzhen", "code": "CHN-sg", "lat": 31.25000, "lon": 121.48333 }],
  "CHN-sh": [{"name": "Shanghai XSG", "code": "CHN-sh", "lat": 31.10000, "lon": 121.53333 }],
  "CHN-sn": [{"name": "Sanya (Hainan)", "code": "CHN-sn", "lat": 18.23333, "lon": 109.31667 }],
  "CHN-sq": [{"name": "Shangqiu (Henan)", "code": "CHN-sq", "lat": 34.94833, "lon": 109.54278 }],
  "CHN-st": [{"name": "Shantou (Guangdong)", "code": "CHN-st", "lat": 23.36667, "lon": 116.70000 }],
  "CHN-sw": [{"name": "Nanping-Shaowu (Fujian)", "code": "CHN-sw", "lat": 27.08333, "lon": 117.28333 }],
  "CHN-sy": [{"name": "Shuangyashan '128' (Heilongjiang)", "code": "CHN-sy", "lat": 46.72194, "lon": 131.21111 }],
  "CHN-t":  [{"name": "Tibet (Lhasa-Baiding '602')", "code": "CHN-t", "lat": 29.65000, "lon": 91.25000 }],
  "CHN-tj": [{"name": "Tianjin", "code": "CHN-tj", "lat": 39.05000, "lon": 117.42500 }],
  "CHN-u":  [{"name": "Urumqi (Xinjiang, CRI)", "code": "CHN-u", "lat": 44.14639, "lon": 86.89528 }],
  "CHN-uc": [{"name": "Urumqi-Changji (Xinjiang, PBS XJ)", "code": "CHN-uc", "lat": 43.97389, "lon": 87.24889 }],
  "CHN-x":  [{"name": "Xian-Xianyang '594' (Shaanxi)", "code": "CHN-x", "lat": 34.20000, "lon": 108.90000 }],
  "CHN-xc": [{"name": "Xichang (Sichuan)", "code": "CHN-xc", "lat": 27.81667, "lon": 102.23333 }],
  "CHN-xd": [{"name": "Xiamen-Xiangdian (Fujian) XSM", "code": "CHN-xd", "lat": 24.504722, "lon": 118.143611 }],
  "CHN-xd": [{"name": "Dong'an", "code": "CHN-xd", "lat": 24.59833, "lon": 118.11833 }],
  "CHN-xg": [{"name": "Xining (Qinghai)", "code": "CHN-xg", "lat": 36.65000, "lon": 101.58333 }],
  "CHN-xm": [{"name": "Xiamen (Fujian)", "code": "CHN-xm", "lat": 24.49222, "lon": 118.07306 }],
  "CHN-xt": [{"name": "Xiangtan (Hunan)", "code": "CHN-xt", "lat": 27.50000, "lon": 112.50000 }],
  "CHN-xw": [{"name": "Xuanwei (Yunnan)", "code": "CHN-xw", "lat": 26.15000, "lon": 104.03333 }],
  "CHN-xy": [{"name": "Xingyang (Henan)", "code": "CHN-xy", "lat": 34.80000, "lon": 113.38333 }],
  "CHN-xz": [{"name": "Xinzhaicun (Fujian)", "code": "CHN-xz", "lat": 25.75000, "lon": 117.18333 }],
  "CHN-ya": [{"name": "Yanbian-Yanji (Jilin)", "code": "CHN-ya", "lat": 42.79167, "lon": 129.48833 }],
  "CHN-yt": [{"name": "Yantai (Shandong)", "code": "CHN-yt", "lat": 37.70639, "lon": 121.13583 }],
  "CHN-zh": [{"name": "Zhuhai '909' (Guangdong)", "code": "CHN-zh", "lat": 22.38333, "lon": 113.55000 }],
  "CHN-zj": [{"name": "Zhanjiang (Guangdong)", "code": "CHN-zj", "lat": 21.18333, "lon": 110.40000 }],
  "CHN-zs": [{"name": "Mount Putuo, Xiaohulu Island, Zhoushan", "code": "CHN-zs", "lat": 30.00000, "lon": 122.38333 }],
  "CKH-rt": [{"name": "Rarotonga", "code": "CKH-rt", "lat": -21.20000, "lon": -159.81667 }],
  "CLM-b":  [{"name": "Barranquilla", "code": "CLM-b", "lat": 10.91667, "lon": -74.76667 }],
  "CLM-bu": [{"name": "Buenaventura", "code": "CLM-bu", "lat": 3.88333, "lon": -77.03333 }],
  "CLM-pl": [{"name": "Puerto Lleras", "code": "CLM-pl", "lat": 3.26667, "lon": -73.36667 }],
  "CLM-r":  [{"name": "Rioblanco, Tolima", "code": "CLM-r", "lat": 3.50000, "lon": -75.83333 }],
  "CLM-sa": [{"name": "San Andrés Island (SAP)", "code": "CLM-sa", "lat": 12.55000, "lon": -81.71667 }],
  "CLN-e":  [{"name": "Ekala (SLBC,RJ)", "code": "CLN-e", "lat": 7.10000, "lon": 79.90000 }],
  "CLN-i":  [{"name": "Iranawila (IBB)", "code": "CLN-i", "lat": 7.51667, "lon": 79.80000 }],
  "CLN-p":  [{"name": "Puttalam", "code": "CLN-p", "lat": 7.98333, "lon": 79.80000 }],
  "CLN-t":  [{"name": "Trincomalee (DW)", "code": "CLN-t", "lat": 8.73333, "lon": 81.16667 }],
  "CME":    [{"name": "Buea", "code": "CME", "lat": 4.15000, "lon": 9.23333 }],
  "CNR-ar": [{"name": "Arrecife (Lanzarote)", "code": "CNR-ar", "lat": 29.13333, "lon": -13.51667 }],
  "CNR-fc": [{"name": "Fuencaliente (Las Palmas)", "code": "CNR-fc", "lat": 28.50889, "lon": -17.83944 }],
  "CNR-gc": [{"name": "Gran Canaria airport", "code": "CNR-gc", "lat": 27.95000, "lon": -15.38333 }],
  "CNR-hr": [{"name": "Haría (Tenerife)", "code": "CNR-hr", "lat": 29.14083, "lon": -13.51722 }],
  "CNR-hy": [{"name": "Los Hoyos (Gran Canaria)", "code": "CNR-hy", "lat": 28.04861, "lon": -15.44972 }],
  "CNR-lm": [{"name": "Las Mesas (Las Palmas)", "code": "CNR-lm", "lat": 28.48278, "lon": -16.26944 }],
  "CNR-pr": [{"name": "Puerto del Rosario", "code": "CNR-pr", "lat": 28.54361, "lon": -13.87806 }],
  "COD-bk": [{"name": "Bukavu", "code": "COD-bk", "lat": -2.50000, "lon": 28.83333 }],
  "COD-bu": [{"name": "Bunia", "code": "COD-bu", "lat": 1.53333, "lon": 30.18333 }],
  "COG-b":  [{"name": "Brazzaville-M'Pila", "code": "COG-b", "lat": -4.25000, "lon": 15.30000 }],
  "COG-bv": [{"name": "Brazzaville Volmet", "code": "COG-bv", "lat": -4.23306, "lon": 15.26167 }],
  "COG-pn": [{"name": "Pointe Noire", "code": "COG-pn", "lat": -4.78333, "lon": 11.86667 }],
  "CPV":    [{"name": "Cape Verde", "code": "CPV", "lat": 14.916389, "lon": -23.509444 }],
  "CTI-a":  [{"name": "Abidjan", "code": "CTI-a", "lat": 5.36667, "lon": -3.96667 }],
  "CTR":    [{"name": "Cariari de Pococí (REE)", "code": "CTR", "lat": 10.41667, "lon": -83.71667 }],
  "CTR-g":  [{"name": "Guápiles (Canton de Pococí, Prov.de Limón) ELCOR", "code": "CTR-g", "lat": 10.21667, "lon": -83.78333 }],
  "CUB":    [{"name": "La Habana sites Quivicán/Bejucal/Bauta", "code": "CUB", "lat": 23.00000, "lon": -82.50000 }],
  "CUB-b":  [{"name": "Bauta (Centro Transmisor No.1)", "code": "CUB-b", "lat": 22.95000, "lon": -82.55000 }],
  "CUB-be": [{"name": "Bejucal (Centro Transmisor No.2)", "code": "CUB-be", "lat": 22.86667, "lon": -82.33333 }],
  "CUB-hr": [{"name": "Havana Radio", "code": "CUB-hr", "lat": 23.16667, "lon": -82.31667 }],
  "CUB-q":  [{"name": "Quivicán/Titan (Centro Transmisor No.3)", "code": "CUB-q", "lat": 22.83333, "lon": -82.30000 }],
  "CVA":    [{"name": "Santa Maria di Galeria", "code": "CVA", "lat": 42.05000, "lon": 12.31667 }],
  "CVA-v":  [{"name": "Citta del Vaticano", "code": "CVA-v", "lat": 41.90000, "lon": 12.45000 }],
  "CYP-a":  [{"name": "Akrotiri (UK territory)", "code": "CYP-a", "lat": 34.61667, "lon": 32.93333 }],
  "CYP-cr": [{"name": "Cyprus Radio", "code": "CYP-cr", "lat": 35.05000, "lon": 33.28333 }],
  "CYP-g":  [{"name": "Cape Greco", "code": "CYP-g", "lat": 34.95000, "lon": 34.08333 }],
  "CYP-m":  [{"name": "Lady's Mile (UK territory)", "code": "CYP-m", "lat": 34.61667, "lon": 33.00000 }],
  "CYP-n":  [{"name": "Nicosia", "code": "CYP-n", "lat": 35.16667, "lon": 33.35000 }],
  "CYP-y":  [{"name": "Yeni Iskele", "code": "CYP-y", "lat": 35.28333, "lon": 33.91667 }],
  "CZE-b":  [{"name": "Brno-Dobrochov", "code": "CZE-b", "lat": 49.38333, "lon": 17.13333 }],
  "CZE-cb": [{"name": "Ceske Budejovice-Husova kolonie", "code": "CZE-cb", "lat": 48.99278, "lon": 14.49361 }],
  "CZE-dl": [{"name": "Dlouhá Louka", "code": "CZE-dl", "lat": 50.64806, "lon": 13.65611 }],
  "CZE-kv": [{"name": "Karlovy Vary-Stará Role", "code": "CZE-kv", "lat": 50.23944, "lon": 12.82389 }],
  "CZE-mb": [{"name": "Moravské Budejovice-Domamil", "code": "CZE-mb", "lat": 49.07639, "lon": 15.70667 }],
  "CZE-os": [{"name": "Ostrava-Svinov", "code": "CZE-os", "lat": 49.81139, "lon": 18.19333 }],
  "CZE-p":  [{"name": "Praha", "code": "CZE-p", "lat": 50.03333, "lon": 14.48333 }],
  "CZE-pl": [{"name": "Praha-Liblice", "code": "CZE-pl", "lat": 50.07861, "lon": 14.88694 }],
  "CZE-pr": [{"name": "Pruhonice / Pruhonice", "code": "CZE-pr", "lat": 49.99111, "lon": 14.53806 }],
  "CZE-pv": [{"name": "Panská Ves", "code": "CZE-pv", "lat": 50.52806, "lon": 14.56694 }],
  "CZE-tr": [{"name": "Trebon / Trebon", "code": "CZE-tr", "lat": 49.00000, "lon": 14.76667 }],
  "CZE-va": [{"name": "Vackov", "code": "CZE-va", "lat": 50.23333, "lon": 12.38333 }],
  "D-al":   [{"name": "Albersloh", "code": "D-al", "lat": 51.88639, "lon": 7.72194 }],
  "D-b":    [{"name": "Biblis", "code": "D-b", "lat": 49.68833, "lon": 8.49222 }],
  "D-be":   [{"name": "Berlin-Britz", "code": "D-be", "lat": 52.45000, "lon": 13.43333 }],
  "D-bl":   [{"name": "Berlin", "code": "D-bl", "lat": 52.51667, "lon": 13.38333 }],
  "D-br":   [{"name": "Braunschweig", "code": "D-br", "lat": 52.28333, "lon": 10.71667 }],
  "D-bu":   [{"name": "Burg", "code": "D-bu", "lat": 52.28694, "lon": 11.89694 }],
  "D-bv":   [{"name": "Bonn-Venusberg", "code": "D-bv", "lat": 50.70806, "lon": 7.09667 }],
  "D-cx":   [{"name": "Cuxhaven-Sahlenburg", "code": "D-cx", "lat": 53.86389, "lon": 8.62556 }],
  "D-d":    [{"name": "Dillberg", "code": "D-d", "lat": 49.31667, "lon": 11.38333 }],
  "D-dd":   [{"name": "Dresden-Wilsdruff", "code": "D-dd", "lat": 51.05861, "lon": 13.50750 }],
  "D-dt":   [{"name": "Datteln", "code": "D-dt", "lat": 51.65000, "lon": 7.35000 }],
  "D-e":    [{"name": "Erlangen-Tennenlohe", "code": "D-e", "lat": 49.58333, "lon": 11.00000 }],
  "D-fl":   [{"name": "Flensburg", "code": "D-fl", "lat": 54.79167, "lon": 9.50333 }],
  "D-g":    [{"name": "Goehren", "code": "D-g", "lat": 53.53556, "lon": 11.61111 }],
  "D-ge":   [{"name": "Gera", "code": "D-ge", "lat": 50.88333, "lon": 12.08333 }],
  "D-ha":   [{"name": "Hannover", "code": "D-ha", "lat": 52.38333, "lon": 9.70000 }],
  "D-hc":   [{"name": "Hamburg-Curslack", "code": "D-hc", "lat": 53.45000, "lon": 10.21667 }],
  "D-he":   [{"name": "Hannover/Hemmingen", "code": "D-he", "lat": 52.32778, "lon": 9.73667 }],
  "D-hh":   [{"name": "Hamburg-Moorfleet", "code": "D-hh", "lat": 53.51917, "lon": 10.10278 }],
  "D-ht":   [{"name": "Hartenstein (Sachsen)", "code": "D-ht", "lat": 50.66667, "lon": 12.66667 }],
  "D-jr":   [{"name": "Juliusruh", "code": "D-jr", "lat": 54.62917, "lon": 13.37389 }],
  "D-k":    [{"name": "Kall-Krekel", "code": "D-k", "lat": 50.47806, "lon": 6.52306 }],
  "D-L":    [{"name": "Lampertheim", "code": "D-L", "lat": 49.60472, "lon": 8.53889 }],
  "D-la":   [{"name": "Langenberg", "code": "D-la", "lat": 51.35611, "lon": 7.13417 }],
  "D-li":   [{"name": "Lingen", "code": "D-li", "lat": 52.53500, "lon": 7.35306 }],
  "D-mf":   [{"name": "Mainflingen", "code": "D-mf", "lat": 50.01556, "lon": 9.01194 }],
  "D-n":    [{"name": "Nauen", "code": "D-n", "lat": 52.64861, "lon": 12.90917 }],
  "D-nh":   [{"name": "Neuharlingersiel DHJ59", "code": "D-nh", "lat": 53.67639, "lon": 7.61250 }],
  "D-nu":   [{"name": "Nuernberg", "code": "D-nu", "lat": 49.45000, "lon": 11.08333 }],
  "D-or":   [{"name": "Oranienburg", "code": "D-or", "lat": 52.78333, "lon": 13.38333 }],
  "D-pi":   [{"name": "Pinneberg", "code": "D-pi", "lat": 53.67306, "lon": 9.80833 }],
  "D-r":    [{"name": "Rohrbach", "code": "D-r", "lat": 48.60000, "lon": 11.55000 }],
  "D-rf":   [{"name": "Rhauderfehn", "code": "D-rf", "lat": 53.08333, "lon": 7.61667 }],
  "D-s":    [{"name": "Stade", "code": "D-s", "lat": 53.60000, "lon": 9.46667 }],
  "D-w":    [{"name": "Wertachtal", "code": "D-w", "lat": 48.08694, "lon": 10.69500 }],
  "D-wa":   [{"name": "Winsen (Aller)", "code": "D-wa", "lat": 52.66667, "lon": 9.76667 }],
  "D-wb":   [{"name": "Wachenbrunn", "code": "D-wb", "lat": 50.48556, "lon": 10.55833 }],
  "D-we":   [{"name": "Weenermoor", "code": "D-we", "lat": 53.20000, "lon": 7.31667 }],
  "DJI-d":  [{"name": "Djibouti", "code": "DJI-d", "lat": 11.50000, "lon": 43.00000 }],
  "DJI-i":  [{"name": "Centre de Transmissions Interarmées FUV", "code": "DJI-i", "lat": 11.53583, "lon": 43.15556 }],
  "DNK-a":  [{"name": "Aarhus-Mårslet", "code": "DNK-a", "lat": 56.15000, "lon": 10.21667 }],
  "DNK-bl": [{"name": "Blaavand", "code": "DNK-bl", "lat": 55.55000, "lon": 8.10000 }],
  "DNK-br": [{"name": "Bramming", "code": "DNK-br", "lat": 55.46667, "lon": 8.70000 }],
  "DNK-bv": [{"name": "Bovbjerg", "code": "DNK-bv", "lat": 56.51667, "lon": 8.16667 }],
  "DNK-co": [{"name": "Copenhagen OXT", "code": "DNK-co", "lat": 55.83333, "lon": 11.41667 }],
  "DNK-f":  [{"name": "Frederikshavn", "code": "DNK-f", "lat": 57.43333, "lon": 10.53333 }],
  "DNK-h":  [{"name": "Hillerod", "code": "DNK-h", "lat": 55.90000, "lon": 12.26667 }],
  "DNK-hv": [{"name": "Copenhagen Hvidovre", "code": "DNK-hv", "lat": 55.65000, "lon": 12.48333 }],
  "DNK-i":  [{"name": "Copenhagen Ishøj", "code": "DNK-i", "lat": 55.61667, "lon": 12.35000 }],
  "DNK-k":  [{"name": "Kalundborg", "code": "DNK-k", "lat": 55.67639, "lon": 11.06944 }],
  "DNK-ra": [{"name": "Randers", "code": "DNK-ra", "lat": 56.46667, "lon": 10.03333 }],
  "DNK-ro": [{"name": "Ronne", "code": "DNK-ro", "lat": 55.03333, "lon": 15.10000 }],
  "DNK-sg": [{"name": "Skagen", "code": "DNK-sg", "lat": 57.73333, "lon": 10.56667 }],
  "DNK-sk": [{"name": "Skamlebaek", "code": "DNK-sk", "lat": 55.83333, "lon": 11.41667 }],
  "DOM-sd": [{"name": "Santo Domingo", "code": "DOM-sd", "lat": 18.46667, "lon": -69.88333 }],
  "E-af":   [{"name": "Alfabia (Mallorca)", "code": "E-af", "lat": 39.73750, "lon": 2.71806 }],
  "E-ag":   [{"name": "Aguilas", "code": "E-ag", "lat": 37.49028, "lon": -1.56333 }],
  "E-ar":   [{"name": "Ares", "code": "E-ar", "lat": 43.45278, "lon": -8.28333 }],
  "E-as":   [{"name": "La Asomada", "code": "E-as", "lat": 37.63000, "lon": -0.96333 }],
  "E-bo":   [{"name": "Boal", "code": "E-bo", "lat": 43.45639, "lon": -6.82056 }],
  "E-c":    [{"name": "Coruna", "code": "E-c", "lat": 43.36694, "lon": -8.45194 }],
  "E-cg":   [{"name": "Cabo de Gata - Sabinar", "code": "E-cg", "lat": 37.20806, "lon": -7.01833 }],
  "E-cp":   [{"name": "Chipiona", "code": "E-cp", "lat": 36.66667, "lon": -6.40000 }],
  "E-fi":   [{"name": "Finisterre", "code": "E-fi", "lat": 42.90000, "lon": -9.26667 }],
  "E-gm":   [{"name": "Torreta de Guardamar, Guardamar del Segura", "code": "E-gm", "lat": 38.07167, "lon": -0.66472 }],
  "E-h":    [{"name": "Huelva", "code": "E-h", "lat": 37.20000, "lon": -7.01667 }],
  "E-hv":   [{"name": "'Huelva'", "code": "E-hv", "lat": 43.34472, "lon": -1.85583 }],
  "E-jq":   [{"name": "Jaizquibel", "code": "E-jq", "lat": 43.34472, "lon": -1.85583 }],
  "E-ma":   [{"name": "Madrid", "code": "E-ma", "lat": 40.46667, "lon": -3.66667 }],
  "E-mu":   [{"name": "Muxía", "code": "E-mu", "lat": 43.07722, "lon": -9.22500 }],
  "E-mx":   [{"name": "Marratxí", "code": "E-mx", "lat": 39.63472, "lon": 2.67000 }],
  "E-n":    [{"name": "Noblejas", "code": "E-n", "lat": 39.95000, "lon": -3.43333 }],
  "E-pm":   [{"name": "Palma de Mallorca", "code": "E-pm", "lat": 39.36667, "lon": 2.78333 }],
  "E-pz":   [{"name": "Pastoriza", "code": "E-pz", "lat": 42.34306, "lon": -8.71917 }],
  "E-rq":   [{"name": "Roquetas", "code": "E-rq", "lat": 36.26611, "lon": -6.01333 }],
  "E-rs":   [{"name": "Rostrío/Cabo de Peñas", "code": "E-rs", "lat": 43.47833, "lon": -5.85028 }],
  "E-sb":   [{"name": "Sabiner", "code": "E-sb", "lat": 36.68333, "lon": -2.70000 }],
  "E-ta":   [{"name": "Tarifa", "code": "E-ta", "lat": 36.05000, "lon": -5.55000 }],
  "E-tj":   [{"name": "Trijueque", "code": "E-tj", "lat": 40.77861, "lon": -2.98528 }],
  "E-to":   [{"name": "Torrejón de Ardoz (Pegaso, Pavon, Brujo)", "code": "E-to", "lat": 40.50000, "lon": -3.45000 }],
  "E-vj":   [{"name": "Vejer", "code": "E-vj", "lat": 43.47833, "lon": -3.85028 }],
  "EGY-a":  [{"name": "Abis", "code": "EGY-a", "lat": 31.16667, "lon": 30.08333 }],
  "EGY-al": [{"name": "Alexandria / Al-Iskandaria", "code": "EGY-al", "lat": 31.20000, "lon": 29.90000 }],
  "EGY-ca": [{"name": "Cairo", "code": "EGY-ca", "lat": 30.06667, "lon": 31.21667 }],
  "EGY-ea": [{"name": "El Arish", "code": "EGY-ea", "lat": 31.11194, "lon": 33.69972 }],
  "EGY-z":  [{"name": "Abu Zaabal", "code": "EGY-z", "lat": 30.26667, "lon": 31.36667 }],
  "EQA-a":  [{"name": "Ambato", "code": "EQA-a", "lat": -1.21667, "lon": -78.61667 }],
  "EQA-c":  [{"name": "Pico Pichincha", "code": "EQA-c", "lat": -0.18333, "lon": -78.53333 }],
  "EQA-g":  [{"name": "Guayaquil", "code": "EQA-g", "lat": -2.26667, "lon": -79.90000 }],
  "EQA-i":  [{"name": "Ibarra", "code": "EQA-i", "lat": 0.35000, "lon": -78.13333 }],
  "EQA-o":  [{"name": "Otavalo", "code": "EQA-o", "lat": 0.30000, "lon": -78.18333 }],
  "EQA-p":  [{"name": "Pifo", "code": "EQA-p", "lat": -0.23333, "lon": -78.33333 }],
  "EQA-s":  [{"name": "Saraguro", "code": "EQA-s", "lat": -3.70000, "lon": -79.30000 }],
  "EQA-t":  [{"name": "Tena", "code": "EQA-t", "lat": -1.00000, "lon": -77.80000 }],
  "EQA-u":  [{"name": "Sucúa", "code": "EQA-u", "lat": -2.46667, "lon": -78.16667 }],
  "ERI":    [{"name": "Asmara-Saladaro", "code": "ERI", "lat": 15.21667, "lon": 38.86667 }],
  "EST-ta": [{"name": "Tallinn Radio", "code": "EST-ta", "lat": 59.45000, "lon": 24.75000 }],
  "EST-tt": [{"name": "Tartu", "code": "EST-tt", "lat": 58.41667, "lon": 27.10000 }],
  "EST-tv": [{"name": "Tallinn Volmet", "code": "EST-tv", "lat": 59.41667, "lon": 24.83333 }],
  "ETH-a":  [{"name": "Addis Abeba", "code": "ETH-a", "lat": 9.00000, "lon": 38.75000 }],
  "ETH-ad": [{"name": "Adama", "code": "ETH-ad", "lat": 8.53333, "lon": 39.26667 }],
  "ETH-b":  [{"name": "Bahir Dar", "code": "ETH-b", "lat": 11.60000, "lon": 37.38333 }],
  "ETH-d":  [{"name": "Geja Dera (HS)", "code": "ETH-d", "lat": 8.76667, "lon": 38.66667 }],
  "ETH-j":  [{"name": "Geja Jewe (FS)", "code": "ETH-j", "lat": 8.71667, "lon": 38.63333 }],
  "ETH-jj": [{"name": "Jijiga", "code": "ETH-jj", "lat": 9.35000, "lon": 42.80000 }],
  "ETH-m":  [{"name": "Mekele", "code": "ETH-m", "lat": 13.48333, "lon": 39.48333 }],
  "ETH-n":  [{"name": "Nekemte", "code": "ETH-n", "lat": 9.08333, "lon": 36.55000 }],
  "ETH-r":  [{"name": "Robe", "code": "ETH-r", "lat": 7.11667, "lon": 40.00000 }],
  "F-a":    [{"name": "Allouis", "code": "F-a", "lat": 47.16667, "lon": 2.20000 }],
  "F-au":   [{"name": "Auros", "code": "F-au", "lat": 44.50000, "lon": -0.15000 }],
  "F-av":   [{"name": "Avord", "code": "F-av", "lat": 47.05000, "lon": 2.46667 }],
  "F-br":   [{"name": "Brest", "code": "F-br", "lat": 48.42583, "lon": -4.24083 }],
  "F-cm":   [{"name": "Col de la Madone", "code": "F-cm", "lat": 43.78333, "lon": 7.41667 }],
  "F-co":   [{"name": "Corsen", "code": "F-co", "lat": 48.41667, "lon": -4.78333 }],
  "F-g":    [{"name": "La Garde (Toulon)", "code": "F-g", "lat": 43.10528, "lon": 5.98917 }],
  "F-gn":   [{"name": "Gris-Nez", "code": "F-gn", "lat": 50.86667, "lon": 1.58333 }],
  "F-hy":   [{"name": "Hyères Island", "code": "F-hy", "lat": 42.98444, "lon": 6.20667 }],
  "F-i":    [{"name": "Issoudun", "code": "F-i", "lat": 46.93333, "lon": 1.90000 }],
  "F-jb":   [{"name": "Jobourg", "code": "F-jb", "lat": 49.68472, "lon": -1.90778 }],
  "F-ma":   [{"name": "Mont Angel/Fontbonne", "code": "F-ma", "lat": 43.76667, "lon": 7.42500 }],
  "F-oe":   [{"name": "Ouessant", "code": "F-oe", "lat": 48.46667, "lon": -5.05000 }],
  "F-p":    [{"name": "Paris", "code": "F-p", "lat": 48.86667, "lon": 2.30000 }],
  "F-r":    [{"name": "Rennes", "code": "F-r", "lat": 48.10000, "lon": -1.68333 }],
  "F-ro":   [{"name": "Roumoules", "code": "F-ro", "lat": 43.78333, "lon": 6.15000 }],
  "F-sa":   [{"name": "Saissac (11)", "code": "F-sa", "lat": 43.39028, "lon": 2.09972 }],
  "F-sb":   [{"name": "Strasbourg", "code": "F-sb", "lat": 48.24972, "lon": 7.44361 }],
  "F-sg":   [{"name": "Saint Guénolé", "code": "F-sg", "lat": 47.81667, "lon": -4.38333 }],
  "F-to":   [{"name": "Toulon", "code": "F-to", "lat": 43.13556, "lon": 6.05972 }],
  "F-v":    [{"name": "Favières FAV", "code": "F-v", "lat": 48.53333, "lon": 1.23333 }],
  "F-ve":   [{"name": "Vernon", "code": "F-ve", "lat": 49.08333, "lon": 1.50000 }],
  "F-wu":   [{"name": "Rosnay (HWU)", "code": "F-wu", "lat": 46.71667, "lon": 1.23333 }],
  "FIN-ha": [{"name": "Hailuoto (Oulu)", "code": "FIN-ha", "lat": 65.00000, "lon": 24.73333 }],
  "FIN-he": [{"name": "Helsinki", "code": "FIN-he", "lat": 60.15000, "lon": 24.73333 }],
  "FIN-mh": [{"name": "Mariehamn (Aland Islands)", "code": "FIN-mh", "lat": 60.10000, "lon": 19.93333 }],
  "FIN-o":  [{"name": "Ovaniemi", "code": "FIN-o", "lat": 60.18028, "lon": 24.82639 }],
  "FIN-p":  [{"name": "Pori", "code": "FIN-p", "lat": 61.46667, "lon": 21.58333 }],
  "FIN-t":  [{"name": "Topeno, Loppi, near Riihimäki", "code": "FIN-t", "lat": 60.76667, "lon": 24.28333 }],
  "FIN-v":  [{"name": "Virrat", "code": "FIN-v", "lat": 62.38333, "lon": 23.61667 }],
  "FIN-va": [{"name": "Vaasa", "code": "FIN-va", "lat": 63.08333, "lon": 21.60000 }],
  "FJI-n":  [{"name": "Nadi-Enamanu", "code": "FJI-n", "lat": -17.78722, "lon": 177.42222 }],
  "FRO-t":  [{"name": "Tórshavn", "code": "FRO-t", "lat": 62.01667, "lon": -6.78333 }],
  "FSM":    [{"name": "Pohnpei", "code": "FSM", "lat": 6.96667, "lon": 158.20000 }],
  "G-ab":   [{"name": "Aberdeen (Gregness)", "code": "G-ab", "lat": 57.12750, "lon": -2.05361 }],
  "G-aq":   [{"name": "Aberdeen (Blaikie's Quay)", "code": "G-aq", "lat": 57.14444, "lon": -2.08778 }],
  "G-an":   [{"name": "Anthorn", "code": "G-an", "lat": 54.91111, "lon": -3.28056 }],
  "G-ba":   [{"name": "Bangor (No.Ireland)", "code": "G-ba", "lat": 54.66417, "lon": -5.66889 }],
  "G-bd":   [{"name": "Bridlington (East Yorkshire)", "code": "G-bd", "lat": 54.09389, "lon": -0.17583 }],
  "G-cf":   [{"name": "Collafirth Hill (Shetland)", "code": "G-cf", "lat": 60.53333, "lon": -1.39167 }],
  "G-cm":   [{"name": "Crimond (Aberdeenshire)", "code": "G-cm", "lat": 57.61667, "lon": -1.88333 }],
  "G-cp":   [{"name": "London-Crystal Palace", "code": "G-cp", "lat": 51.41667, "lon": -0.08333 }],
  "G-cr":   [{"name": "London-Croydon", "code": "G-cr", "lat": 51.41667, "lon": -0.08333 }],
  "G-ct":   [{"name": "Croughton (Northants)", "code": "G-ct", "lat": 51.99722, "lon": -1.20917 }],
  "G-cu":   [{"name": "Cullercoats, Newcastle", "code": "G-cu", "lat": 55.07472, "lon": -1.46306 }],
  "G-d":    [{"name": "Droitwich", "code": "G-d", "lat": 52.30000, "lon": -2.10000 }],
  "G-dv":   [{"name": "Dover", "code": "G-dv", "lat": 51.13306, "lon": 1.34333 }],
  "G-ev":   [{"name": "St.Eval (Cornwall)", "code": "G-ev", "lat": 50.48333, "lon": -5.00000 }],
  "G-fh":   [{"name": "Fareham (Hampshire)", "code": "G-fh", "lat": 50.85833, "lon": -1.24972 }],
  "G-fl":   [{"name": "Falmouth Coastguard", "code": "G-fl", "lat": 50.14528, "lon": -5.04556 }],
  "G-fm":   [{"name": "Falmouth (Lizard)", "code": "G-fm", "lat": 49.96028, "lon": -5.20167 }],
  "G-hh":   [{"name": "Holyhead (Isle of Anglesey, Wales)", "code": "G-hh", "lat": 53.31639, "lon": -4.63250 }],
  "G-hu":   [{"name": "Humber (Flamborough)", "code": "G-hu", "lat": 54.11667, "lon": -0.07806 }],
  "G-ic":   [{"name": "Inskip (Lancashire)", "code": "G-ic", "lat": 53.833333, "lon": -2.833333 }],
  "G-ic":   [{"name": "St.Eval (Cornwall)", "code": "G-ic", "lat": 50.483333, "lon": -5.000000 }],
  "G-ic":   [{"name": "Crimond (Aberdeenshire)", "code": "G-ic", "lat": 57.61667, "lon": -1.88333 }],
  "G-in":   [{"name": "Inskip (Lancashire)", "code": "G-in", "lat": 53.83333, "lon": -2.83333 }],
  "G-lo":   [{"name": "London", "code": "G-lo", "lat": 51.50000, "lon": -0.18333 }],
  "G-lw":   [{"name": "Lerwick (Shetland)", "code": "G-lw", "lat": 60.14861, "lon": -1.14028 }],
  "G-mh":   [{"name": "Milford Haven, Wales", "code": "G-mh", "lat": 51.70778, "lon": -5.05250 }],
  "G-ni":   [{"name": "Niton Navtex, Isle of Wight", "code": "G-ni", "lat": 50.58639, "lon": -1.25472 }],
  "G-nw":   [{"name": "Northwood", "code": "G-nw", "lat": 51.50000, "lon": -0.16667 }],
  "G-o":    [{"name": "Orfordness", "code": "G-o", "lat": 52.10000, "lon": 1.58333 }],
  "G-p":    [{"name": "Portland", "code": "G-p", "lat": 50.60000, "lon": -2.45000 }],
  "G-pp":   [{"name": "Portpatrick Navtex (Dumfries and Galloway)", "code": "G-pp", "lat": 54.84417, "lon": -5.12444 }],
  "G-r":    [{"name": "Rampisham", "code": "G-r", "lat": 50.80833, "lon": -2.64306 }],
  "G-s":    [{"name": "Skelton", "code": "G-s", "lat": 54.73333, "lon": -2.90000 }],
  "G-sc":   [{"name": "Shetland (Lerwick)", "code": "G-sc", "lat": 60.40167, "lon": -1.22417 }],
  "G-sm":   [{"name": "St Mary's (Isles of Scilly)", "code": "G-sm", "lat": 49.92889, "lon": -6.30389 }],
  "G-sp":   [{"name": "Saint Peter Port (Guernsey)", "code": "G-sp", "lat": 49.45000, "lon": -2.53333 }],
  "G-st":   [{"name": "Stornoway (Butt of Lewis)", "code": "G-st", "lat": 58.46139, "lon": -6.23111 }],
  "G-sw":   [{"name": "Stornoway port", "code": "G-sw", "lat": 58.20333, "lon": -6.37556 }],
  "G-ti":   [{"name": "Tiree (Inner Hebrides)", "code": "G-ti", "lat": 56.50000, "lon": -6.80694 }],
  "G-w":    [{"name": "Woofferton", "code": "G-w", "lat": 52.31667, "lon": -2.71667 }],
  "G-wa":   [{"name": "Washford (Somerset)", "code": "G-wa", "lat": 51.16056, "lon": -3.34861 }],
  "GAB":    [{"name": "Moyabi", "code": "GAB", "lat": -1.66667, "lon": 13.51667 }],
  "GEO-s":  [{"name": "Sukhumi", "code": "GEO-s", "lat": 42.98833, "lon": 41.06611 }],
  "GNE-b":  [{"name": "Bata", "code": "GNE-b", "lat": 1.76667, "lon": 9.76667 }],
  "GNE-m":  [{"name": "Malabo", "code": "GNE-m", "lat": 3.75000, "lon": 8.78333 }],
  "GRC-a":  [{"name": "Avlis", "code": "GRC-a", "lat": 38.38333, "lon": 23.60000 }],
  "GRC-i":  [{"name": "Iraklion", "code": "GRC-i", "lat": 35.33333, "lon": 25.11667 }],
  "GRC-k":  [{"name": "Kerkyra", "code": "GRC-k", "lat": 39.61667, "lon": 19.91667 }],
  "GRC-L":  [{"name": "Limnos (Myrina)", "code": "GRC-L", "lat": 39.86667, "lon": 25.06667 }],
  "GRC-o":  [{"name": "Olimpia", "code": "GRC-o", "lat": 37.60750, "lon": 21.48750 }],
  "GRC-r":  [{"name": "Rhodos", "code": "GRC-r", "lat": 36.41667, "lon": 28.21667 }],
  "GRL-aa": [{"name": "Aasiaat", "code": "GRL-aa", "lat": 68.68333, "lon": -52.83333 }],
  "GRL-ik": [{"name": "Ikerasassuaq (Prins Christian Sund)", "code": "GRL-ik", "lat": 60.05000, "lon": -43.15000 }],
  "GRL-ko": [{"name": "Kook Island", "code": "GRL-ko", "lat": 64.06667, "lon": -52.00000 }],
  "GRL-ma": [{"name": "Maniitsoq", "code": "GRL-ma", "lat": 65.40000, "lon": -52.86667 }],
  "GRL-n":  [{"name": "Nuuk", "code": "GRL-n", "lat": 64.06667, "lon": -52.00000 }],
  "GRL-pa": [{"name": "Paamiut", "code": "GRL-pa", "lat": 62.00000, "lon": -49.71667 }],
  "GRL-qa": [{"name": "Qaqortoq", "code": "GRL-qa", "lat": 60.68333, "lon": -46.60000 }],
  "GRL-qe": [{"name": "Qeqertarsuaq", "code": "GRL-qe", "lat": 69.23333, "lon": -53.51667 }],
  "GRL-si": [{"name": "Sisimiut", "code": "GRL-si", "lat": 66.93333, "lon": -53.66667 }],
  "GRL-sq": [{"name": "Simiutaq", "code": "GRL-sq", "lat": 60.68333, "lon": -46.60000 }],
  "GRL-t":  [{"name": "Tasiilaq/Ammassalik", "code": "GRL-t", "lat": 65.60000, "lon": -37.63333 }],
  "GRL-up": [{"name": "Upernavik", "code": "GRL-up", "lat": 72.78333, "lon": -56.16667 }],
  "GRL-uu": [{"name": "Uummannaq", "code": "GRL-uu", "lat": 70.78333, "lon": -52.13333 }],
  "GUF":    [{"name": "Montsinery", "code": "GUF", "lat": 4.90000, "lon": -52.48333 }],
  "GUI-c":  [{"name": "Conakry-Sonfonia", "code": "GUI-c", "lat": 9.68611, "lon": -13.53639 }],
  "GUM-a":  [{"name": "Station KSDA, Agat,", "code": "GUM-a", "lat": 13.34111, "lon": 144.64889 }],
  "GUM-an": [{"name": "Andersen Air Force Base", "code": "GUM-an", "lat": 13.56667, "lon": 144.91667 }],
  "GUM-b":  [{"name": "Barrigada", "code": "GUM-b", "lat": 13.48333, "lon": 144.83333 }],
  "GUM-h":  [{"name": "Agana HFDL site", "code": "GUM-h", "lat": 13.46667, "lon": 144.80000 }],
  "GUM-m":  [{"name": "Station KTWR, Agana/Merizo", "code": "GUM-m", "lat": 13.27722, "lon": 144.67111 }],
  "GUM-n":  [{"name": "Naval station NPN", "code": "GUM-n", "lat": 13.43333, "lon": 144.65000 }],
  "GUY":    [{"name": "Sparendaam", "code": "GUY", "lat": 6.81667, "lon": -58.08333 }],
  "HKG-a":  [{"name": "Cape d'Aguilar", "code": "HKG-a", "lat": 22.21667, "lon": 114.25000 }],
  "HKG-m":  [{"name": "Marine Rescue Radio VRC", "code": "HKG-m", "lat": 22.29000, "lon": 114.15333 }],
  "HND-t":  [{"name": "Tegucigalpa", "code": "HND-t", "lat": 14.06667, "lon": -87.21667 }],
  "HNG-b":  [{"name": "Budapest", "code": "HNG-b", "lat": 47.50000, "lon": 19.05000 }],
  "HNG-lh": [{"name": "Lakihegy", "code": "HNG-lh", "lat": 47.36667, "lon": 19.00000 }],
  "HOL-a":  [{"name": "Alphen aan den Rijn", "code": "HOL-a", "lat": 52.13333, "lon": 4.63333 }],
  "HOL-b":  [{"name": "Borculo", "code": "HOL-b", "lat": 52.11667, "lon": 6.51667 }],
  "HOL-cg": [{"name": "Coast Guard Den Helder - Scheveningen", "code": "HOL-cg", "lat": 52.10000, "lon": 4.25000 }],
  "HOL-e":  [{"name": "Elburg", "code": "HOL-e", "lat": 52.43333, "lon": 5.86667 }],
  "HOL-he": [{"name": "Heerde", "code": "HOL-he", "lat": 52.38333, "lon": 6.03333 }],
  "HOL-k":  [{"name": "Klazienaveen", "code": "HOL-k", "lat": 52.73333, "lon": 6.98333 }],
  "HOL-m":  [{"name": "Margraten", "code": "HOL-m", "lat": 50.80000, "lon": 5.80000 }],
  "HOL-n":  [{"name": "Nijmegen", "code": "HOL-n", "lat": 51.85000, "lon": 5.83333 }],
  "HOL-o":  [{"name": "Ouddorp, Goeree-Overflakkee island", "code": "HOL-o", "lat": 51.80000, "lon": 3.90000 }],
  "HOL-ov": [{"name": "Overslag (Westdorpe)", "code": "HOL-ov", "lat": 51.20000, "lon": 3.86667 }],
  "HOL-w":  [{"name": "Winterswijk", "code": "HOL-w", "lat": 51.966667, "lon": 6.716667 }],
  "HOL-zw": [{"name": "Zwolle", "code": "HOL-zw", "lat": 52.51667, "lon": 6.08333 }],
  "HRV":    [{"name": "Deanovec", "code": "HRV", "lat": 45.65000, "lon": 16.45000 }],
  "HWA-a":  [{"name": "WWVH", "code": "HWA-a", "lat": 21.98917, "lon": -159.76444 }],
  "HWA-b":  [{"name": "WWVH", "code": "HWA-b", "lat": 21.98639, "lon": -159.76250 }],
  "HWA-c":  [{"name": "WWVH", "code": "HWA-c", "lat": 21.98833, "lon": -159.76417 }],
  "HWA-d":  [{"name": "WWVH", "code": "HWA-d", "lat": 21.98750, "lon": -159.76389 }],
  "HWA-hi": [{"name": "Hickam AFB", "code": "HWA-hi", "lat": 21.31667, "lon": -157.91667 }],
  "HWA-ho": [{"name": "Honolulu/Iroquois Point", "code": "HWA-ho", "lat": 21.32306, "lon": -157.99333 }],
  "HWA-m":  [{"name": "Moloka'i", "code": "HWA-m", "lat": 21.18333, "lon": -157.18333 }],
  "HWA-n":  [{"name": "Naalehu", "code": "HWA-n", "lat": 19.01667, "lon": -155.66667 }],
  "HWA-nm": [{"name": "NMO Honolulu/Maili", "code": "HWA-nm", "lat": 21.42806, "lon": -158.15306 }],
  "HWA-p":  [{"name": "Pearl Harbour", "code": "HWA-p", "lat": 21.42806, "lon": -158.15306 }],
  "I-a":    [{"name": "Andrate", "code": "I-a", "lat": 45.51667, "lon": 7.88333 }],
  "I-an":   [{"name": "Ancona IPA", "code": "I-an", "lat": 43.60306, "lon": 13.47056 }],
  "I-au":   [{"name": "Augusta IQA (Sicily)", "code": "I-au", "lat": 37.23722, "lon": 15.24056 }],
  "I-b":    [{"name": "San Benedetto de Tronto IQP", "code": "I-b", "lat": 42.97083, "lon": 13.86528 }],
  "I-ba":   [{"name": "Bari IPB", "code": "I-ba", "lat": 41.08917, "lon": 16.99556 }],
  "I-cg":   [{"name": "Cagliari IDC (Sardinia)", "code": "I-cg", "lat": 39.22778, "lon": 9.23444 }],
  "I-cv":   [{"name": "Civitavecchia IPD", "code": "I-cv", "lat": 42.03333, "lon": 11.83333 }],
  "I-ge":   [{"name": "Genova ICB", "code": "I-ge", "lat": 44.42917, "lon": 8.93306 }],
  "I-kr":   [{"name": "Crotone IPC", "code": "I-kr", "lat": 39.05000, "lon": 17.13333 }],
  "I-li":   [{"name": "Livorno-Montenero IPL", "code": "I-li", "lat": 43.49028, "lon": 10.36083 }],
  "I-lm":   [{"name": "Lampedusa-Ponente IQN", "code": "I-lm", "lat": 35.51750, "lon": 12.56611 }],
  "I-me":   [{"name": "Messina IDF (Sicily)", "code": "I-me", "lat": 38.26833, "lon": 15.62194 }],
  "I-mp":   [{"name": "Monteparano (IPC)", "code": "I-mp", "lat": 40.44194, "lon": 17.41889 }],
  "I-mz":   [{"name": "Mazara del Vallo IQQ", "code": "I-mz", "lat": 37.67000, "lon": 12.61306 }],
  "I-na":   [{"name": "Napoli-Posillipo IQH", "code": "I-na", "lat": 40.80056, "lon": 14.18333 }],
  "I-p":    [{"name": "Padova", "code": "I-p", "lat": 45.15000, "lon": 11.70000 }],
  "I-pa":   [{"name": "Palermo-Punta Raisi IPP (Sicily)", "code": "I-pa", "lat": 38.19000, "lon": 13.10833 }],
  "I-pt":   [{"name": "Porto Torres IZN (Sardinia)", "code": "I-pt", "lat": 40.79778, "lon": 8.32528 }],
  "I-r":    [{"name": "Roma", "code": "I-r", "lat": 41.80000, "lon": 12.51667 }],
  "I-ra":   [{"name": "Roma IMB", "code": "I-ra", "lat": 41.78333, "lon": 12.46667 }],
  "I-re":   [{"name": "Rome", "code": "I-re", "lat": 41.91667, "lon": 12.48333 }],
  "I-si":   [{"name": "Sigonella (Sicilia)", "code": "I-si", "lat": 37.40000, "lon": 14.91667 }],
  "I-sr":   [{"name": "Santa Rosa (Maritele), Roma", "code": "I-sr", "lat": 41.98333, "lon": 12.36667 }],
  "I-sy":   [{"name": "NSY", "code": "I-sy", "lat": 37.11667, "lon": 14.43333 }],
  "I-t":    [{"name": "Trieste (Monte Radio) IQX", "code": "I-t", "lat": 45.67667, "lon": 13.76917 }],
  "I-v":    [{"name": "Viareggio, Toscana", "code": "I-v", "lat": 43.90000, "lon": 10.28333 }],
  "IND-a":  [{"name": "Aligarh (4x250kW)", "code": "IND-a", "lat": 28.00000, "lon": 78.10000 }],
  "IND-ah": [{"name": "Ahmedabad", "code": "IND-ah", "lat": 22.86667, "lon": 72.61667 }],
  "IND-az": [{"name": "Aizawl(10kW)", "code": "IND-az", "lat": 23.71667, "lon": 92.71667 }],
  "IND-b":  [{"name": "Bengaluru-Doddaballapur (Bangalore)", "code": "IND-b", "lat": 13.23333, "lon": 77.50000 }],
  "IND-bh": [{"name": "Bhopal(50kW)", "code": "IND-bh", "lat": 23.16667, "lon": 77.50000 }],
  "IND-c":  [{"name": "Chennai (Madras)", "code": "IND-c", "lat": 13.13333, "lon": 80.11667 }],
  "IND-d":  [{"name": "Delhi (Kingsway)", "code": "IND-d", "lat": 28.71667, "lon": 77.20000 }],
  "IND-dn": [{"name": "Delhi-Nangli Poona", "code": "IND-dn", "lat": 28.76667, "lon": 77.13333 }],
  "IND-g":  [{"name": "Gorakhpur", "code": "IND-g", "lat": 26.86667, "lon": 83.46667 }],
  "IND-gt": [{"name": "Gangtok", "code": "IND-gt", "lat": 27.36667, "lon": 88.61667 }],
  "IND-hy": [{"name": "Hyderabad", "code": "IND-hy", "lat": 17.33333, "lon": 78.55000 }],
  "IND-im": [{"name": "Imphal", "code": "IND-im", "lat": 24.61667, "lon": 93.90000 }],
  "IND-it": [{"name": "Itanagar", "code": "IND-it", "lat": 27.06667, "lon": 93.60000 }],
  "IND-j":  [{"name": "Jalandhar", "code": "IND-j", "lat": 31.31667, "lon": 75.30000 }],
  "IND-ja": [{"name": "Jaipur", "code": "IND-ja", "lat": 26.90000, "lon": 75.75000 }],
  "IND-je": [{"name": "Jeypore", "code": "IND-je", "lat": 18.91667, "lon": 82.56667 }],
  "IND-jm": [{"name": "Jammu", "code": "IND-jm", "lat": 32.75000, "lon": 75.00000 }],
  "IND-k":  [{"name": "Kham Pur, Delhi 110036 (Khampur)", "code": "IND-k", "lat": 28.81667, "lon": 77.13333 }],
  "IND-kc": [{"name": "Kolkata-Chandi", "code": "IND-kc", "lat": 22.36667, "lon": 88.28333 }],
  "IND-kh": [{"name": "Kohima", "code": "IND-kh", "lat": 25.65000, "lon": 94.10000 }],
  "IND-ko": [{"name": "Kolkata(Calcutta)-Chinsurah", "code": "IND-ko", "lat": 23.01667, "lon": 88.35000 }],
  "IND-ku": [{"name": "Kurseong", "code": "IND-ku", "lat": 26.91667, "lon": 88.31667 }],
  "IND-kv": [{"name": "Kolkata Volmet", "code": "IND-kv", "lat": 22.65000, "lon": 88.45000 }],
  "IND-le": [{"name": "Leh", "code": "IND-le", "lat": 34.13333, "lon": 77.48333 }],
  "IND-lu": [{"name": "Lucknow", "code": "IND-lu", "lat": 26.88333, "lon": 81.05000 }],
  "IND-m":  [{"name": "Mumbai (Bombay)", "code": "IND-m", "lat": 19.18333, "lon": 72.81667 }],
  "IND-mv": [{"name": "Mumbai Volmet", "code": "IND-mv", "lat": 19.08333, "lon": 72.85000 }],
  "IND-n":  [{"name": "Nagpur, Maharashtra", "code": "IND-n", "lat": 20.90000, "lon": 78.98333 }],
  "IND-nj": [{"name": "Najibabad, Uttar Pradesh", "code": "IND-nj", "lat": 29.63333, "lon": 78.38333 }],
  "IND-p":  [{"name": "Panaji (Goa)", "code": "IND-p", "lat": 15.51667, "lon": 73.86667 }],
  "IND-pb": [{"name": "Port Blair-Brookshabad", "code": "IND-pb", "lat": 11.61667, "lon": 92.75000 }],
  "IND-r":  [{"name": "Rajkot", "code": "IND-r", "lat": 22.36667, "lon": 70.68333 }],
  "IND-ra": [{"name": "Ranchi", "code": "IND-ra", "lat": 23.40000, "lon": 85.36667 }],
  "IND-sg": [{"name": "Shillong", "code": "IND-sg", "lat": 25.43333, "lon": 91.81667 }],
  "IND-si": [{"name": "Siliguri", "code": "IND-si", "lat": 26.76667, "lon": 88.43333 }],
  "IND-sm": [{"name": "Shimla", "code": "IND-sm", "lat": 31.10000, "lon": 77.15000 }],
  "IND-sr": [{"name": "Srinagar", "code": "IND-sr", "lat": 34.00000, "lon": 74.83333 }],
  "IND-su": [{"name": "Suratgarh (Rajasthan)", "code": "IND-su", "lat": 29.30000, "lon": 73.91667 }],
  "IND-t":  [{"name": "Tuticorin (Tamil Nadu)", "code": "IND-t", "lat": 8.81667, "lon": 78.08333 }],
  "IND-tv": [{"name": "Thiruvananthapuram(Trivendrum)", "code": "IND-tv", "lat": 8.48333, "lon": 76.98333 }],
  "IND-v":  [{"name": "Vijayanarayanam (Tamil Nadu)", "code": "IND-v", "lat": 8.38333, "lon": 77.75000 }],
  "IND-vs": [{"name": "Vishakapatnam (Andhra Pradesh)", "code": "IND-vs", "lat": 17.71667, "lon": 83.30000 }],
  "IND-w":  [{"name": "Guwahati (1x200kW, 1x50kW)", "code": "IND-w", "lat": 26.18333, "lon": 91.83333 }],
  "INS-am": [{"name": "Ambon, Ambon Island, Maluku", "code": "INS-am", "lat": -3.69694, "lon": 128.17472 }],
  "INS-ap": [{"name": "Amamapare, Papua", "code": "INS-ap", "lat": -4.88333, "lon": 136.80000 }],
  "INS-at": [{"name": "Atapupu, Timor", "code": "INS-at", "lat": -9.02500, "lon": 124.86111 }],
  "INS-ba": [{"name": "Banggai, Banggai Island, Sulawesi Tengah", "code": "INS-ba", "lat": -1.59028, "lon": 123.49889 }],
  "INS-bb": [{"name": "Banabungi, Buton Island, Sulawesi Tenggara", "code": "INS-bb", "lat": -5.51389, "lon": 122.84444 }],
  "INS-bd": [{"name": "Badas, Sumbawa Island, West Nusa Tenggara", "code": "INS-bd", "lat": -8.46222, "lon": 117.37722 }],
  "INS-be": [{"name": "Bade, Papua", "code": "INS-be", "lat": -7.16444, "lon": 139.59694 }],
  "INS-bg": [{"name": "Bagan Siapi-Api, Riau, Sumatra", "code": "INS-bg", "lat": 2.15250, "lon": 100.80278 }],
  "INS-bi": [{"name": "Biak, Papua", "code": "INS-bi", "lat": -1.00000, "lon": 135.50000 }],
  "INS-bj": [{"name": "Banjarmasin, Kalimantan Selatan", "code": "INS-bj", "lat": -3.33333, "lon": 114.58333 }],
  "INS-bk": [{"name": "Bengkalis, Bengkalis Island, Riau", "code": "INS-bk", "lat": 1.45139, "lon": 102.10944 }],
  "INS-bl": [{"name": "Batu Licin, Kalimantan Selatan", "code": "INS-bl", "lat": -3.43194, "lon": 116.00194 }],
  "INS-bm": [{"name": "Batu Ampar, Batam Island next to Singapore", "code": "INS-bm", "lat": 1.18083, "lon": 104.01444 }],
  "INS-bn": [{"name": "Bawean, Bawean Island, Jawa Timur", "code": "INS-bn", "lat": -5.85556, "lon": 112.65556 }],
  "INS-bo": [{"name": "Benoa, Denpasar, Bali", "code": "INS-bo", "lat": -8.75611, "lon": 115.21667 }],
  "INS-bp": [{"name": "Balikpapan, Kalimantan Timur", "code": "INS-bp", "lat": -1.26222, "lon": 116.82028 }],
  "INS-bt": [{"name": "Benete, Sumbawa Island, West Nusa Tenggara", "code": "INS-bt", "lat": -8.90083, "lon": 116.74722 }],
  "INS-bu": [{"name": "Bukittinggi, Sumatera Barat", "code": "INS-bu", "lat": -0.30000, "lon": 100.36667 }],
  "INS-bw": [{"name": "Belawan, Medan, Sumatera Utara", "code": "INS-bw", "lat": 3.72139, "lon": 98.66889 }],
  "INS-by": [{"name": "Biak, Biak Island, Papua", "code": "INS-by", "lat": -1.18361, "lon": 136.07806 }],
  "INS-b2": [{"name": "Bau-Bau, Buton Island, Sulawesi Tenggara", "code": "INS-b2", "lat": -5.48028, "lon": 122.58111 }],
  "INS-b3": [{"name": "Bengkulu, Sumatra", "code": "INS-b3", "lat": -3.89972, "lon": 102.30889 }],
  "INS-b4": [{"name": "Bima, Sumbawa Island, West Nusa Tenggara", "code": "INS-b4", "lat": -8.44056, "lon": 118.72556 }],
  "INS-b5": [{"name": "Bintuni, Papua Barat", "code": "INS-b5", "lat": -2.11972, "lon": 133.50111 }],
  "INS-b6": [{"name": "Bitung, Sulawesi Utara", "code": "INS-b6", "lat": 1.46472, "lon": 125.18417 }],
  "INS-b7": [{"name": "Bontang, Kalimantan Timur", "code": "INS-b7", "lat": -0.13333, "lon": 117.50000 }],
  "INS-cb": [{"name": "Celukan Bawang, Bali", "code": "INS-cb", "lat": -8.18611, "lon": 114.83111 }],
  "INS-cc": [{"name": "Cilacap, Java", "code": "INS-cc", "lat": -7.73333, "lon": 109.00000 }],
  "INS-ci": [{"name": "Cigading, Merak, Banten, Java", "code": "INS-ci", "lat": -5.93333, "lon": 106.00000 }],
  "INS-cr": [{"name": "Cirebon, Jawa Barat", "code": "INS-cr", "lat": -6.71667, "lon": 108.56667 }],
  "INS-dg": [{"name": "Donggala, Sulawesi Tengah", "code": "INS-dg", "lat": -0.67500, "lon": 119.74472 }],
  "INS-dm": [{"name": "Dumai, Riau, Sumatra", "code": "INS-dm", "lat": 1.68611, "lon": 101.45556 }],
  "INS-do": [{"name": "Dobo, Wamar Island, Maluku", "code": "INS-do", "lat": -5.75000, "lon": 134.23333 }],
  "INS-ds": [{"name": "Dabo Singkep, Singkep Island, Riau, Sumatra", "code": "INS-ds", "lat": -0.50000, "lon": 104.56667 }],
  "INS-en": [{"name": "Ende, Flores Island, Nusa Tenggara Timur", "code": "INS-en", "lat": -8.83889, "lon": 121.64389 }],
  "INS-f":  [{"name": "Fakfak, Papua Barat", "code": "INS-f", "lat": -2.93333, "lon": 132.30000 }],
  "INS-fj": [{"name": "Fatujuring, Wokam Island, Maluku", "code": "INS-fj", "lat": -6.01667, "lon": 134.15000 }],
  "INS-g":  [{"name": "Gorontalo, Sulawesi", "code": "INS-g", "lat": 0.56667, "lon": 123.06667 }],
  "INS-gi": [{"name": "Gilimanuk, Bali", "code": "INS-gi", "lat": -8.17806, "lon": 114.43472 }],
  "INS-go": [{"name": "Gorontalo port, Sulawesi", "code": "INS-go", "lat": 0.50806, "lon": 123.06361 }],
  "INS-gr": [{"name": "Gresik, Surabaya, Jawa Timur", "code": "INS-gr", "lat": -7.16417, "lon": 112.66028 }],
  "INS-gs": [{"name": "Gunung Sitoli, Nias Island, Sumatera Utara", "code": "INS-gs", "lat": 1.30667, "lon": 97.60972 }],
  "INS-j":  [{"name": "Jakarta (Cimanggis)", "code": "INS-j", "lat": -6.20000, "lon": 106.85000 }],
  "INS-ja": [{"name": "Jambi PKC3", "code": "INS-ja", "lat": -1.61389, "lon": 103.61417 }],
  "INS-jb": [{"name": "Jakarta BMG", "code": "INS-jb", "lat": -6.28333, "lon": 106.86667 }],
  "INS-jm": [{"name": "Jambi, Sumatera", "code": "INS-jm", "lat": -1.63333, "lon": 103.56667 }],
  "INS-jp": [{"name": "Jepara, Jawa Tengah", "code": "INS-jp", "lat": -6.58639, "lon": 110.66139 }],
  "INS-js": [{"name": "Jakarta, Sunda Kelapa port", "code": "INS-js", "lat": -6.12333, "lon": 106.80833 }],
  "INS-jw": [{"name": "Juwana, Jawa Tengah", "code": "INS-jw", "lat": -6.70417, "lon": 111.15361 }],
  "INS-jx": [{"name": "Jakarta PKX", "code": "INS-jx", "lat": -6.11889, "lon": 106.85417 }],
  "INS-jy": [{"name": "Jayapura, Papua", "code": "INS-jy", "lat": -2.51944, "lon": 140.72278 }],
  "INS-ka": [{"name": "Kaimana, Papua", "code": "INS-ka", "lat": -3.66667, "lon": 133.76667 }],
  "INS-kb": [{"name": "Kalabahi, Alor Island, East Nusa Tenggara", "code": "INS-kb", "lat": -8.22167, "lon": 124.51111 }],
  "INS-kd": [{"name": "Kendari, Sulawesi Tenggara", "code": "INS-kd", "lat": -3.98333, "lon": 122.60000 }],
  "INS-kg": [{"name": "Kalianget, Sumenep, Madura Island, Jawa Timur", "code": "INS-kg", "lat": -7.06667, "lon": 113.96667 }],
  "INS-ki": [{"name": "Kumai, Kalimantan Tengah", "code": "INS-ki", "lat": -2.75556, "lon": 111.71667 }],
  "INS-kj": [{"name": "Kijang, Bintan Island", "code": "INS-kj", "lat": 0.85111, "lon": 104.60861 }],
  "INS-kl": [{"name": "Kolonodale, Sulawesi Tenggara", "code": "INS-kl", "lat": -2.02111, "lon": 121.34111 }],
  "INS-km": [{"name": "Karimunjawa Island, off Java", "code": "INS-km", "lat": -5.88333, "lon": 110.43333 }],
  "INS-kn": [{"name": "Kupang, Timor", "code": "INS-kn", "lat": -10.20000, "lon": 123.61667 }],
  "INS-ko": [{"name": "Kolaka, Sulawesi Tenggara", "code": "INS-ko", "lat": -4.04861, "lon": 121.57833 }],
  "INS-kp": [{"name": "Ketapang, Kalimantan Barat", "code": "INS-kp", "lat": -1.81667, "lon": 109.96667 }],
  "INS-ks": [{"name": "Kota Langsa, Aceh, Sumatra", "code": "INS-ks", "lat": 4.48333, "lon": 97.95000 }],
  "INS-kt": [{"name": "Kuala Tungkal, Jambi, Sumatra", "code": "INS-kt", "lat": -0.82083, "lon": 103.46833 }],
  "INS-ku": [{"name": "Kota Baru, Laut Island, Kalimantan Selatan", "code": "INS-ku", "lat": -3.23333, "lon": 116.23333 }],
  "INS-kw": [{"name": "Kwandang, Gorontalo, Sulawesi", "code": "INS-kw", "lat": 0.85778, "lon": 122.79222 }],
  "INS-le": [{"name": "Lembar, Lombok", "code": "INS-le", "lat": -8.72806, "lon": 116.07306 }],
  "INS-lh": [{"name": "Lhokseumawe, Aceh, Sumatra", "code": "INS-lh", "lat": 5.21139, "lon": 97.03917 }],
  "INS-lo": [{"name": "Lombok", "code": "INS-lo", "lat": -8.50167, "lon": 116.67833 }],
  "INS-lu": [{"name": "Luwuky, Sulawesi Tengah", "code": "INS-lu", "lat": -0.89972, "lon": 122.79417 }],
  "INS-ma": [{"name": "Manokwari, Papua Barat", "code": "INS-ma", "lat": -0.86556, "lon": 134.07694 }],
  "INS-mb": [{"name": "Masalembo Island, Java Sea", "code": "INS-mb", "lat": -5.58333, "lon": 114.43333 }],
  "INS-md": [{"name": "Manado, Sulawesi Utara", "code": "INS-md", "lat": 1.20000, "lon": 124.90000 }],
  "INS-me": [{"name": "Meneng, Banyuwangi, Java", "code": "INS-me", "lat": -8.12500, "lon": 114.39722 }],
  "INS-mk": [{"name": "Manokwari, Irian Jaya Barat", "code": "INS-mk", "lat": -0.80000, "lon": 134.00000 }],
  "INS-mm": [{"name": "Maumere, Flores, Nusa Tenggara Timur", "code": "INS-mm", "lat": -8.61667, "lon": 122.21667 }],
  "INS-mn": [{"name": "Manado, Sulawesi Utara", "code": "INS-mn", "lat": 1.54028, "lon": 124.83444 }],
  "INS-mr": [{"name": "Merauke, Papua", "code": "INS-mr", "lat": -8.47972, "lon": 140.39389 }],
  "INS-ms": [{"name": "Makassar, Sulawesi Selatan", "code": "INS-ms", "lat": -5.10556, "lon": 119.44194 }],
  "INS-mu": [{"name": "Muntok, Bangka Island", "code": "INS-mu", "lat": -2.05611, "lon": 105.15111 }],
  "INS-n":  [{"name": "Nabire, Papua", "code": "INS-n", "lat": -3.23333, "lon": 135.58333 }],
  "INS-na": [{"name": "Natuna, Tiga Island, Riau Islands", "code": "INS-na", "lat": 3.66944, "lon": 108.12917 }],
  "INS-nu": [{"name": "Nunukan, Nunukan Island, Kalimantan Utara", "code": "INS-nu", "lat": 4.12167, "lon": 117.68889 }],
  "INS-p":  [{"name": "Palangkaraya, Kalimantan Tengah", "code": "INS-p", "lat": -0.18333, "lon": 113.90000 }],
  "INS-pa": [{"name": "Palu, Sulawesi Tengah", "code": "INS-pa", "lat": -0.60000, "lon": 129.60000 }],
  "INS-pb": [{"name": "Padang Bai, Bali", "code": "INS-pb", "lat": -8.52694, "lon": 115.50778 }],
  "INS-pd": [{"name": "Padang, Sumatera Barat", "code": "INS-pd", "lat": -0.10000, "lon": 100.35000 }],
  "INS-pe": [{"name": "Pekalongan, Java", "code": "INS-pe", "lat": -6.85972, "lon": 109.69167 }],
  "INS-pf": [{"name": "Pare-Pare, Sulawesi Selatan", "code": "INS-pf", "lat": -4.01667, "lon": 119.61667 }],
  "INS-pg": [{"name": "Pangkal Baru, Bangkal Island", "code": "INS-pg", "lat": -2.16667, "lon": 106.13333 }],
  "INS-ph": [{"name": "Panipahan, Riau, Sumatra", "code": "INS-ph", "lat": 2.41667, "lon": 100.33333 }],
  "INS-pi": [{"name": "Parigi, Sulawesi Tengah", "code": "INS-pi", "lat": -0.82750, "lon": 120.17750 }],
  "INS-pj": [{"name": "Panjang, Lampung, Sumatra", "code": "INS-pj", "lat": -5.47306, "lon": 105.31750 }],
  "INS-pk": [{"name": "Pontianak", "code": "INS-pk", "lat": -0.02667, "lon": 109.28833 }],
  "INS-pl": [{"name": "Plaju, Palembang, Sumatera Selatan", "code": "INS-pl", "lat": -3.00000, "lon": 104.83333 }],
  "INS-pm": [{"name": "Palembang, Sumatera Selatan", "code": "INS-pm", "lat": -2.96667, "lon": 104.78333 }],
  "INS-po": [{"name": "Pomalaa, Sulawesi Tenggara", "code": "INS-po", "lat": -4.18306, "lon": 121.64889 }],
  "INS-pp": [{"name": "Palopo, Sulawesi Selatan", "code": "INS-pp", "lat": -2.98889, "lon": 120.20278 }],
  "INS-pq": [{"name": "Probolinggo, Jawa Timur", "code": "INS-pq", "lat": -7.73333, "lon": 113.21667 }],
  "INS-pr": [{"name": "Panarukan, Jawa Timur", "code": "INS-pr", "lat": -7.70000, "lon": 113.93333 }],
  "INS-ps": [{"name": "Poso, Sulawesi Tengah", "code": "INS-ps", "lat": -1.38333, "lon": 120.75000 }],
  "INS-pt": [{"name": "Pantoloan, Sulawesi Tengah", "code": "INS-pt", "lat": -0.71667, "lon": 119.86667 }],
  "INS-pu": [{"name": "Pekanbaru, Riau, Sumatra", "code": "INS-pu", "lat": 0.31667, "lon": 103.16667 }],
  "INS-pv": [{"name": "Pulau Sambu, Riau Islands", "code": "INS-pv", "lat": 1.15833, "lon": 103.90000 }],
  "INS-ra": [{"name": "Raha, Muna Island, Sulawesi Tenggara", "code": "INS-ra", "lat": -4.83333, "lon": 122.71667 }],
  "INS-re": [{"name": "Rengat, Riau, Sumatra", "code": "INS-re", "lat": -0.46667, "lon": 102.68333 }],
  "INS-ro": [{"name": "Reo, Flores", "code": "INS-ro", "lat": -8.28611, "lon": 120.45222 }],
  "INS-sa": [{"name": "Sabang, We Island, Aceh", "code": "INS-sa", "lat": 5.90000, "lon": 95.35000 }],
  "INS-sb": [{"name": "Seba, Sawu Island", "code": "INS-sb", "lat": -10.50000, "lon": 121.83333 }],
  "INS-se": [{"name": "Serui, Japen Island, Papua", "code": "INS-se", "lat": -1.88333, "lon": 136.23333 }],
  "INS-sg": [{"name": "Semarang, Java", "code": "INS-sg", "lat": -6.97639, "lon": 110.34722 }],
  "INS-sh": [{"name": "Susoh, Aceh, Sumatra", "code": "INS-sh", "lat": 3.71917, "lon": 96.80944 }],
  "INS-si": [{"name": "Sarmi, Papua", "code": "INS-si", "lat": -1.85000, "lon": 138.75000 }],
  "INS-sj": [{"name": "Selat Panjang, Tebingtinggi Island, Riau", "code": "INS-sj", "lat": 1.02083, "lon": 102.71944 }],
  "INS-sk": [{"name": "Singkil, Aceh, Sumatra", "code": "INS-sk", "lat": 2.30000, "lon": 97.75000 }],
  "INS-sn": [{"name": "Sanana, Sulabes Island, Maluku", "code": "INS-sn", "lat": -2.05000, "lon": 125.96667 }],
  "INS-so": [{"name": "Sorong, Papua Barat", "code": "INS-so", "lat": -0.88333, "lon": 131.26667 }],
  "INS-sp": [{"name": "Sipange, Tapanuli, Sumatera Utara", "code": "INS-sp", "lat": 1.20556, "lon": 99.37917 }],
  "INS-sq": [{"name": "Sibolga, Sumatera Utara", "code": "INS-sq", "lat": 1.73333, "lon": 98.78333 }],
  "INS-sr": [{"name": "Samarinda, Kalimantan Timur", "code": "INS-sr", "lat": -0.50833, "lon": 117.15417 }],
  "INS-st": [{"name": "Sampit, Kalimantan Tengah", "code": "INS-st", "lat": -2.55722, "lon": 112.95667 }],
  "INS-su": [{"name": "Siau Island", "code": "INS-su", "lat": 2.73333, "lon": 125.40000 }],
  "INS-sy": [{"name": "Selayar, Sulawesi Selatan", "code": "INS-sy", "lat": -6.11944, "lon": 120.45833 }],
  "INS-s2": [{"name": "Sinabang, Simeulue Island, Aceh", "code": "INS-s2", "lat": 2.46667, "lon": 96.38333 }],
  "INS-s3": [{"name": "Sipura Island, Sumatera Barat", "code": "INS-s3", "lat": -2.20000, "lon": 99.66667 }],
  "INS-s4": [{"name": "Surabaya, Jawa Timur", "code": "INS-s4", "lat": -7.20000, "lon": 112.73333 }],
  "INS-s8": [{"name": "Sorong PKY8, Papua Barat", "code": "INS-s8", "lat": -0.65000, "lon": 130.71667 }],
  "INS-s9": [{"name": "Sorong PKY9, Papua Barat", "code": "INS-s9", "lat": -1.13333, "lon": 131.26667 }],
  "INS-t":  [{"name": "Ternate, Ternate Island, Maluku Utara", "code": "INS-t", "lat": 0.78333, "lon": 127.36667 }],
  "INS-ta": [{"name": "Tahuna, Sulawesi Utara", "code": "INS-ta", "lat": 3.60556, "lon": 125.50417 }],
  "INS-tb": [{"name": "Tanjung Balai Karimun, Karimunbesar Island, Riau Islands", "code": "INS-tb", "lat": 0.98806, "lon": 103.43722 }],
  "INS-td": [{"name": "Teluk Dalam, Dima Island, Sumatera Utara", "code": "INS-td", "lat": 0.56667, "lon": 97.81667 }],
  "INS-te": [{"name": "Tegal, Java", "code": "INS-te", "lat": -6.85000, "lon": 109.13333 }],
  "INS-tg": [{"name": "Tanjung Selor, Kalimantan Utara", "code": "INS-tg", "lat": 2.80000, "lon": 117.36667 }],
  "INS-tk": [{"name": "Tarakan, Tarakan Island, Kalimantan Utara", "code": "INS-tk", "lat": 3.28889, "lon": 117.59028 }],
  "INS-tl": [{"name": "Tembilahan, Riau, Sumatra", "code": "INS-tl", "lat": -0.31694, "lon": 103.16139 }],
  "INS-tm": [{"name": "Tarempa, Siantan Island, Riau Islands", "code": "INS-tm", "lat": 3.21667, "lon": 106.21667 }],
  "INS-to": [{"name": "Tobelo, Halmahera Island, Maluku Utara", "code": "INS-to", "lat": 1.72500, "lon": 128.00861 }],
  "INS-ts": [{"name": "Tanjung Santan, Kalimantan Timur", "code": "INS-ts", "lat": -0.10222, "lon": 117.46417 }],
  "INS-tt": [{"name": "Toli-Toli, Sulawesi Tengah,", "code": "INS-tt", "lat": 1.05500, "lon": 120.80556 }],
  "INS-tu": [{"name": "Tanjung Uban, Bintan Island, Riau Islands", "code": "INS-tu", "lat": 1.06583, "lon": 104.22417 }],
  "INS-tw": [{"name": "Tual, Dullah Island, Maluku", "code": "INS-tw", "lat": -5.63333, "lon": 132.75000 }],
  "INS-ty": [{"name": "Taluk Bayur, Sumatera Barat", "code": "INS-ty", "lat": -1.04139, "lon": 100.38056 }],
  "INS-ul": [{"name": "Ulee-Lheue, Banda Aceh, Aceh, Sumatra", "code": "INS-ul", "lat": 5.56667, "lon": 95.28333 }],
  "INS-w":  [{"name": "Wamena, Papua", "code": "INS-w", "lat": -4.10000, "lon": 138.93333 }],
  "INS-wa": [{"name": "Waingapu, Sumba Island, East Nusa Tenggara", "code": "INS-wa", "lat": -9.66167, "lon": 120.25611 }],
  "IRL-mh": [{"name": "Malin Head, Co. Donegal", "code": "IRL-mh", "lat": 55.37194, "lon": -7.33917 }],
  "IRL-s":  [{"name": "Shannon", "code": "IRL-s", "lat": 52.74444, "lon": -8.92694 }],
  "IRL-sk": [{"name": "Sheskin, Co. Donegal", "code": "IRL-sk", "lat": 55.35222, "lon": -7.27389 }],
  "IRL-tr": [{"name": "Tralee, Co. Kerry", "code": "IRL-tr", "lat": 52.26667, "lon": -9.70000 }],
  "IRL-v":  [{"name": "Valentia, Co. Kerry", "code": "IRL-v", "lat": 51.86778, "lon": -10.33417 }],
  "IRN-a":  [{"name": "Ahwaz", "code": "IRN-a", "lat": 31.33333, "lon": 48.66667 }],
  "IRN-b":  [{"name": "Bandar-e Torkeman", "code": "IRN-b", "lat": 36.90000, "lon": 54.06667 }],
  "IRN-ba": [{"name": "Bandar Abbas", "code": "IRN-ba", "lat": 27.10167, "lon": 56.06333 }],
  "IRN-bb": [{"name": "Bonab", "code": "IRN-bb", "lat": 37.30000, "lon": 46.05000 }],
  "IRN-c":  [{"name": "Chah Bahar", "code": "IRN-c", "lat": 25.48333, "lon": 60.53333 }],
  "IRN-j":  [{"name": "Jolfa", "code": "IRN-j", "lat": 38.93333, "lon": 45.60000 }],
  "IRN-k":  [{"name": "Kamalabad", "code": "IRN-k", "lat": 35.76667, "lon": 51.45000 }],
  "IRN-ke": [{"name": "Kish Island", "code": "IRN-ke", "lat": 26.56667, "lon": 53.93333 }],
  "IRN-ki": [{"name": "Kiashar", "code": "IRN-ki", "lat": 37.40000, "lon": 50.01667 }],
  "IRN-m":  [{"name": "Mashhad", "code": "IRN-m", "lat": 36.25000, "lon": 59.55000 }],
  "IRN-mh": [{"name": "Bandar-e Mahshahr", "code": "IRN-mh", "lat": 30.61667, "lon": 49.20000 }],
  "IRN-q":  [{"name": "Qasr Shirin", "code": "IRN-q", "lat": 34.45000, "lon": 45.61667 }],
  "IRN-s":  [{"name": "Sirjan", "code": "IRN-s", "lat": 29.45000, "lon": 55.68333 }],
  "IRN-t":  [{"name": "Tayebad", "code": "IRN-t", "lat": 34.73333, "lon": 60.80000 }],
  "IRN-te": [{"name": "Tehran", "code": "IRN-te", "lat": 35.75000, "lon": 51.41667 }],
  "IRN-z":  [{"name": "Zahedan", "code": "IRN-z", "lat": 29.46667, "lon": 60.88333 }],
  "IRN-zb": [{"name": "Zabol", "code": "IRN-zb", "lat": 31.03333, "lon": 61.55000 }],
  "IRQ-d":  [{"name": "Salah al-Din (Saladin)", "code": "IRQ-d", "lat": 34.45000, "lon": 43.58333 }],
  "IRQ-s":  [{"name": "Sulaimaniya", "code": "IRQ-s", "lat": 35.55000, "lon": 45.43333 }],
  "ISL-f":  [{"name": "Fjallabyggd", "code": "ISL-f", "lat": 66.15000, "lon": -18.91667 }],
  "ISL-g":  [{"name": "Keflavik/Grindavik", "code": "ISL-g", "lat": 64.01667, "lon": -22.56667 }],
  "ISL-gt": [{"name": "Grindavik Thorbjöen", "code": "ISL-gt", "lat": 63.85222, "lon": -22.43333 }],
  "ISL-hf": [{"name": "Hornafjördur", "code": "ISL-hf", "lat": 64.25000, "lon": -15.21667 }],
  "ISL-if": [{"name": "Isafjördur", "code": "ISL-if", "lat": 66.08333, "lon": -23.03333 }],
  "ISL-n":  [{"name": "Neskaupstadur", "code": "ISL-n", "lat": 65.15000, "lon": -13.70000 }],
  "ISL-r":  [{"name": "Reykjavik Aero/HFDL", "code": "ISL-r", "lat": 64.08333, "lon": -21.85000 }],
  "ISL-rf": [{"name": "Raufarhöfn", "code": "ISL-rf", "lat": 66.45000, "lon": -15.93333 }],
  "ISL-rs": [{"name": "Reykjavik-Seltjarjarnes", "code": "ISL-rs", "lat": 64.15083, "lon": -22.02778 }],
  "ISL-s":  [{"name": "Saudanes", "code": "ISL-s", "lat": 66.18556, "lon": -18.95139 }],
  "ISL-sh": [{"name": "Stórhöfði", "code": "ISL-sh", "lat": 63.39944, "lon": -20.28861 }],
  "ISL-v":  [{"name": "Vestmannaeyjar", "code": "ISL-v", "lat": 63.45000, "lon": -20.26667 }],
  "ISR-h":  [{"name": "Haifa", "code": "ISR-h", "lat": 32.81667, "lon": 35.00000 }],
  "ISR-L":  [{"name": "Lod (Galei Zahal)", "code": "ISR-L", "lat": 31.96667, "lon": 34.86667 }],
  "ISR-sy": [{"name": "She'ar-Yeshuv", "code": "ISR-sy", "lat": 33.21528, "lon": 35.64472 }],
  "ISR-y":  [{"name": "Yavne", "code": "ISR-y", "lat": 31.90000, "lon": 34.75000 }],
  "J-ao":   [{"name": "Aonoyama Signal Station, Utazu (Kagawa)", "code": "J-ao", "lat": 34.30000, "lon": 133.81667 }],
  "J-as":   [{"name": "Asahikawa AF JJU22", "code": "J-as", "lat": 43.80000, "lon": 142.36667 }],
  "J-ay":   [{"name": "Ashiya AB JJZ59", "code": "J-ay", "lat": 33.88333, "lon": 130.65000 }],
  "J-c":    [{"name": "Chiba Nagara", "code": "J-c", "lat": 35.46667, "lon": 140.21667 }],
  "J-ct":   [{"name": "Chitose AB, Hokkaido JJR20", "code": "J-ct", "lat": 42.80000, "lon": 141.66667 }],
  "J-es":   [{"name": "Esaki Signal Station (Osaka Bay), Awaji island (Hyogo)", "code": "J-es", "lat": 34.59833, "lon": 134.99222 }],
  "J-f":    [{"name": "Chofu Campus, Tokyo", "code": "J-f", "lat": 35.65000, "lon": 139.55000 }],
  "J-fu":   [{"name": "Fuchu AB JJT55", "code": "J-fu", "lat": 35.68333, "lon": 139.50000 }],
  "J-gf":   [{"name": "Gifu AB JJV67", "code": "J-gf", "lat": 35.40000, "lon": 136.86667 }],
  "J-h":    [{"name": "Mount Hagane", "code": "J-h", "lat": 33.46500, "lon": 130.17556 }],
  "J-hf":   [{"name": "Hofu / Bofu AB JJX36", "code": "J-hf", "lat": 34.03333, "lon": 131.55000 }],
  "J-hm":   [{"name": "Hamamatsu AB JJV56", "code": "J-hm", "lat": 34.75000, "lon": 137.70000 }],
  "J-hy":   [{"name": "Hyakurigahara AB JJT33", "code": "J-hy", "lat": 36.18333, "lon": 140.41667 }],
  "J-io":   [{"name": "Imabari Ohama Vessel Station (Kurushima), Imabari (Ehime)", "code": "J-io", "lat": 34.09028, "lon": 132.98778 }],
  "J-ir":   [{"name": "Iruma / Irumagawa AB JJT44", "code": "J-ir", "lat": 35.85000, "lon": 139.41667 }],
  "J-it":   [{"name": "Itoman, Okinawa JFE", "code": "J-it", "lat": 26.15000, "lon": 127.66667 }],
  "J-iw":   [{"name": "Isewan Signal Station, Cape Irago, Tahara (Aichi)", "code": "J-iw", "lat": 34.58333, "lon": 137.01667 }],
  "J-k":    [{"name": "Kagoshima JMH", "code": "J-k", "lat": 31.31667, "lon": 130.51667 }],
  "J-kg":   [{"name": "Kumagaya AB JJT66", "code": "J-kg", "lat": 36.16667, "lon": 139.31667 }],
  "J-ki":   [{"name": "Kisarazu AB JJT22", "code": "J-ki", "lat": 35.40000, "lon": 139.91667 }],
  "J-kk":   [{"name": "Komaki AB (Nagoya) JJV23", "code": "J-kk", "lat": 35.25000, "lon": 136.91667 }],
  "J-km":   [{"name": "Kume Shima / Kumejina, Okinawa JJU66", "code": "J-km", "lat": 26.36667, "lon": 126.71667 }],
  "J-kn":   [{"name": "Kanmon Oseto Strait Signal Station,", "code": "J-kn", "lat": 33.91667, "lon": 130.93333 }],
  "J-ko":   [{"name": "Komatsu AB JJV90", "code": "J-ko", "lat": 36.40000, "lon": 136.40000 }],
  "J-ks":   [{"name": "Kasuga AB JJZ37", "code": "J-ks", "lat": 33.53333, "lon": 130.46667 }],
  "J-ku":   [{"name": "Kumamoto JJE20", "code": "J-ku", "lat": 32.83333, "lon": 130.85000 }],
  "J-ky":   [{"name": "Kyodo", "code": "J-ky", "lat": 36.18333, "lon": 139.85000 }],
  "J-kz":   [{"name": "Kannonzaki Signal Station, Yokosuka (Kanagawa)", "code": "J-kz", "lat": 35.25611, "lon": 139.74333 }],
  "J-m":    [{"name": "Miura", "code": "J-m", "lat": 35.13972, "lon": 139.64222 }],
  "J-mh":   [{"name": "Miho AB, Yonago JJX25", "code": "J-mh", "lat": 35.50000, "lon": 133.23333 }],
  "J-ms":   [{"name": "Misawa AB JJS21", "code": "J-ms", "lat": 40.70000, "lon": 141.36667 }],
  "J-mt":   [{"name": "Matsushima AB JJS32", "code": "J-mt", "lat": 38.40000, "lon": 141.21667 }],
  "J-mu":   [{"name": "Muroto (Kochi, Shikoku)", "code": "J-mu", "lat": 33.28333, "lon": 134.15000 }],
  "J-mz":   [{"name": "Makurazaki (Kagoshima, Kyushu)", "code": "J-mz", "lat": 31.26667, "lon": 130.30000 }],
  "J-n":    [{"name": "Nemuro", "code": "J-n", "lat": 43.30000, "lon": 145.56667 }],
  "J-nh":   [{"name": "Naha AB, Okinawa", "code": "J-nh", "lat": 26.20000, "lon": 127.65000 }],
  "J-nk":   [{"name": "Nagoya Kinjo Signal Station, Nagoya (Aichi)", "code": "J-nk", "lat": 35.03500, "lon": 136.84639 }],
  "J-nr":   [{"name": "Nara JJW24", "code": "J-nr", "lat": 34.56667, "lon": 135.76667 }],
  "J-ny":   [{"name": "Nyutabaru AB JJZ26", "code": "J-ny", "lat": 32.08333, "lon": 131.45000 }],
  "J-o":    [{"name": "Mount Otakadoya", "code": "J-o", "lat": 37.37278, "lon": 140.84889 }],
  "J-oe":   [{"name": "Okinoerabu JJZ44", "code": "J-oe", "lat": 27.43333, "lon": 128.70000 }],
  "J-ok":   [{"name": "Okinawa", "code": "J-ok", "lat": 26.48333, "lon": 127.93333 }],
  "J-os":   [{"name": "Osaka JJD20", "code": "J-os", "lat": 34.78333, "lon": 135.43333 }],
  "J-ot":   [{"name": "Otaru, Hokkaido JJS65", "code": "J-ot", "lat": 43.18333, "lon": 141.00000 }],
  "J-sa":   [{"name": "Sapporo / Chitose AB JJA20", "code": "J-sa", "lat": 42.78333, "lon": 141.66667 }],
  "J-sg":   [{"name": "Sodegaura (Kubota), Chiba", "code": "J-sg", "lat": 35.44667, "lon": 140.01972 }],
  "J-sn":   [{"name": "Sendai / Kasuminome AB JJB20", "code": "J-sn", "lat": 38.23333, "lon": 140.91667 }],
  "J-sz":   [{"name": "Shizuoka", "code": "J-sz", "lat": 34.98333, "lon": 138.38333 }],
  "J-tk":   [{"name": "Tokyo / Tachikawa AF JJC20 JJT88", "code": "J-tk", "lat": 35.71667, "lon": 139.40000 }],
  "J-ts":   [{"name": "Tsuiki AB JJZ48", "code": "J-ts", "lat": 33.68333, "lon": 131.03333 }],
  "J-tv":   [{"name": "Tokyo Volmet, Kagoshima Broadcasting Station", "code": "J-tv", "lat": 31.71667, "lon": 130.73333 }],
  "J-y":    [{"name": "Yamata", "code": "J-y", "lat": 36.16667, "lon": 139.83333 }],
  "J-yo":   [{"name": "Yokota AFB", "code": "J-yo", "lat": 35.74861, "lon": 139.34861 }],
  "J-yz":   [{"name": "Yozadake (Okinawa)", "code": "J-yz", "lat": 26.13333, "lon": 127.70000 }],
  "JOR-ak": [{"name": "Al Karanah / Qast Kherane", "code": "JOR-ak", "lat": 31.73333, "lon": 36.43333 }],
  "JOR-am": [{"name": "Amman", "code": "JOR-am", "lat": 31.96667, "lon": 35.88333 }],
  "KAZ-a":  [{"name": "Almaty", "code": "KAZ-a", "lat": 43.25000, "lon": 76.91667 }],
  "KAZ-ak": [{"name": "Aktyubinsk/Aktöbe", "code": "KAZ-ak", "lat": 50.25000, "lon": 57.21667 }],
  "KAZ-av": [{"name": "Almaty Volmet", "code": "KAZ-av", "lat": 43.35000, "lon": 77.05000 }],
  "KAZ-n":  [{"name": "Nursultan (Akmolinsk/Tselinograd/Astana)", "code": "KAZ-n", "lat": 51.01667, "lon": 71.46667 }],
  "KEN-ny": [{"name": "Nairobi 5YE", "code": "KEN-ny", "lat": -1.25000, "lon": 36.86667 }],
  "KGZ-b":  [{"name": "Bishkek (Krasnaya Rechka)", "code": "KGZ-b", "lat": 42.88333, "lon": 74.98333 }],
  "KGZ-bk": [{"name": "Bishkek Beta", "code": "KGZ-bk", "lat": 43.06667, "lon": 73.65000 }],
  "KOR-c":  [{"name": "Chuncheon", "code": "KOR-c", "lat": 37.93333, "lon": 127.76667 }],
  "KOR-d":  [{"name": "Dangjin", "code": "KOR-d", "lat": 36.96667, "lon": 126.61667 }],
  "KOR-db": [{"name": "Daebu-do (Ansan)", "code": "KOR-db", "lat": 37.22056, "lon": 126.55806 }],
  "KOR-g":  [{"name": "Goyang / Koyang, Gyeonggi-do / Kyonggi-do", "code": "KOR-g", "lat": 37.60000, "lon": 126.85000 }],
  "KOR-h":  [{"name": "Hwasung/Hwaseong", "code": "KOR-h", "lat": 37.21667, "lon": 126.78333 }],
  "KOR-j":  [{"name": "Jeju/Aewol HLAZ", "code": "KOR-j", "lat": 33.48333, "lon": 126.38333 }],
  "KOR-k":  [{"name": "Kimjae", "code": "KOR-k", "lat": 35.83333, "lon": 126.83333 }],
  "KOR-m":  [{"name": "Muan HFDL", "code": "KOR-m", "lat": 35.03222, "lon": 126.23861 }],
  "KOR-n":  [{"name": "Hwaseong?", "code": "KOR-n", "lat": 37.21667, "lon": 126.78333 }],
  "KOR-o":  [{"name": "Suwon-Osan/Hwaseong-Jeongnam", "code": "KOR-o", "lat": 37.15000, "lon": 127.00000 }],
  "KOR-s":  [{"name": "Seoul-Incheon HLKX", "code": "KOR-s", "lat": 37.41667, "lon": 126.75000 }],
  "KOR-sg": [{"name": "Seoul-Gangseo-gu", "code": "KOR-sg", "lat": 37.56667, "lon": 126.96667 }],
  "KOR-t":  [{"name": "Taedok", "code": "KOR-t", "lat": 36.38333, "lon": 127.36667 }],
  "KOR-w":  [{"name": "Nowon Gyeonggi-do / Seoul-Taereung", "code": "KOR-w", "lat": 37.63333, "lon": 127.11667 }],
  "KOS-b":  [{"name": "Camp Bondsteel", "code": "KOS-b", "lat": 42.36667, "lon": 21.25000 }],
  "KRE-c":  [{"name": "Chongjin", "code": "KRE-c", "lat": 41.78333, "lon": 129.83333 }],
  "KRE-e":  [{"name": "Hyesan", "code": "KRE-e", "lat": 41.06667, "lon": 128.03333 }],
  "KRE-h":  [{"name": "Hamhung", "code": "KRE-h", "lat": 39.93333, "lon": 127.65000 }],
  "KRE-hw": [{"name": "Hwadae/Kimchaek", "code": "KRE-hw", "lat": 40.68333, "lon": 129.20000 }],
  "KRE-j":  [{"name": "Haeju", "code": "KRE-j", "lat": 38.01667, "lon": 125.71667 }],
  "KRE-k":  [{"name": "Kanggye", "code": "KRE-k", "lat": 40.96667, "lon": 126.60000 }],
  "KRE-kn": [{"name": "Kangnam", "code": "KRE-kn", "lat": 38.90000, "lon": 125.65000 }],
  "KRE-p":  [{"name": "Pyongyang", "code": "KRE-p", "lat": 39.08333, "lon": 125.38333 }],
  "KRE-s":  [{"name": "Sariwon", "code": "KRE-s", "lat": 38.08333, "lon": 125.13333 }],
  "KRE-sg": [{"name": "Samgo", "code": "KRE-sg", "lat": 38.03333, "lon": 126.53333 }],
  "KRE-sn": [{"name": "Sinuiju", "code": "KRE-sn", "lat": 40.08333, "lon": 124.45000 }],
  "KRE-sw": [{"name": "Sangwon", "code": "KRE-sw", "lat": 38.85000, "lon": 125.51667 }],
  "KRE-u":  [{"name": "Kujang", "code": "KRE-u", "lat": 40.08333, "lon": 126.08333 }],
  "KRE-w":  [{"name": "Wonsan", "code": "KRE-w", "lat": 39.08333, "lon": 127.41667 }],
  "KRE-y":  [{"name": "Pyongsong", "code": "KRE-y", "lat": 40.08333, "lon": 124.40000 }],
  "KWT-j":  [{"name": "Jahra/Umm al-Rimam", "code": "KWT-j", "lat": 29.50000, "lon": 47.66667 }],
  "KWT-k":  [{"name": "Kabd/Sulaibiyah", "code": "KWT-k", "lat": 29.15000, "lon": 47.76667 }],
  "KWT-kw": [{"name": "Kuwait", "code": "KWT-kw", "lat": 29.38333, "lon": 47.65000 }],
  "LAO-s":  [{"name": "Sam Neua", "code": "LAO-s", "lat": 20.26667, "lon": 104.06667 }],
  "LAO-v":  [{"name": "Vientiane", "code": "LAO-v", "lat": 17.96667, "lon": 102.55000 }],
  "LBN-be": [{"name": "Beirut", "code": "LBN-be", "lat": 33.85000, "lon": 35.55000 }],
  "LBR-e":  [{"name": "Monrovia ELWA", "code": "LBR-e", "lat": 6.23333, "lon": -10.70000 }],
  "LBR-m":  [{"name": "Monrovia Mamba Point", "code": "LBR-m", "lat": 6.31667, "lon": -10.81667 }],
  "LBR-s":  [{"name": "Star Radio Monrovia", "code": "LBR-s", "lat": 6.30000, "lon": -10.78333 }],
  "LBY":    [{"name": "Sabrata", "code": "LBY", "lat": 32.90000, "lon": 13.18333 }],
  "LTU":    [{"name": "Sitkunai", "code": "LTU", "lat": 55.04361, "lon": 23.80778 }],
  "LTU-v":  [{"name": "Viesintos", "code": "LTU-v", "lat": 55.70000, "lon": 24.98333 }],
  "LUX-j":  [{"name": "Junglinster", "code": "LUX-j", "lat": 49.71667, "lon": 6.25000 }],
  "LUX-m":  [{"name": "Marnach", "code": "LUX-m", "lat": 50.05000, "lon": 6.08333 }],
  "LVA":    [{"name": "Ulbroka", "code": "LVA", "lat": 56.93333, "lon": 24.28333 }],
  "MAU-m":  [{"name": "Malherbes", "code": "MAU-m", "lat": -20.34111, "lon": 57.51278 }],
  "MDA":    [{"name": "Maiac near Grigoriopol", "code": "MDA", "lat": 47.28333, "lon": 29.40000 }],
  "MDA-ca": [{"name": "Cahul", "code": "MDA-ca", "lat": 45.93333, "lon": 28.28333 }],
  "MDA-ce": [{"name": "Chisinau", "code": "MDA-ce", "lat": 47.01667, "lon": 28.81667 }],
  "MDA-co": [{"name": "Codru-Costiujeni", "code": "MDA-co", "lat": 46.95000, "lon": 28.83333 }],
  "MDA-ed": [{"name": "Edinet", "code": "MDA-ed", "lat": 48.18333, "lon": 27.30000 }],
  "MDG":    [{"name": "Talata Volonondry", "code": "MDG", "lat": -18.83333, "lon": 47.58333 }],
  "MDG-a":  [{"name": "Ambohidrano/Sabotsy", "code": "MDG-a", "lat": -18.91667, "lon": 47.53333 }],
  "MDG-m":  [{"name": "Mahajanga (WCBC)", "code": "MDG-m", "lat": -15.72722, "lon": 46.44583 }],
  "MDR-ps": [{"name": "Porto Santo", "code": "MDR-ps", "lat": 33.06667, "lon": -16.35000 }],
  "MEX-c":  [{"name": "Cuauhtémoc, Mexico City", "code": "MEX-c", "lat": 19.43333, "lon": -99.15000 }],
  "MEX-cb": [{"name": "Choya Bay, Sonora", "code": "MEX-cb", "lat": 31.333333, "lon": -113.633333 }],
  "MEX-e":  [{"name": "Mexico City (Radio Educación)", "code": "MEX-e", "lat": 19.26667, "lon": -99.05000 }],
  "MEX-i":  [{"name": "Iztacalco, Mexico City", "code": "MEX-i", "lat": 19.38333, "lon": -98.95000 }],
  "MEX-m":  [{"name": "Merida", "code": "MEX-m", "lat": 20.96667, "lon": -89.60000 }],
  "MEX-pr": [{"name": "Progreso", "code": "MEX-pr", "lat": 21.26667, "lon": -89.78333 }],
  "MEX-s":  [{"name": "San Luis Potosi", "code": "MEX-s", "lat": 22.16667, "lon": -101.00000 }],
  "MEX-p":  [{"name": "Chiapas", "code": "MEX-p", "lat": 17.00000, "lon": -92.00000 }],
  "MEX-u":  [{"name": "UNAM, Mexico City", "code": "MEX-u", "lat": 19.38333, "lon": -99.16667 }],
  "MEX-vh": [{"name": "Villahermosa, Tabasco", "code": "MEX-vh", "lat": 18.00000, "lon": -93.00000 }],
  "MLA-ka": [{"name": "Kajang", "code": "MLA-ka", "lat": 3.01667, "lon": 101.76667 }],
  "MLA-kk": [{"name": "Kota Kinabalu", "code": "MLA-kk", "lat": 6.20000, "lon": 116.23333 }],
  "MLA-ku": [{"name": "Kuching-Stapok (closed 2011)", "code": "MLA-ku", "lat": 1.55000, "lon": 110.33333 }],
  "MLA-l":  [{"name": "Lumut", "code": "MLA-l", "lat": 4.23333, "lon": 100.63333 }],
  "MLA-s":  [{"name": "Sibu", "code": "MLA-s", "lat": 2.30000, "lon": 111.81667 }],
  "MLI-c":  [{"name": "CRI-Bamako", "code": "MLI-c", "lat": 12.68333, "lon": -8.03333 }],
  "MLI-k":  [{"name": "Kati(Bamako)", "code": "MLI-k", "lat": 12.75000, "lon": -8.05000 }],
  "MLT-mr": [{"name": "Malta Radio", "code": "MLT-mr", "lat": 35.81667, "lon": 14.53333 }],
  "MNE-oc": [{"name": "Ocas", "code": "MNE-oc", "lat": 42.01667, "lon": 19.13333 }],
  "MNG-a":  [{"name": "Altay", "code": "MNG-a", "lat": 46.31667, "lon": 96.25000 }],
  "MNG-m":  [{"name": "Moron/Mörön", "code": "MNG-m", "lat": 49.61667, "lon": 100.16667 }],
  "MNG-u":  [{"name": "Ulaanbaatar-Khonkhor", "code": "MNG-u", "lat": 47.91667, "lon": 107.00000 }],
  "MRA-m":  [{"name": "Marpi, Saipan (KFBS)", "code": "MRA-m", "lat": 15.26667, "lon": 145.80000 }],
  "MRA-s":  [{"name": "Saipan/Agingan Point (IBB)", "code": "MRA-s", "lat": 15.11667, "lon": 145.68333 }],
  "MRA-t":  [{"name": "Tinian (IBB)", "code": "MRA-t", "lat": 15.05000, "lon": 145.60000 }],
  "MRC-ag": [{"name": "Agadir", "code": "MRC-ag", "lat": 30.36667, "lon": -9.55000 }],
  "MRC-b":  [{"name": "Briech (VoA/RL/RFE)", "code": "MRC-b", "lat": 35.55000, "lon": -5.96667 }],
  "MRC-ca": [{"name": "Casablanca", "code": "MRC-ca", "lat": 33.61667, "lon": -7.63333 }],
  "MRC-L":  [{"name": "Laayoune (UN 6678)", "code": "MRC-L", "lat": 27.15000, "lon": -13.21667 }],
  "MRC-n":  [{"name": "Nador (RTM,Medi1)", "code": "MRC-n", "lat": 34.96667, "lon": -2.91667 }],
  "MRC-s":  [{"name": "Safi", "code": "MRC-s", "lat": 32.30000, "lon": -9.23333 }],
  "MRT-fc": [{"name": "Fort-de-France CROSS", "code": "MRT-fc", "lat": 14.60000, "lon": -61.08333 }],
  "MRT-u":  [{"name": "FUF Martinique", "code": "MRT-u", "lat": 14.53194, "lon": -60.97889 }],
  "MTN":    [{"name": "Nouakchott", "code": "MTN", "lat": 18.11667, "lon": -15.95000 }],
  "MYA-n":  [{"name": "Naypyidaw", "code": "MYA-n", "lat": 20.18333, "lon": 96.13333 }],
  "MYA-p":  [{"name": "Phin Oo Lwin, Mandalay", "code": "MYA-p", "lat": 22.01667, "lon": 96.55000 }],
  "MYA-t":  [{"name": "Taunggyi(Kalaw)", "code": "MYA-t", "lat": 20.63333, "lon": 96.58333 }],
  "MYA-y":  [{"name": "Yegu (Yangon/Rangoon)", "code": "MYA-y", "lat": 16.86667, "lon": 96.16667 }],
  "NCL-n":  [{"name": "Nouméa - Ouen-Toro", "code": "NCL-n", "lat": -22.30528, "lon": 166.45472 }],
  "NFK":    [{"name": "Norfolk Island", "code": "NFK", "lat": -29.03333, "lon": 167.95000 }],
  "NGR":    [{"name": "Niamey", "code": "NGR", "lat": 13.50000, "lon": 2.10000 }],
  "NIG-a":  [{"name": "Abuja-Gwagwalada", "code": "NIG-a", "lat": 8.93333, "lon": 7.06667 }],
  "NIG-b":  [{"name": "Ibadan", "code": "NIG-b", "lat": 7.38333, "lon": 3.90000 }],
  "NIG-e":  [{"name": "Enugu", "code": "NIG-e", "lat": 6.45000, "lon": 7.45000 }],
  "NIG-i":  [{"name": "Ikorodu", "code": "NIG-i", "lat": 6.60000, "lon": 3.50000 }],
  "NIG-j":  [{"name": "Abuja-Lugbe (new site, opened March 2012)", "code": "NIG-j", "lat": 8.96667, "lon": 7.35000 }],
  "NIG-k":  [{"name": "Kaduna", "code": "NIG-k", "lat": 10.51667, "lon": 7.41667 }],
  "NMB-wb": [{"name": "Walvis Bay", "code": "NMB-wb", "lat": -23.08750, "lon": 14.62500 }],
  "NOR-a":  [{"name": "Andenes", "code": "NOR-a", "lat": 69.26889, "lon": 16.04056 }],
  "NOR-as": [{"name": "Andenes-Saura", "code": "NOR-as", "lat": 69.14000, "lon": 16.02000 }],
  "NOR-at": [{"name": "Andenes (Telenor site)", "code": "NOR-at", "lat": 69.28333, "lon": 16.05000 }],
  "NOR-be": [{"name": "Bergen (LLE station, Erdal, Askoy Island)", "code": "NOR-be", "lat": 60.43333, "lon": 5.21667 }],
  "NOR-bj": [{"name": "Bjørnøya / Bear Island", "code": "NOR-bj", "lat": 74.43333, "lon": 19.05000 }],
  "NOR-bk": [{"name": "Bergen-Kvarren", "code": "NOR-bk", "lat": 60.38944, "lon": 5.25000 }],
  "NOR-bo": [{"name": "Bodø", "code": "NOR-bo", "lat": 67.28333, "lon": 14.38333 }],
  "NOR-bs": [{"name": "Bodø-Seines", "code": "NOR-bs", "lat": 67.20000, "lon": 14.36667 }],
  "NOR-bv": [{"name": "Berlevåg", "code": "NOR-bv", "lat": 70.86667, "lon": 29.06667 }],
  "NOR-e":  [{"name": "Erdal", "code": "NOR-e", "lat": 60.44889, "lon": 5.21639 }],
  "NOR-f":  [{"name": "Florø", "code": "NOR-f", "lat": 61.60000, "lon": 5.03333 }],
  "NOR-fs": [{"name": "Farsund", "code": "NOR-fs", "lat": 58.08333, "lon": 6.78333 }],
  "NOR-hf": [{"name": "Hammerfest", "code": "NOR-hf", "lat": 70.66667, "lon": 23.68333 }],
  "NOR-hp": [{"name": "Hopen Island", "code": "NOR-hp", "lat": 76.55000, "lon": 25.11667 }],
  "NOR-jm": [{"name": "Jan Mayen Island", "code": "NOR-jm", "lat": 71.00000, "lon": -8.50000 }],
  "NOR-ly": [{"name": "Longyearbyen, Svalbard", "code": "NOR-ly", "lat": 78.06667, "lon": 13.61667 }],
  "NOR-ma": [{"name": "Marøy", "code": "NOR-ma", "lat": 60.70000, "lon": 4.88333 }],
  "NOR-mg": [{"name": "Molde-Gossen", "code": "NOR-mg", "lat": 62.84167, "lon": 6.79306 }],
  "NOR-mr": [{"name": "Mågerø", "code": "NOR-mr", "lat": 59.15000, "lon": 10.43333 }],
  "NOR-nm": [{"name": "Nordmela-Andøya", "code": "NOR-nm", "lat": 69.12778, "lon": 15.63333 }],
  "NOR-no": [{"name": "Novik", "code": "NOR-no", "lat": 66.982778, "lon": 13.873056 }],
  "NOR-oh": [{"name": "Oslo-Helgelandsmoen", "code": "NOR-oh", "lat": 60.11667, "lon": 10.20000 }],
  "NOR-or": [{"name": "Ørlandet", "code": "NOR-or", "lat": 63.68333, "lon": 9.65000 }],
  "NOR-ro": [{"name": "Rogaland (Vigreskogen)", "code": "NOR-ro", "lat": 58.65000, "lon": 5.58333 }],
  "NOR-sa": [{"name": "Sandnessjøen", "code": "NOR-sa", "lat": 66.01667, "lon": 12.63333 }],
  "NOR-sr": [{"name": "Sørreisa", "code": "NOR-sr", "lat": 69.06667, "lon": 18.00000 }],
  "NOR-st": [{"name": "Stavanger-Ulsnes", "code": "NOR-st", "lat": 59.00000, "lon": 5.71667 }],
  "NOR-tj": [{"name": "Tjøme", "code": "NOR-tj", "lat": 59.110578, "lon": 10.402599 }],
  "NOR-va": [{"name": "Vardø", "code": "NOR-va", "lat": 70.36667, "lon": 31.10000 }],
  "NPL":    [{"name": "Khumaltar", "code": "NPL", "lat": 27.50000, "lon": 85.50000 }],
  "NZL-a":  [{"name": "Auckland (Wiroa Island)", "code": "NZL-a", "lat": -37.01667, "lon": 174.81667 }],
  "NZL-du": [{"name": "Dunedin", "code": "NZL-du", "lat": -45.86667, "lon": 170.50000 }],
  "NZL-r":  [{"name": "Rangitaiki", "code": "NZL-r", "lat": -38.83333, "lon": 176.41667 }],
  "NZL-ru": [{"name": "Russell", "code": "NZL-ru", "lat": -35.28333, "lon": 174.11667 }],
  "NZL-t":  [{"name": "Taupo", "code": "NZL-t", "lat": -38.86667, "lon": 176.43333 }],
  "OCE-fa": [{"name": "Faa'a airport", "code": "OCE-fa", "lat": -17.55000, "lon": -149.61667 }],
  "OCE-ma": [{"name": "Mahina (Tahiti)", "code": "OCE-ma", "lat": -17.50583, "lon": -149.48250 }],
  "OMA-a":  [{"name": "A'Seela", "code": "OMA-a", "lat": 21.91667, "lon": 59.61667 }],
  "OMA-s":  [{"name": "Seeb", "code": "OMA-s", "lat": 23.66667, "lon": 58.16667 }],
  "OMA-t":  [{"name": "Thumrait", "code": "OMA-t", "lat": 17.63333, "lon": 53.93333 }],
  "PAK-i":  [{"name": "Islamabad", "code": "PAK-i", "lat": 33.45000, "lon": 73.20000 }],
  "PAK-kv": [{"name": "Karachi Volmet", "code": "PAK-kv", "lat": 24.90000, "lon": 67.16667 }],
  "PAK-m":  [{"name": "Multan", "code": "PAK-m", "lat": 30.08944, "lon": 71.49167 }],
  "PAK-p":  [{"name": "Peshawar", "code": "PAK-p", "lat": 34.00000, "lon": 71.50000 }],
  "PAK-q":  [{"name": "Quetta", "code": "PAK-q", "lat": 30.25000, "lon": 67.00000 }],
  "PAK-r":  [{"name": "Rawalpindi", "code": "PAK-r", "lat": 33.50000, "lon": 73.00000 }],
  "PAQ":    [{"name": "Easter Island", "code": "PAQ", "lat": -27.116667, "lon": -109.366667 }],
  "PHL-b":  [{"name": "Bocaue (FEBC)", "code": "PHL-b", "lat": 14.80000, "lon": 120.91667 }],
  "PHL-dv": [{"name": "Davao City, Mindanao", "code": "PHL-dv", "lat": 7.08333, "lon": 125.60000 }],
  "PHL-i":  [{"name": "Iba (FEBC)", "code": "PHL-i", "lat": 15.33333, "lon": 119.96667 }],
  "PHL-ko": [{"name": "Koronadal City, Mindanao", "code": "PHL-ko", "lat": 6.51667, "lon": 124.81667 }],
  "PHL-m":  [{"name": "Marulas/Quezon City, Valenzuela (PBS 6170,9581)", "code": "PHL-m", "lat": 14.68333, "lon": 120.96667 }],
  "PHL-p":  [{"name": "Palauig, Zembales (RVA)", "code": "PHL-p", "lat": 15.46667, "lon": 119.83333 }],
  "PHL-po": [{"name": "Poro", "code": "PHL-po", "lat": 16.43333, "lon": 120.28333 }],
  "PHL-sc": [{"name": "Santiago City, Luzon", "code": "PHL-sc", "lat": 16.70000, "lon": 121.60000 }],
  "PHL-t":  [{"name": "Tinang (VoA)", "code": "PHL-t", "lat": 15.35000, "lon": 120.61667 }],
  "PHL-x":  [{"name": "Tinang-2/portable 50kW (VoA)", "code": "PHL-x", "lat": 15.35000, "lon": 120.61667 }],
  "PHL-zm": [{"name": "Zamboanga City, Mindanao", "code": "PHL-zm", "lat": 6.91667, "lon": 122.11667 }],
  "PLW":    [{"name": "Koror-Babeldaob (Medorn)", "code": "PLW", "lat": 7.45611, "lon": 134.47333 }],
  "PNG-a":  [{"name": "Alotau", "code": "PNG-a", "lat": -10.30000, "lon": 150.46667 }],
  "PNG-b":  [{"name": "Bougainville/Buka-Kubu", "code": "PNG-b", "lat": -5.41667, "lon": 154.66667 }],
  "PNG-d":  [{"name": "Daru", "code": "PNG-d", "lat": -9.08333, "lon": 143.16667 }],
  "PNG-g":  [{"name": "Goroka", "code": "PNG-g", "lat": -6.03333, "lon": 145.36667 }],
  "PNG-ka": [{"name": "Kavieng", "code": "PNG-ka", "lat": -2.56667, "lon": 150.80000 }],
  "PNG-kb": [{"name": "Kimbe", "code": "PNG-kb", "lat": -5.55000, "lon": 150.15000 }],
  "PNG-ke": [{"name": "Kerema", "code": "PNG-ke", "lat": -7.98333, "lon": 145.76667 }],
  "PNG-ki": [{"name": "Kiunga", "code": "PNG-ki", "lat": -6.11667, "lon": 141.30000 }],
  "PNG-ku": [{"name": "Kundiawa", "code": "PNG-ku", "lat": -6.00000, "lon": 144.95000 }],
  "PNG-la": [{"name": "Lae (Morobe)", "code": "PNG-la", "lat": -6.73333, "lon": 147.00000 }],
  "PNG-ln": [{"name": "Lae Nadzab airport", "code": "PNG-ln", "lat": -6.56667, "lon": 146.73333 }],
  "PNG-lo": [{"name": "Lorengau", "code": "PNG-lo", "lat": -2.01667, "lon": 147.25000 }],
  "PNG-ma": [{"name": "Madang", "code": "PNG-ma", "lat": -5.23333, "lon": 145.75000 }],
  "PNG-me": [{"name": "Mendi", "code": "PNG-me", "lat": -6.21667, "lon": 143.65000 }],
  "PNG-mh": [{"name": "Mount Hagen", "code": "PNG-mh", "lat": -5.90000, "lon": 144.21667 }],
  "PNG-pm": [{"name": "Port Moresby (Waigani)", "code": "PNG-pm", "lat": -9.46667, "lon": 147.18333 }],
  "PNG-pr": [{"name": "Port Moresby Maritime Radio", "code": "PNG-pr", "lat": -9.46667, "lon": 147.18333 }],
  "PNG-po": [{"name": "Popondetta", "code": "PNG-po", "lat": -8.75000, "lon": 148.25000 }],
  "PNG-r":  [{"name": "Rabaul", "code": "PNG-r", "lat": -4.21667, "lon": 152.21667 }],
  "PNG-v":  [{"name": "Vanimo", "code": "PNG-v", "lat": -2.66667, "lon": 141.28333 }],
  "PNG-va": [{"name": "Vanimo", "code": "PNG-va", "lat": -2.68333, "lon": 141.30000 }],
  "PNG-wa": [{"name": "Port Moresby (Radio Wantok)", "code": "PNG-wa", "lat": -9.46667, "lon": 147.18333 }],
  "PNG-ww": [{"name": "Wewak", "code": "PNG-ww", "lat": -3.58333, "lon": 143.66667 }],
  "PNR-al": [{"name": "Albrook, Panama City", "code": "PNR-al", "lat": 8.96889, "lon": -79.55083 }],
  "POL-b":  [{"name": "Babice", "code": "POL-b", "lat": 52.25000, "lon": 20.83333 }],
  "POL-p":  [{"name": "Puchaly, in Falenty", "code": "POL-p", "lat": 52.14361, "lon": 20.90000 }],
  "POL-sk": [{"name": "Solec Kujawski", "code": "POL-sk", "lat": 53.02028, "lon": 18.26222 }],
  "POL-u":  [{"name": "Ustka", "code": "POL-u", "lat": 54.58250, "lon": 16.83667 }],
  "POL-w":  [{"name": "Witowo", "code": "POL-w", "lat": 54.55000, "lon": 16.53333 }],
  "POR-ms": [{"name": "Monsanto", "code": "POR-ms", "lat": 38.73333, "lon": -9.18333 }],
  "PRG-c":  [{"name": "Capiatá", "code": "PRG-c", "lat": -25.40000, "lon": -57.46667 }],
  "PRG-f":  [{"name": "Filadelfia", "code": "PRG-f", "lat": -22.35000, "lon": -60.03333 }],
  "PRU-ar": [{"name": "Arequipa", "code": "PRU-ar", "lat": -16.41667, "lon": -71.53333 }],
  "PRU-at": [{"name": "Atalaya", "code": "PRU-at", "lat": -10.73000, "lon": -73.75556 }],
  "PRU-bv": [{"name": "Bolívar", "code": "PRU-bv", "lat": -7.35000, "lon": -77.83333 }],
  "PRU-cc": [{"name": "Chiclayo/Santa Ana (Carretera a Lambayeque)", "code": "PRU-cc", "lat": -6.73333, "lon": -79.85000 }],
  "PRU-ce": [{"name": "Celendín", "code": "PRU-ce", "lat": -6.88333, "lon": -78.15000 }],
  "PRU-ch": [{"name": "Chachapoyas", "code": "PRU-ch", "lat": -6.23333, "lon": -77.86667 }],
  "PRU-cl": [{"name": "Callalli", "code": "PRU-cl", "lat": -15.50000, "lon": -71.43333 }],
  "PRU-cp": [{"name": "Cerre de Pasco", "code": "PRU-cp", "lat": -10.66667, "lon": -76.25000 }],
  "PRU-ct": [{"name": "Chota", "code": "PRU-ct", "lat": -6.55000, "lon": -78.65000 }],
  "PRU-cu": [{"name": "Cuzco-Cerro Oscollo", "code": "PRU-cu", "lat": -13.51889, "lon": -72.01028 }],
  "PRU-cz": [{"name": "Chazuta/Tarapoto, San Martin", "code": "PRU-cz", "lat": -6.56667, "lon": -76.13333 }],
  "PRU-hb": [{"name": "Huancabamba", "code": "PRU-hb", "lat": -5.23333, "lon": -79.45000 }],
  "PRU-hc": [{"name": "Huancayo/Viques", "code": "PRU-hc", "lat": -12.20167, "lon": -75.21972 }],
  "PRU-ho": [{"name": "Huánuco", "code": "PRU-ho", "lat": -9.93333, "lon": -76.23333 }],
  "PRU-ht": [{"name": "Huanta/Tirapampa", "code": "PRU-ht", "lat": -12.95000, "lon": -74.25000 }],
  "PRU-hu": [{"name": "Huanta/Vista Alegre (Pasaje Amauta)", "code": "PRU-hu", "lat": -12.93667, "lon": -74.25278 }],
  "PRU-hv": [{"name": "Huancavelica", "code": "PRU-hv", "lat": -12.78333, "lon": -74.98333 }],
  "PRU-hz": [{"name": "Huaraz", "code": "PRU-hz", "lat": -9.51667, "lon": -77.53333 }],
  "PRU-in": [{"name": "Chau Alto/Independencia, Huarez, Ancash (planned for 6090 kHz in 2015/16)", "code": "PRU-in", "lat": -9.51806, "lon": -77.54917 }],
  "PRU-iq": [{"name": "Iquitos/Moronacocha", "code": "PRU-iq", "lat": -3.75000, "lon": -73.26667 }],
  "PRU-ja": [{"name": "Jaén", "code": "PRU-ja", "lat": -5.75000, "lon": -78.85000 }],
  "PRU-ju": [{"name": "Junín/Cuncush", "code": "PRU-ju", "lat": -11.16667, "lon": -76.00000 }],
  "PRU-li": [{"name": "Lima", "code": "PRU-li", "lat": -12.10000, "lon": -77.05000 }],
  "PRU-or": [{"name": "La Oroya", "code": "PRU-or", "lat": -11.53333, "lon": -75.90000 }],
  "PRU-pc": [{"name": "Paucartambo", "code": "PRU-pc", "lat": -10.90000, "lon": -75.85000 }],
  "PRU-pm": [{"name": "Puerto Maldonado", "code": "PRU-pm", "lat": -12.60000, "lon": -69.16667 }],
  "PRU-qb": [{"name": "Quillabamba/Macamango", "code": "PRU-qb", "lat": -12.86667, "lon": -72.70000 }],
  "PRU-qt": [{"name": "Quillabamba/Tiobamba Baja", "code": "PRU-qt", "lat": -12.81667, "lon": -72.68333 }],
  "PRU-rm": [{"name": "Rodrigues de Mendoza", "code": "PRU-rm", "lat": -6.38333, "lon": -77.50000 }],
  "PRU-sc": [{"name": "Santa Cruz (R Satelite)", "code": "PRU-sc", "lat": -6.68333, "lon": -79.03333 }],
  "PRU-si": [{"name": "Sicuani", "code": "PRU-si", "lat": -14.26667, "lon": -71.23333 }],
  "PRU-su": [{"name": "Santiago de Chuco", "code": "PRU-su", "lat": -8.15000, "lon": -78.18333 }],
  "PRU-ta": [{"name": "Tarma/Cerro Penitencia", "code": "PRU-ta", "lat": -11.40889, "lon": -75.69194 }],
  "PRU-tc": [{"name": "Tacna", "code": "PRU-tc", "lat": -18.00000, "lon": -70.21667 }],
  "PRU-ur": [{"name": "valle de Urubamba, Cusco", "code": "PRU-ur", "lat": -13.35000, "lon": -72.11667 }],
  "PRU-vv": [{"name": "Valle de Vitor, San Luís, Arequipa", "code": "PRU-vv", "lat": -16.46833, "lon": -71.90389 }],
  "PRU-yu": [{"name": "Yurimaguas", "code": "PRU-yu", "lat": -5.90000, "lon": -76.11667 }],
  "PTR-i":  [{"name": "Isabela", "code": "PTR-i", "lat": 18.38333, "lon": -67.18333 }],
  "PTR-s":  [{"name": "Salinas, Camp Santiago", "code": "PTR-s", "lat": 17.98333, "lon": 66.30000 }],
  "QAT-dr": [{"name": "Doha Radio", "code": "QAT-dr", "lat": 25.70000, "lon": 16.53333 }],
  "REU-su": [{"name": "Sainte-Suzanne", "code": "REU-su", "lat": -20.91000, "lon": 55.58472 }],
  "ROU-b":  [{"name": "Bucuresti/Bucharest airport", "code": "ROU-b", "lat": 44.56667, "lon": 26.08333 }],
  "ROU-c":  [{"name": "Constanta", "code": "ROU-c", "lat": 44.10000, "lon": 28.61667 }],
  "ROU-g":  [{"name": "Galbeni", "code": "ROU-g", "lat": 46.75000, "lon": 26.68333 }],
  "ROU-s":  [{"name": "Saftica 100kW", "code": "ROU-s", "lat": 44.63333, "lon": 27.08333 }],
  "ROU-t":  [{"name": "Tiganesti 300kW", "code": "ROU-t", "lat": 44.75000, "lon": 26.08333 }],
  "RRW":    [{"name": "Kigali", "code": "RRW", "lat": -1.91667, "lon": 30.11667 }],
  "RUS-a":  [{"name": "Armavir/Tblisskaya/Krasnodar", "code": "RUS-a", "lat": 45.48333, "lon": 40.11667 }],
  "RUS-af": [{"name": "Astrakhan Fedorovka", "code": "RUS-af", "lat": 45.84361, "lon": 47.64361 }],
  "RUS-ag": [{"name": "Angarsk", "code": "RUS-ag", "lat": 56.13556, "lon": 101.63889 }],
  "RUS-ak": [{"name": "Arkhangelsk Beta", "code": "RUS-ak", "lat": 64.40000, "lon": 41.53333 }],
  "RUS-am": [{"name": "Amderma", "code": "RUS-am", "lat": 69.76667, "lon": 61.56667 }],
  "RUS-an": [{"name": "Astrakhan Narimanovo", "code": "RUS-an", "lat": 46.28333, "lon": 48.00000 }],
  "RUS-ar": [{"name": "Arkhangelsk", "code": "RUS-ar", "lat": 64.58333, "lon": 40.60000 }],
  "RUS-as": [{"name": "Astrakhan Military Base", "code": "RUS-as", "lat": 47.41667, "lon": 47.91667 }],
  "RUS-at": [{"name": "Arkhangelsk-Talagi", "code": "RUS-at", "lat": 64.60000, "lon": 40.71667 }],
  "RUS-ay": [{"name": "Anadyr, Chukotka", "code": "RUS-ay", "lat": 64.74000, "lon": 177.50889 }],
  "RUS-b":  [{"name": "Blagoveshchensk (Amur)", "code": "RUS-b", "lat": 50.26667, "lon": 127.55000 }],
  "RUS-ba": [{"name": "Barnaul, Altay", "code": "RUS-ba", "lat": 53.33333, "lon": 83.80000 }],
  "RUS-bg": [{"name": "Belaya Gora, Sakha(Yakutia)", "code": "RUS-bg", "lat": 68.53333, "lon": 146.18333 }],
  "RUS-bo": [{"name": "Bolotnoye, Novosibirsk oblast", "code": "RUS-bo", "lat": 55.75611, "lon": 84.44778 }],
  "RUS-B1": [{"name": "Buzzer sites: Kerro (St Petersburg)", "code": "RUS-B1", "lat": 60.31111, "lon": 30.27778 }],
  "RUS-B2": [{"name": "Buzzer sites: Naro-Fominsk (Moscow)", "code": "RUS-B2", "lat": 55.42639, "lon": 36.70917 }],
  "RUS-c":  [{"name": "Chita (Atamanovka) (S Siberia)", "code": "RUS-c", "lat": 51.83333, "lon": 113.71667 }],
  "RUS-cs": [{"name": "Cherskiy, Yakutia", "code": "RUS-cs", "lat": 68.75000, "lon": 161.33333 }],
  "RUS-cy": [{"name": "Chelyabinsk", "code": "RUS-cy", "lat": 55.30000, "lon": 61.50000 }],
  "RUS-di": [{"name": "Dikson", "code": "RUS-di", "lat": 73.50000, "lon": 80.53333 }],
  "RUS-el": [{"name": "Elista", "code": "RUS-el", "lat": 46.36667, "lon": 44.33333 }],
  "RUS-ey": [{"name": "Yeysk port", "code": "RUS-ey", "lat": 46.72361, "lon": 38.27611 }],
  "RUS-ge": [{"name": "Gelendzhik", "code": "RUS-ge", "lat": 44.59889, "lon": 37.96444 }],
  "RUS-gk": [{"name": "Goryachiy Klyuch, Omsk", "code": "RUS-gk", "lat": 55.01778, "lon": 73.19222 }],
  "RUS-go": [{"name": "Gorbusha", "code": "RUS-go", "lat": 55.85778, "lon": 38.22944 }],
  "RUS-gr": [{"name": "Grozny", "code": "RUS-gr", "lat": 43.26667, "lon": 45.71667 }],
  "RUS-i":  [{"name": "Irkutsk (Angarsk) (S Siberia)", "code": "RUS-i", "lat": 52.41667, "lon": 103.66667 }],
  "RUS-ig": [{"name": "Igrim XMAO", "code": "RUS-ig", "lat": 63.18333, "lon": 64.41667 }],
  "RUS-ik": [{"name": "Ivashka, Kamchatka", "code": "RUS-ik", "lat": 58.56194, "lon": 162.30722 }],
  "RUS-ir": [{"name": "Irkutsk port", "code": "RUS-ir", "lat": 52.32083, "lon": 104.28472 }],
  "RUS-iv": [{"name": "Irkutsk Volmet", "code": "RUS-iv", "lat": 52.26667, "lon": 104.38333 }],
  "RUS-iz": [{"name": "Izhevsk sites", "code": "RUS-iz", "lat": 56.83333, "lon": 53.25000 }],
  "RUS-k":  [{"name": "Kaliningrad-Bolshakovo", "code": "RUS-k", "lat": 54.90000, "lon": 21.71667 }],
  "RUS-ka": [{"name": "Komsomolsk-na-Amur (Far East)", "code": "RUS-ka", "lat": 50.65000, "lon": 136.91667 }],
  "RUS-kd": [{"name": "Krasnodar Beta", "code": "RUS-kd", "lat": 44.60000, "lon": 39.56667 }],
  "RUS-kf": [{"name": "Krasnoyarsk HFDL site", "code": "RUS-kf", "lat": 56.10000, "lon": 92.30000 }],
  "RUS-kg": [{"name": "Kaliningrad Radio UIW23", "code": "RUS-kg", "lat": 54.71667, "lon": 20.73333 }],
  "RUS-kh": [{"name": "Khabarovsk (Far East)", "code": "RUS-kh", "lat": 48.55000, "lon": 135.25000 }],
  "RUS-ki": [{"name": "Kirinskoye, Sakhalin", "code": "RUS-ki", "lat": 51.41667, "lon": 143.43333 }],
  "RUS-kl": [{"name": "Kaliningrad Military Base", "code": "RUS-kl", "lat": 54.65000, "lon": 19.91667 }],
  "RUS-km": [{"name": "Khanty-Mansiysk", "code": "RUS-km", "lat": 61.03333, "lon": 69.08333 }],
  "RUS-ko": [{"name": "Korsakov, Sakhalin", "code": "RUS-ko", "lat": 46.61667, "lon": 142.76667 }],
  "RUS-kp": [{"name": "Krasnodar-Poltovskaya", "code": "RUS-kp", "lat": 45.40500, "lon": 38.15806 }],
  "RUS-kr": [{"name": "Krasnoyarsk", "code": "RUS-kr", "lat": 56.03333, "lon": 92.73333 }],
  "RUS-kt": [{"name": "Kotlas", "code": "RUS-kt", "lat": 61.23333, "lon": 46.70000 }],
  "RUS-ku": [{"name": "Kurovskaya-Avsyunino (near Moscow)", "code": "RUS-ku", "lat": 55.56667, "lon": 39.15000 }],
  "RUS-kv": [{"name": "Kirensk Volmet", "code": "RUS-kv", "lat": 57.76667, "lon": 108.06667 }],
  "RUS-kx": [{"name": "Kamenka, Sakha", "code": "RUS-kx", "lat": 69.50000, "lon": 161.33333 }],
  "RUS-ky": [{"name": "Kyzyl", "code": "RUS-ky", "lat": 51.68333, "lon": 94.60000 }],
  "RUS-kz": [{"name": "Kazan", "code": "RUS-kz", "lat": 55.60000, "lon": 49.28333 }],
  "RUS-k1": [{"name": "Krasnodar Pashkovsky", "code": "RUS-k1", "lat": 45.03333, "lon": 39.16667 }],
  "RUS-k2": [{"name": "Kamskoye Ustye, Tatarstan", "code": "RUS-k2", "lat": 55.19556, "lon": 49.28694 }],
  "RUS-k3": [{"name": "Kolpashevo, Tomsk", "code": "RUS-k3", "lat": 58.31833, "lon": 82.98611 }],
  "RUS-k4": [{"name": "Komsomolsk-na-Amure", "code": "RUS-k4", "lat": 50.53333, "lon": 137.03333 }],
  "RUS-k5": [{"name": "Kultayevo, Perm", "code": "RUS-k5", "lat": 57.91056, "lon": 55.91861 }],
  "RUS-k6": [{"name": "Kozmino, Cape Povorotny, Primorye", "code": "RUS-k6", "lat": 42.68056, "lon": 133.03611 }],
  "RUS-L":  [{"name": "Lesnoy (near Moscow)", "code": "RUS-L", "lat": 56.06667, "lon": 37.96667 }],
  "RUS-li": [{"name": "Liski, Voronezh", "code": "RUS-li", "lat": 50.96833, "lon": 39.52139 }],
  "RUS-ln": [{"name": "Labytnangi, YNAO", "code": "RUS-ln", "lat": 66.64361, "lon": 66.52972 }],
  "RUS-l2": [{"name": "Labytnangi, YNAO Gazprom", "code": "RUS-l2", "lat": 66.65833, "lon": 66.22417 }],
  "RUS-m":  [{"name": "Moscow/Moskva (one of ku,L,se,t)", "code": "RUS-m", "lat": 55.75000, "lon": 37.30000 }],
  "RUS-ma": [{"name": "Magadan/Arman", "code": "RUS-ma", "lat": 59.69444, "lon": 150.15861 }],
  "RUS-mg": [{"name": "Magadan Military Base", "code": "RUS-mg", "lat": 59.70000, "lon": 150.16667 }],
  "RUS-mi": [{"name": "Mineralnye Vody", "code": "RUS-mi", "lat": 44.23333, "lon": 43.08333 }],
  "RUS-mk": [{"name": "Makhachkala, Dagestan", "code": "RUS-mk", "lat": 42.81667, "lon": 47.65000 }],
  "RUS-mm": [{"name": "Murmansk Meteo", "code": "RUS-mm", "lat": 68.86667, "lon": 33.07500 }],
  "RUS-mp": [{"name": "Maykop", "code": "RUS-mp", "lat": 44.66667, "lon": 40.03333 }],
  "RUS-mr": [{"name": "Moscow-Razdory", "code": "RUS-mr", "lat": 55.75000, "lon": 37.30000 }],
  "RUS-mt": [{"name": "MTUSI University, Moscow", "code": "RUS-mt", "lat": 55.75417, "lon": 37.71194 }],
  "RUS-mu": [{"name": "Murmansk/Monchegorsk", "code": "RUS-mu", "lat": 67.91667, "lon": 32.98333 }],
  "RUS-mv": [{"name": "Magadan Volmet", "code": "RUS-mv", "lat": 59.91667, "lon": 150.71667 }],
  "RUS-mx": [{"name": "Makhachkala, Dagestan", "code": "RUS-mx", "lat": 43.00361, "lon": 47.47028 }],
  "RUS-mz": [{"name": "Mozdok, North Ossetia", "code": "RUS-mz", "lat": 43.75000, "lon": 44.65000 }],
  "RUS-m2": [{"name": "Magadan Rosmorport", "code": "RUS-m2", "lat": 59.71222, "lon": 150.99417 }],
  "RUS-m3": [{"name": "Makhachkala, Dagestan", "code": "RUS-m3", "lat": 42.98333, "lon": 47.51667 }],
  "RUS-m4": [{"name": "Mezen, Arkhangelsk", "code": "RUS-m4", "lat": 65.90000, "lon": 44.26667 }],
  "RUS-m5": [{"name": "Murmansk MRCC", "code": "RUS-m5", "lat": 68.85278, "lon": 32.98750 }],
  "RUS-n":  [{"name": "Novosibirsk / Oyash, (500 kW, 1000 kW)", "code": "RUS-n", "lat": 55.51667, "lon": 83.75000 }],
  "RUS-nc": [{"name": "Nalchik, Kabardino-Balkaria", "code": "RUS-nc", "lat": 43.51667, "lon": 43.63333 }],
  "RUS-ne": [{"name": "Nevelsk, Sakhalin", "code": "RUS-ne", "lat": 46.51667, "lon": 141.85000 }],
  "RUS-ni": [{"name": "Nizhnevartovsk", "code": "RUS-ni", "lat": 60.95000, "lon": 76.48333 }],
  "RUS-nm": [{"name": "Naryan-Mar", "code": "RUS-nm", "lat": 67.63333, "lon": 53.11667 }],
  "RUS-nn": [{"name": "Nizhni Novgorod sites", "code": "RUS-nn", "lat": 56.18333, "lon": 43.96667 }],
  "RUS-no": [{"name": "Novosibirsk City", "code": "RUS-no", "lat": 55.03333, "lon": 82.91667 }],
  "RUS-np": [{"name": "Novosibirsk city port", "code": "RUS-np", "lat": 55.01528, "lon": 82.91750 }],
  "RUS-nr": [{"name": "Novorossiysk", "code": "RUS-nr", "lat": 44.74611, "lon": 37.68583 }],
  "RUS-ns": [{"name": "Novosibirsk Shipping Canal", "code": "RUS-ns", "lat": 54.84500, "lon": 83.04028 }],
  "RUS-nu": [{"name": "Novy Urengoy", "code": "RUS-nu", "lat": 66.06667, "lon": 76.51667 }],
  "RUS-nv": [{"name": "Novosibirsk Volmet", "code": "RUS-nv", "lat": 55.00444, "lon": 82.56222 }],
  "RUS-ny": [{"name": "Nadym", "code": "RUS-ny", "lat": 65.48333, "lon": 72.70000 }],
  "RUS-oe": [{"name": "Okhotskoye, Sakhalin", "code": "RUS-oe", "lat": 46.87167, "lon": 143.15306 }],
  "RUS-og": [{"name": "Orenburg-Gagarin airport", "code": "RUS-og", "lat": 51.80000, "lon": 55.45000 }],
  "RUS-ok": [{"name": "Oktyarbskiy, Kamchatka", "code": "RUS-ok", "lat": 52.65722, "lon": 156.24472 }],
  "RUS-ol": [{"name": "Oleniy, Yamalo-Nenets", "code": "RUS-ol", "lat": 72.59639, "lon": 77.65944 }],
  "RUS-om": [{"name": "Omsk", "code": "RUS-om", "lat": 54.96667, "lon": 73.31667 }],
  "RUS-or": [{"name": "Orenburg", "code": "RUS-or", "lat": 51.76667, "lon": 55.10000 }],
  "RUS-ox": [{"name": "Okhotsk Bulgin", "code": "RUS-ox", "lat": 59.36667, "lon": 143.15000 }],
  "RUS-p":  [{"name": "Petropavlovsk-Kamchatskiy (Yelizovo)", "code": "RUS-p", "lat": 53.18333, "lon": 158.40000 }],
  "RUS-pc": [{"name": "Pechora", "code": "RUS-pc", "lat": 65.11667, "lon": 57.13333 }],
  "RUS-pe": [{"name": "Perm", "code": "RUS-pe", "lat": 58.05000, "lon": 56.23333 }],
  "RUS-pk": [{"name": "Petropavlovsk-Kamchatskij Military Base", "code": "RUS-pk", "lat": 53.16667, "lon": 158.45000 }],
  "RUS-pm": [{"name": "Perm airport", "code": "RUS-pm", "lat": 57.91667, "lon": 56.01667 }],
  "RUS-po": [{"name": "Preobrazhenie, Primorye", "code": "RUS-po", "lat": 42.89806, "lon": 133.89833 }],
  "RUS-pp": [{"name": "Petropavlovsk-Kamchatskiy Port", "code": "RUS-pp", "lat": 53.06306, "lon": 158.56917 }],
  "RUS-pt": [{"name": "Sankt Peterburg Military Base", "code": "RUS-pt", "lat": 60.00000, "lon": 30.00000 }],
  "RUS-pu": [{"name": "Puteyets, Pechora, Rep.Komi", "code": "RUS-pu", "lat": 65.16667, "lon": 57.08333 }],
  "RUS-pv": [{"name": "Sankt Peterburg Volmet / Pulkovo", "code": "RUS-pv", "lat": 59.78056, "lon": 30.24833 }],
  "RUS-py": [{"name": "Peleduy, Sakha", "code": "RUS-py", "lat": 59.61056, "lon": 112.74389 }],
  "RUS-p2": [{"name": "Petropavlovsk-Kamchatskiy Port", "code": "RUS-p2", "lat": 53.03306, "lon": 158.64222 }],
  "RUS-p3": [{"name": "Petropavlovsk-Kamchatskiy Commercial sea port", "code": "RUS-p3", "lat": 53.00639, "lon": 158.65417 }],
  "RUS-p4": [{"name": "Pevek, Chukotka", "code": "RUS-p4", "lat": 69.70083, "lon": 170.25722 }],
  "RUS-p5": [{"name": "Plastun, Primorye", "code": "RUS-p5", "lat": 44.72833, "lon": 136.31750 }],
  "RUS-p6": [{"name": "Poronaisk, Sakhalin", "code": "RUS-p6", "lat": 49.23167, "lon": 143.11444 }],
  "RUS-rd": [{"name": "Reydovo, Etorofu Island, Kuril", "code": "RUS-rd", "lat": 45.27778, "lon": 148.02500 }],
  "RUS-re": [{"name": "Revda", "code": "RUS-re", "lat": 68.03556, "lon": 34.51667 }],
  "RUS-ro": [{"name": "Rossosh, Voronezhskaya oblast", "code": "RUS-ro", "lat": 50.20000, "lon": 39.58333 }],
  "RUS-rp": [{"name": "Rostov-na-Donu", "code": "RUS-rp", "lat": 47.29944, "lon": 39.67361 }],
  "RUS-ru": [{"name": "Russkoye Ustye", "code": "RUS-ru", "lat": 71.26667, "lon": 150.26667 }],
  "RUS-rv": [{"name": "Rostov Volmet, Rostov-na-Donu", "code": "RUS-rv", "lat": 47.25333, "lon": 39.81722 }],
  "RUS-ry": [{"name": "Rybachi, Primorye", "code": "RUS-ry", "lat": 43.37500, "lon": 131.89444 }],
  "RUS-s":  [{"name": "Samara (Zhygulevsk)", "code": "RUS-s", "lat": 53.28333, "lon": 50.25000 }],
  "RUS-sa": [{"name": "Samara Centre", "code": "RUS-sa", "lat": 53.20000, "lon": 50.16667 }],
  "RUS-sb": [{"name": "Sabetta, Yamalo-Nenets", "code": "RUS-sb", "lat": 71.28333, "lon": 72.03333 }],
  "RUS-sd": [{"name": "Severodvinsk", "code": "RUS-sd", "lat": 64.56667, "lon": 39.85000 }],
  "RUS-se": [{"name": "Sevastopol", "code": "RUS-se", "lat": 44.93333, "lon": 34.46667 }],
  "RUS-sh": [{"name": "Salekhard", "code": "RUS-sh", "lat": 66.58333, "lon": 66.60000 }],
  "RUS-sk": [{"name": "Smolensk", "code": "RUS-sk", "lat": 54.78333, "lon": 32.05000 }],
  "RUS-sl": [{"name": "Seleznevo, Sakhalin", "code": "RUS-sl", "lat": 46.60861, "lon": 141.83306 }],
  "RUS-sm": [{"name": "Severomorsk/Arkhangelsk Military Base", "code": "RUS-sm", "lat": 69.05000, "lon": 33.31667 }],
  "RUS-so": [{"name": "Sochi", "code": "RUS-so", "lat": 43.45000, "lon": 39.95000 }],
  "RUS-sp": [{"name": "St.Petersburg (Popovka/Krasnyj Bor)", "code": "RUS-sp", "lat": 59.65000, "lon": 30.70000 }],
  "RUS-sr": [{"name": "Sevastopol Radio", "code": "RUS-sr", "lat": 44.56667, "lon": 33.53333 }],
  "RUS-st": [{"name": "Saratov", "code": "RUS-st", "lat": 51.53333, "lon": 45.85000 }],
  "RUS-su": [{"name": "Surgut", "code": "RUS-su", "lat": 61.33333, "lon": 73.40000 }],
  "RUS-sv": [{"name": "Samara airport", "code": "RUS-sv", "lat": 53.50000, "lon": 50.15000 }],
  "RUS-sy": [{"name": "Syktyvkar Volmet", "code": "RUS-sy", "lat": 61.63806, "lon": 50.86361 }],
  "RUS-s1": [{"name": "Saratov airport", "code": "RUS-s1", "lat": 51.56667, "lon": 46.05000 }],
  "RUS-s2": [{"name": "Stavropol airport", "code": "RUS-s2", "lat": 45.11667, "lon": 42.11667 }],
  "RUS-s3": [{"name": "St Petersburg port", "code": "RUS-s3", "lat": 59.90000, "lon": 30.25000 }],
  "RUS-s4": [{"name": "St Petersburg Shepelevo", "code": "RUS-s4", "lat": 59.98500, "lon": 29.12694 }],
  "RUS-s5": [{"name": "Sterlegova Cape, Taymyr, Krasnoyarski krai", "code": "RUS-s5", "lat": 75.39861, "lon": 88.75944 }],
  "RUS-s6": [{"name": "Stolbovoy Island, New Siberian Islands, Sakha", "code": "RUS-s6", "lat": 74.16944, "lon": 135.46528 }],
  "RUS-s7": [{"name": "Svobodny, Amur", "code": "RUS-s7", "lat": 51.33528, "lon": 128.17583 }],
  "RUS-t":  [{"name": "Taldom - Severnyj, Radiotsentr 3 (near Moscow)", "code": "RUS-t", "lat": 56.73333, "lon": 37.63333 }],
  "RUS-tg": [{"name": "Taganrog", "code": "RUS-tg", "lat": 47.20528, "lon": 38.95167 }],
  "RUS-ti": [{"name": "Tiksi, Sakha", "code": "RUS-ti", "lat": 71.64111, "lon": 128.88000 }],
  "RUS-tm": [{"name": "Tomsk", "code": "RUS-tm", "lat": 56.48944, "lon": 84.94528 }],
  "RUS-tr": [{"name": "Temryuk, Krasnodar", "code": "RUS-tr", "lat": 45.33028, "lon": 37.22917 }],
  "RUS-ts": [{"name": "Tarko-Sale", "code": "RUS-ts", "lat": 64.93333, "lon": 77.81667 }],
  "RUS-tu": [{"name": "Tulagino, Sakha", "code": "RUS-tu", "lat": 62.23778, "lon": 129.81278 }],
  "RUS-tv": [{"name": "Tavrichanka (Vladivostok, 549, 1377)", "code": "RUS-tv", "lat": 43.33333, "lon": 131.90000 }],
  "RUS-ty": [{"name": "Tyumen Volmet", "code": "RUS-ty", "lat": 57.16667, "lon": 65.31667 }],
  "RUS-t3": [{"name": "Tiksi-3, Sakha", "code": "RUS-t3", "lat": 71.69333, "lon": 128.88083 }],
  "RUS-u":  [{"name": "Ulan-Ude", "code": "RUS-u", "lat": 51.73333, "lon": 107.43333 }],
  "RUS-ub": [{"name": "Ust-Barguzin, Buryatia", "code": "RUS-ub", "lat": 53.42056, "lon": 109.01556 }],
  "RUS-uf": [{"name": "Ufa", "code": "RUS-uf", "lat": 54.56667, "lon": 55.88333 }],
  "RUS-ug": [{"name": "Uglegorsk, Sakhalin", "code": "RUS-ug", "lat": 49.08333, "lon": 142.08333 }],
  "RUS-uk": [{"name": "Ust-Kamchatsk, Kamchatka", "code": "RUS-uk", "lat": 56.21778, "lon": 162.51333 }],
  "RUS-us": [{"name": "Ulan-Ude/Selenginsk", "code": "RUS-us", "lat": 52.03333, "lon": 106.93333 }],
  "RUS-uu": [{"name": "Ulan-Ude port", "code": "RUS-uu", "lat": 51.83806, "lon": 107.57278 }],
  "RUS-uy": [{"name": "Ustyevoye, Kamchatka", "code": "RUS-uy", "lat": 54.15444, "lon": 155.84194 }],
  "RUS-v":  [{"name": "Vladivostok Razdolnoye (Ussuriysk)", "code": "RUS-v", "lat": 43.53333, "lon": 131.95000 }],
  "RUS-va": [{"name": "Vanino, Khabarovski kray", "code": "RUS-va", "lat": 49.08333, "lon": 140.26667 }],
  "RUS-vg": [{"name": "Vologda", "code": "RUS-vg", "lat": 59.28333, "lon": 39.95000 }],
  "RUS-vk": [{"name": "Vorkuta", "code": "RUS-vk", "lat": 67.48333, "lon": 63.98333 }],
  "RUS-vl": [{"name": "Vladivostok Military Base", "code": "RUS-vl", "lat": 43.11667, "lon": 131.90000 }],
  "RUS-vm": [{"name": "Vzmorye, Kaliningrad", "code": "RUS-vm", "lat": 54.68333, "lon": 20.25000 }],
  "RUS-vo": [{"name": "Volgograd", "code": "RUS-vo", "lat": 48.78333, "lon": 44.35000 }],
  "RUS-vp": [{"name": "Vladivostok port", "code": "RUS-vp", "lat": 43.11667, "lon": 131.88333 }],
  "RUS-vr": [{"name": "Varandey, Nenets", "code": "RUS-vr", "lat": 68.80167, "lon": 57.98278 }],
  "RUS-vv": [{"name": "Veselo-Voznesenka", "code": "RUS-vv", "lat": 47.14167, "lon": 38.32806 }],
  "RUS-vz": [{"name": "Vladikavkaz Beslan, North Ossetia", "code": "RUS-vz", "lat": 43.20000, "lon": 44.60000 }],
  "RUS-xe": [{"name": "Khabarovsk-Elban", "code": "RUS-xe", "lat": 50.07333, "lon": 136.60667 }],
  "RUS-xo": [{"name": "Kholmsk, Sakhalin", "code": "RUS-xo", "lat": 47.03333, "lon": 142.05000 }],
  "RUS-xv": [{"name": "Khabarovsk Volmet", "code": "RUS-xv", "lat": 48.51667, "lon": 135.16667 }],
  "RUS-ya": [{"name": "Yakutsk/Tulagino", "code": "RUS-ya", "lat": 62.01667, "lon": 129.80000 }],
  "RUS-ys": [{"name": "Yuzhno-Sakhalinsk (Vestochka)", "code": "RUS-ys", "lat": 46.91667, "lon": 142.90000 }],
  "RUS-yv": [{"name": "Yekaterinburg Volmet (Koltsovo)", "code": "RUS-yv", "lat": 56.75000, "lon": 60.80000 }],
  "RUS-yy": [{"name": "Yakutsk Volmet", "code": "RUS-yy", "lat": 62.08333, "lon": 129.76667 }],
  "RUS-za": [{"name": "Zyryanka, Sakha", "code": "RUS-za", "lat": 65.75722, "lon": 150.83917 }],
  "RUS-zg": [{"name": "Zhigalovo, Irkutsk region", "code": "RUS-zg", "lat": 54.80944, "lon": 105.16944 }],
  "RUS-zp": [{"name": "Zaporozhye, Kamchatka", "code": "RUS-zp", "lat": 51.50806, "lon": 156.49194 }],
  "RUS-zy": [{"name": "Zhatay, Yakutsk", "code": "RUS-zy", "lat": 62.17222, "lon": 129.80583 }],
  "S-b":    [{"name": "Bjuröklubb", "code": "S-b", "lat": 64.46167, "lon": 21.59167 }],
  "S-d":    [{"name": "Delsbo", "code": "S-d", "lat": 61.80000, "lon": 16.55000 }],
  "S-gr":   [{"name": "Varberg-Grimeton", "code": "S-gr", "lat": 57.10833, "lon": 12.39333 }],
  "S-gs":   [{"name": "Gislövshammar", "code": "S-gs", "lat": 55.48944, "lon": 14.31444 }],
  "S-h":    [{"name": "Härnösand", "code": "S-h", "lat": 62.70667, "lon": 18.12972 }],
  "S-j":    [{"name": "Julita", "code": "S-j", "lat": 59.13333, "lon": 16.05000 }],
  "S-k":    [{"name": "Kvarnberget-Vallentuna", "code": "S-k", "lat": 59.50000, "lon": 18.13333 }],
  "S-s":    [{"name": "Sala", "code": "S-s", "lat": 59.60694, "lon": 17.21222 }],
  "S-st":   [{"name": "Stavsnäs", "code": "S-st", "lat": 59.28333, "lon": 18.68333 }],
  "S-t":    [{"name": "Tingstäde", "code": "S-t", "lat": 57.72778, "lon": 18.59861 }],
  "S-v":    [{"name": "Vaxholm, The Castle", "code": "S-v", "lat": 59.40000, "lon": 18.35000 }],
  "SDN-a":  [{"name": "Al-Aitahab", "code": "SDN-a", "lat": 15.58333, "lon": 32.43333 }],
  "SDN-r":  [{"name": "Reiba", "code": "SDN-r", "lat": 13.55000, "lon": 33.51667 }],
  "SEN-r":  [{"name": "Rufisque DIRISI", "code": "SEN-r", "lat": 14.76028, "lon": -17.27389 }],
  "SEN-y":  [{"name": "Dakar Yoff", "code": "SEN-y", "lat": 14.73333, "lon": -17.48333 }],
  "SEY-mh": [{"name": "Mahe", "code": "SEY-mh", "lat": -4.61667, "lon": 55.43333 }],
  "SLM":    [{"name": "Honiara", "code": "SLM", "lat": -9.43333, "lon": 160.05000 }],
  "SNG":    [{"name": "Kranji", "code": "SNG", "lat": 1.41667, "lon": 103.73333 }],
  "SNG-j":  [{"name": "Jurong", "code": "SNG-j", "lat": 1.26667, "lon": 103.66667 }],
  "SNG-v":  [{"name": "Singapore Volmet", "code": "SNG-v", "lat": 1.33639, "lon": 103.68611 }],
  "SOM-b":  [{"name": "Baydhabo", "code": "SOM-b", "lat": 3.11667, "lon": 43.65000 }],
  "SOM-g":  [{"name": "Garoowe", "code": "SOM-g", "lat": 8.40000, "lon": 48.48333 }],
  "SOM-h":  [{"name": "Hargeisa", "code": "SOM-h", "lat": 9.55000, "lon": 44.05000 }],
  "SOM-ma": [{"name": "Mogadishu Airport", "code": "SOM-ma", "lat": 2.01667, "lon": 45.30000 }],
  "SRB-be": [{"name": "Beograd/Belgrade", "code": "SRB-be", "lat": 44.80000, "lon": 20.46667 }],
  "SRB-s":  [{"name": "Stubline", "code": "SRB-s", "lat": 44.56667, "lon": 20.15000 }],
  "SSD-n":  [{"name": "Narus", "code": "SSD-n", "lat": 4.76667, "lon": 33.58333 }],
  "STP":    [{"name": "Pinheira", "code": "STP", "lat": 0.30000, "lon": 6.70000 }],
  "SUI-be": [{"name": "Bern Radio HEB, Prangins", "code": "SUI-be", "lat": 46.40000, "lon": 6.25000 }],
  "SUI-ge": [{"name": "Geneva", "code": "SUI-ge", "lat": 46.23333, "lon": 6.13333 }],
  "SUI-lu": [{"name": "Luzern (approx; Ampegon?)", "code": "SUI-lu", "lat": 46.83333, "lon": 8.40000 }],
  "SUR-pm": [{"name": "Paramaribo", "code": "SUR-pm", "lat": 5.85000, "lon": -55.15000 }],
  "SVK":    [{"name": "Rimavska Sobota", "code": "SVK", "lat": 48.40000, "lon": 20.13333 }],
  "SWZ":    [{"name": "Manzini/Mpangela Ranch", "code": "SWZ", "lat": -26.56667, "lon": 31.98333 }],
  "SYR":    [{"name": "Adra", "code": "SYR", "lat": 33.45000, "lon": 36.50000 }],
  "TCD":    [{"name": "N'Djamena-Gredia", "code": "TCD", "lat": 12.13333, "lon": 15.05000 }],
  "THA-b":  [{"name": "Bangkok / Prathum Thani", "code": "THA-b", "lat": 14.05000, "lon": 100.71667 }],
  "THA-bm": [{"name": "Bangkok Meteo", "code": "THA-bm", "lat": 13.73333, "lon": 100.50000 }],
  "THA-bv": [{"name": "Bangkok Volmet", "code": "THA-bv", "lat": 13.69444, "lon": 100.77056 }],
  "THA-hy": [{"name": "Hat Yai", "code": "THA-hy", "lat": 6.93639, "lon": 100.38833 }],
  "THA-n":  [{"name": "Nakhon Sawan", "code": "THA-n", "lat": 15.81667, "lon": 100.06667 }],
  "THA-u":  [{"name": "Udon Thani", "code": "THA-u", "lat": 17.41667, "lon": 102.80000 }],
  "TJK-da": [{"name": "Dushanbe airport", "code": "TJK-da", "lat": 38.53333, "lon": 68.81667 }],
  "TJK-y":  [{"name": "Yangi Yul (Dushanbe)", "code": "TJK-y", "lat": 38.48333, "lon": 68.80000 }],
  "TJK-o":  [{"name": "Orzu", "code": "TJK-o", "lat": 37.53333, "lon": 68.70000 }],
  "TKM-a":  [{"name": "Asgabat", "code": "TKM-a", "lat": 37.85000, "lon": 58.36667 }],
  "TKM-as": [{"name": "Ashgabat airport", "code": "TKM-as", "lat": 37.98333, "lon": 58.36667 }],
  "TKM-ds": [{"name": "Dasoguz/Dashoguz", "code": "TKM-ds", "lat": 41.76667, "lon": 59.83333 }],
  "TKM-s":  [{"name": "Seyda/Seidi", "code": "TKM-s", "lat": 39.47111, "lon": 62.71861 }],
  "TKM-tb": [{"name": "Turkmenbashi", "code": "TKM-tb", "lat": 40.05000, "lon": 53.00000 }],
  "TRD-np": [{"name": "North Post", "code": "TRD-np", "lat": 10.75000, "lon": -61.56667 }],
  "TUN-bz": [{"name": "Bizerte", "code": "TUN-bz", "lat": 37.28333, "lon": 9.88333 }],
  "TUN-gu": [{"name": "La Goulette", "code": "TUN-gu", "lat": 36.81667, "lon": 10.30000 }],
  "TUN-ke": [{"name": "Kelibia", "code": "TUN-ke", "lat": 36.83333, "lon": 11.08333 }],
  "TUN-mh": [{"name": "Mahdia", "code": "TUN-mh", "lat": 35.51667, "lon": 11.06667 }],
  "TUN-s":  [{"name": "Sfax", "code": "TUN-s", "lat": 34.80000, "lon": 10.88333 }],
  "TUN-sf": [{"name": "Sfax", "code": "TUN-sf", "lat": 34.73333, "lon": 10.73333 }],
  "TUN-tb": [{"name": "Tabarka", "code": "TUN-tb", "lat": 36.95000, "lon": 8.75000 }],
  "TUN-te": [{"name": "Tunis", "code": "TUN-te", "lat": 36.83333, "lon": 10.18333 }],
  "TUN-tu": [{"name": "Tunis", "code": "TUN-tu", "lat": 36.90000, "lon": 10.18333 }],
  "TUN-zz": [{"name": "Zarzis", "code": "TUN-zz", "lat": 33.50000, "lon": 11.10000 }],
  "TUR-a":  [{"name": "Ankara", "code": "TUR-a", "lat": 39.91667, "lon": 32.85000 }],
  "TUR-c":  [{"name": "Cakirlar", "code": "TUR-c", "lat": 39.96667, "lon": 32.66667 }],
  "TUR-e":  [{"name": "Emirler", "code": "TUR-e", "lat": 39.48333, "lon": 32.85000 }],
  "TUR-is": [{"name": "Istanbul TAH", "code": "TUR-is", "lat": 40.98333, "lon": 28.81667 }],
  "TUR-iz": [{"name": "Izmir", "code": "TUR-iz", "lat": 38.41667, "lon": 27.15000 }],
  "TUR-m":  [{"name": "Mersin", "code": "TUR-m", "lat": 36.80000, "lon": 34.63333 }],
  "TWN-f":  [{"name": "Fangliao FAN", "code": "TWN-f", "lat": 22.38333, "lon": 120.56667 }],
  "TWN-h":  [{"name": "Huwei (Yunlin province)", "code": "TWN-h", "lat": 23.71667, "lon": 120.41667 }],
  "TWN-k":  [{"name": "Kouhu (Yunlin province)", "code": "TWN-k", "lat": 23.53333, "lon": 120.16667 }],
  "TWN-L":  [{"name": "Lukang", "code": "TWN-L", "lat": 24.05000, "lon": 120.41667 }],
  "TWN-m":  [{"name": "Minhsiung (Chiayi province)", "code": "TWN-m", "lat": 23.56667, "lon": 120.41667 }],
  "TWN-n":  [{"name": "Tainan/Annan (Tainan city)", "code": "TWN-n", "lat": 23.03333, "lon": 120.16667 }],
  "TWN-p":  [{"name": "Paochung/Baujong (Yunlin province) PAO/BAJ", "code": "TWN-p", "lat": 23.71667, "lon": 120.30000 }],
  "TWN-pe": [{"name": "Penghu (Pescadores), Jiangmei", "code": "TWN-pe", "lat": 23.63333, "lon": 119.60000 }],
  "TWN-q":  [{"name": "Tainan-Qigu/Cigu, Mount Wufen (Central Weather Bureau)", "code": "TWN-q", "lat": 25.004784, "lon": 121.33216 }],
  "TWN-s":  [{"name": "Danshui/Tanshui/Tamsui (Taipei province)", "code": "TWN-s", "lat": 25.18333, "lon": 121.41667 }],
  "TWN-t":  [{"name": "Taipei (Pali)", "code": "TWN-t", "lat": 25.10000, "lon": 121.43333 }],
  "TWN-w":  [{"name": "Taipei, Mount Wufen (Central Weather Bureau)", "code": "TWN-w", "lat": 25.15000, "lon": 121.56667 }],
  "TWN-y":  [{"name": "Kuanyin (Han Sheng)", "code": "TWN-y", "lat": 25.03333, "lon": 121.10000 }],
  "TWN-yl": [{"name": "Yilin", "code": "TWN-yl", "lat": 24.75000, "lon": 121.73333 }],
  "TZA-d":  [{"name": "Daressalam", "code": "TZA-d", "lat": -6.83333, "lon": 39.23333 }],
  "TZA-z":  [{"name": "Zanzibar/Dole", "code": "TZA-z", "lat": -6.08333, "lon": 39.23333 }],
  "UAE":    [{"name": "Dhabbaya", "code": "UAE", "lat": 24.18333, "lon": 54.23333 }],
  "UAE-aj": [{"name": "Al-Abjan", "code": "UAE-aj", "lat": 24.59750, "lon": 55.39750 }],
  "UAE-da": [{"name": "Das Island", "code": "UAE-da", "lat": 25.15444, "lon": 52.87667 }],
  "UAE-mu": [{"name": "Musaffah, Abu Dhabi", "code": "UAE-mu", "lat": 24.38278, "lon": 54.51444 }],
  "UAE-sj": [{"name": "Sharjah", "code": "UAE-sj", "lat": 25.35000, "lon": 55.38333 }],
  "UGA-k":  [{"name": "Kampala-Bugolobi", "code": "UGA-k", "lat": 0.33333, "lon": 32.60000 }],
  "UGA-m":  [{"name": "Mukono", "code": "UGA-m", "lat": 0.35000, "lon": 32.76667 }],
  "UKR-be": [{"name": "Berdiansk", "code": "UKR-be", "lat": 46.63333, "lon": 36.76667 }],
  "UKR-c":  [{"name": "Chernivtsi", "code": "UKR-c", "lat": 48.30000, "lon": 25.83333 }],
  "UKR-k":  [{"name": "Kyyiv/Kiev/Brovary", "code": "UKR-k", "lat": 50.51667, "lon": 30.76667 }],
  "UKR-ke": [{"name": "Kiev", "code": "UKR-ke", "lat": 50.43333, "lon": 30.53333 }],
  "UKR-L":  [{"name": "Lviv (Krasne)", "code": "UKR-L", "lat": 49.85000, "lon": 24.66667 }],
  "UKR-lu": [{"name": "Luch", "code": "UKR-lu", "lat": 46.81667, "lon": 32.21667 }],
  "UKR-m":  [{"name": "Mykolaiv (Kopani)", "code": "UKR-m", "lat": 46.81667, "lon": 32.23333 }],
  "UKR-od": [{"name": "Odessa", "code": "UKR-od", "lat": 46.48333, "lon": 30.73333 }],
  "UKR-pe": [{"name": "Petrivka", "code": "UKR-pe", "lat": 46.90000, "lon": 30.71667 }],
  "UKR-rv": [{"name": "Rivne", "code": "UKR-rv", "lat": 50.61667, "lon": 26.25000 }],
  "UKR-x":  [{"name": "Kharkiv (Taranivka)", "code": "UKR-x", "lat": 49.63333, "lon": 36.11667 }],
  "UKR-z":  [{"name": "Zaporizhzhya", "code": "UKR-z", "lat": 47.83333, "lon": 35.13333 }],
  "URG-lp": [{"name": "La Paloma", "code": "URG-lp", "lat": -34.65000, "lon": -54.13333 }],
  "URG-m":  [{"name": "Montevideo", "code": "URG-m", "lat": -34.83333, "lon": -56.25000 }],
  "URG-pc": [{"name": "Punta Carretas", "code": "URG-pc", "lat": -34.80000, "lon": -56.35000 }],
  "URG-pe": [{"name": "Punta del Este", "code": "URG-pe", "lat": -34.96667, "lon": -54.85000 }],
  "URG-rb": [{"name": "Rio Branco", "code": "URG-rb", "lat": -32.56667, "lon": -53.38333 }],
  "URG-t":  [{"name": "Tacuarembó", "code": "URG-t", "lat": -31.63333, "lon": -55.96667 }],
  "URG-tr": [{"name": "Trouville", "code": "URG-tr", "lat": -34.86667, "lon": -56.30000 }],
  "USA-a":  [{"name": "Andrews AFB, MD", "code": "USA-a", "lat": 38.81083, "lon": -76.86694 }],
  "USA-b":  [{"name": "Birmingham / Vandiver, AL (WEWN)", "code": "USA-b", "lat": 33.50361, "lon": -86.47417 }],
  "USA-ba": [{"name": "WBMD Baltimore, MD", "code": "USA-ba", "lat": 39.32389, "lon": -76.54889 }],
  "USA-bg": [{"name": "Barnegat, NJ", "code": "USA-bg", "lat": 39.75000, "lon": -74.39167 }],
  "USA-bo": [{"name": "Boston, MA", "code": "USA-bo", "lat": 41.70833, "lon": -70.55000 }],
  "USA-bt": [{"name": "Bethel, PA (WMLK)", "code": "USA-bt", "lat": 40.47944, "lon": -76.27972 }],
  "USA-c":  [{"name": "Cypress Creek, SC (WHRI)", "code": "USA-c", "lat": 32.68417, "lon": -81.13056 }],
  "USA-ch": [{"name": "Chesapeake - Pungo Airfield, VA", "code": "USA-ch", "lat": 36.67778, "lon": -76.02778 }],
  "USA-cu": [{"name": "Cutler, ME", "code": "USA-cu", "lat": 44.63333, "lon": -67.28333 }],
  "USA-ds": [{"name": "Destin, FL", "code": "USA-ds", "lat": 30.38333, "lon": -86.43333 }],
  "USA-dv": [{"name": "Dover, NC (KNC)", "code": "USA-dv", "lat": 35.21694, "lon": -77.43833 }],
  "USA-dx": [{"name": "Dixon, CA", "code": "USA-dx", "lat": 38.37944, "lon": -121.76389 }],
  "USA-ej": [{"name": "Ellijay, GA (KJM)", "code": "USA-ej", "lat": 34.63556, "lon": -84.46222 }],
  "USA-fa": [{"name": "Fort Collins, CO", "code": "USA-fa", "lat": 40.68194, "lon": -105.04194 }],
  "USA-fb": [{"name": "Fort Collins, CO", "code": "USA-fb", "lat": 40.67833, "lon": -105.04028 }],
  "USA-fc": [{"name": "Fort Collins, CO", "code": "USA-fc", "lat": 40.68000, "lon": -105.04028 }],
  "USA-fd": [{"name": "Fort Collins, CO", "code": "USA-fd", "lat": 40.67917, "lon": -105.04028 }],
  "USA-fe": [{"name": "Fort Collins, CO", "code": "USA-fe", "lat": 40.68139, "lon": -105.04139 }],
  "USA-ff": [{"name": "Fort Collins, CO", "code": "USA-ff", "lat": 40.67806, "lon": -105.04694 }],
  "USA-fg": [{"name": "Fort Collins, CO", "code": "USA-fg", "lat": 40.68083, "lon": -105.04250 }],
  "USA-fv": [{"name": "Forest, VA", "code": "USA-fv", "lat": 37.39500, "lon": -79.20972 }],
  "USA-g":  [{"name": "Greenville, NC", "code": "USA-g", "lat": 35.46667, "lon": -77.20000 }],
  "USA-hw": [{"name": "KWHW Altus, OK", "code": "USA-hw", "lat": 34.62639, "lon": -99.33611 }],
  "USA-jc": [{"name": "Jim Creek, WA", "code": "USA-jc", "lat": 48.20000, "lon": -121.91667 }],
  "USA-k":  [{"name": "Key Saddlebunch, FL", "code": "USA-k", "lat": 24.65000, "lon": -81.60000 }],
  "USA-L":  [{"name": "Lebanon, TN (WTWW)", "code": "USA-L", "lat": 36.27639, "lon": -86.09944 }],
  "USA-LL": [{"name": "Lakeland, FL (WCY)", "code": "USA-LL", "lat": 27.98139, "lon": -82.05139 }],
  "USA-lm": [{"name": "Lamoure, ND", "code": "USA-lm", "lat": 46.36667, "lon": -98.33333 }],
  "USA-m":  [{"name": "Miami / Hialeah Gardens, FL (WRMI)", "code": "USA-m", "lat": 25.90000, "lon": -80.36361 }],
  "USA-mi": [{"name": "Milton, FL (WJHR)", "code": "USA-mi", "lat": 30.65083, "lon": -87.09083 }],
  "USA-mo": [{"name": "Mobile, AL (WLO)", "code": "USA-mo", "lat": 30.59500, "lon": -88.22139 }],
  "USA-n":  [{"name": "Nashville, TN (WWCR)", "code": "USA-n", "lat": 36.20833, "lon": -86.89389 }],
  "USA-nm": [{"name": "NMG New Orleans, LA", "code": "USA-nm", "lat": 29.88444, "lon": -89.94528 }],
  "USA-no": [{"name": "New Orleans, LA (WRNO)", "code": "USA-no", "lat": 29.83611, "lon": -90.11583 }],
  "USA-np": [{"name": "Newport, NC (WTJC)", "code": "USA-np", "lat": 34.77806, "lon": -76.87694 }],
  "USA-o":  [{"name": "Okeechobee, FL (WYFR)", "code": "USA-o", "lat": 27.45833, "lon": -80.93333 }],
  "USA-ob": [
      {"name": "San Luis Obispo, CA", "code": "USA-ob", "lat": 35.216667, "lon": -120.866667 },
      {"name": "Ragged Point, San Luis Obispo, CA", "code": "USA-ob", "lat": 35.783333, "lon": -121.333333 },
      {"name": "Vandenberg, San Luis Obispo, CA", "code": "USA-ob", "lat": 34.58333, "lon": -120.65000 }
  ],
  "USA-of": [{"name": "Offutt AFB, NE", "code": "USA-of", "lat": 41.11361, "lon": -95.92833 }],
  "USA-q":  [{"name": "Monticello, ME (WBCQ)", "code": "USA-q", "lat": 46.34167, "lon": -67.82778 }],
  "USA-pg": [{"name": "Punta Gorda, FL (KPK)", "code": "USA-pg", "lat": 26.89417, "lon": -82.05972 }],
  "USA-pr": [{"name": "Point Reyes, CA", "code": "USA-pr", "lat": 37.92556, "lon": -122.73111 }],
  "USA-rh": [{"name": "Riverhead, Long Island, NY ", "code": "USA-rh", "lat": 40.88333, "lon": -72.63333 }],
  "USA-rl": [{"name": "Red Lion (York), PA (WINB)", "code": "USA-rl", "lat": 39.90611, "lon": -76.58222 }],
  "USA-rs": [{"name": "Los Angeles / Rancho Simi, CA (KVOH)", "code": "USA-rs", "lat": 34.25639, "lon": -118.64139 }],
  "USA-sc": [{"name": "KEBR Sacramento, CA", "code": "USA-sc", "lat": 38.46278, "lon": -121.13028 }],
  "USA-se": [{"name": "Seattle, WA", "code": "USA-se", "lat": 48.12556, "lon": -122.25056 }],
  "USA-uc": [
      {"name": "UC Davis, Bodega Bay", "code": "USA-uc", "lat": 38.316667, "lon": -123.066667 },
      {"name": "UC Davis, Jenner", "code": "USA-uc", "lat": 38.566667, "lon": -123.333333 },
      {"name": "UC Davis, Inverness", "code": "USA-uc", "lat": 38.046944, "lon": -122.988889 },
      {"name": "UC Davis, Muir Beach", "code": "USA-uc", "lat": 37.866667, "lon": -122.600000 },
      {"name": "UC Davis, Sausalito", "code": "USA-uc", "lat": 37.816667, "lon": -122.533333 },
      {"name": "UC Davis, Inverness", "code": "USA-uc", "lat": 38.02806, "lon": -122.96111 }
  ],
  "USA-ud": [
      {"name": "UC Davis, Bodega Bay", "code": "USA-ud", "lat": 38.316667, "lon": -123.066667 },
      {"name": "UC Davis, Fort Bragg", "code": "USA-ud", "lat": 39.433333, "lon": -123.816667 },
      {"name": "UC Davis, Point Arena", "code": "USA-ud", "lat": 39.933333, "lon": -123.733333 },
      {"name": "UC Davis, Trinidad", "code": "USA-ud", "lat": 41.066667, "lon": -124.150000 },
      {"name": "UC Davis, Samoa", "code": "USA-ud", "lat": 40.76667, "lon": -124.21667 }
  ],
  "USA-v":  [{"name": "Vado, NM (KJES)", "code": "USA-v", "lat": 32.13389, "lon": -106.59000 }],
  "USA-vs": [{"name": "Vashon Island, WA", "code": "USA-vs", "lat": 47.37083, "lon": -122.48778 }],
  "USA-wa": [{"name": "Washington, DC", "code": "USA-wa", "lat": 38.91667, "lon": -77.05000 }],
  "USA-wc": [{"name": "'West Coast' Beale AFB Marysville, CA", "code": "USA-wc", "lat": 39.13333, "lon": -121.43333 }],
  "USA-wg": [{"name": "WGM Fort Lauderdale, FL", "code": "USA-wg", "lat": 26.56667, "lon": -80.08333 }],
  "USA-ws": [{"name": "KOVR West Sacramento, CA", "code": "USA-ws", "lat": 38.58333, "lon": -121.55000 }],
  "USA-wx": [{"name": "WHX Hillsboro, WV", "code": "USA-wx", "lat": 38.26861, "lon": -80.26861 }],
  "UZB":    [{"name": "Tashkent", "code": "UZB", "lat": 41.21667, "lon": 69.15000 }],
  "UZB-a":  [{"name": "Tashkent Airport", "code": "UZB-a", "lat": 41.26667, "lon": 69.28333 }],
  "UZB-nu": [{"name": "Nukus, Karakalpakstan", "code": "UZB-nu", "lat": 42.48333, "lon": 59.61667 }],
  "UZB-s":  [{"name": "Samarkand", "code": "UZB-s", "lat": 39.70000, "lon": 66.98333 }],
  "UZB-ta": [{"name": "Tashkent I/II", "code": "UZB-ta", "lat": 41.31667, "lon": 69.25000 }],
  "VEN-t":  [{"name": "El Tigre", "code": "VEN-t", "lat": 8.88333, "lon": -64.26667 }],
  "VEN-y":  [{"name": "YVTO Caracas", "code": "VEN-y", "lat": 10.50361, "lon": -66.92889 }],
  "VTN-b":  [{"name": "Buon Me Thuot, Daclac", "code": "VTN-b", "lat": 12.63333, "lon": 108.01667 }],
  "VTN-bt": [{"name": "Ben Thuy", "code": "VTN-bt", "lat": 18.81667, "lon": 105.71667 }],
  "VTN-cm": [{"name": "Ca Mau", "code": "VTN-cm", "lat": 9.18972, "lon": 105.13361 }],
  "VTN-co": [{"name": "Cua Ong", "code": "VTN-co", "lat": 21.02611, "lon": 107.36694 }],
  "VTN-cr": [{"name": "Cam Ranh", "code": "VTN-cr", "lat": 12.07972, "lon": 109.18194 }],
  "VTN-ct": [{"name": "Can Tho", "code": "VTN-ct", "lat": 10.07167, "lon": 105.75889 }],
  "VTN-db": [{"name": "Dien Bien", "code": "VTN-db", "lat": 21.36667, "lon": 103.00000 }],
  "VTN-dn": [{"name": "Da Nang", "code": "VTN-dn", "lat": 16.05472, "lon": 108.15778 }],
  "VTN-hc": [{"name": "Ho Chi Minh City / Vung Tau", "code": "VTN-hc", "lat": 10.39472, "lon": 107.14500 }],
  "VTN-hg": [{"name": "Hon Gai (Ha Long)", "code": "VTN-hg", "lat": 20.95000, "lon": 107.06667 }],
  "VTN-hp": [{"name": "Hai Phong", "code": "VTN-hp", "lat": 20.85028, "lon": 106.73361 }],
  "VTN-hu": [{"name": "Hue", "code": "VTN-hu", "lat": 16.55056, "lon": 107.64111 }],
  "VTN-kg": [{"name": "Kien Giang", "code": "VTN-kg", "lat": 9.99139, "lon": 105.10250 }],
  "VTN-L":  [{"name": "Son La", "code": "VTN-L", "lat": 21.33333, "lon": 103.91667 }],
  "VTN-m":  [{"name": "Hanoi-Metri", "code": "VTN-m", "lat": 21.00000, "lon": 105.78333 }],
  "VTN-mc": [{"name": "Mong Cai", "code": "VTN-mc", "lat": 21.52583, "lon": 107.96639 }],
  "VTN-mh": [{"name": "My Hao", "code": "VTN-mh", "lat": 20.91667, "lon": 106.08333 }],
  "VTN-nt": [{"name": "Nha Trang", "code": "VTN-nt", "lat": 12.22194, "lon": 109.18056 }],
  "VTN-pr": [{"name": "Phan Rang", "code": "VTN-pr", "lat": 11.56667, "lon": 109.01667 }],
  "VTN-pt": [{"name": "Phan Tiet", "code": "VTN-pt", "lat": 10.91778, "lon": 108.10611 }],
  "VTN-py": [{"name": "Phu Yen", "code": "VTN-py", "lat": 13.10611, "lon": 109.31139 }],
  "VTN-qn": [{"name": "Quy Nhon", "code": "VTN-qn", "lat": 13.77778, "lon": 109.23917 }],
  "VTN-s":  [{"name": "Hanoi-Sontay", "code": "VTN-s", "lat": 21.20000, "lon": 105.36667 }],
  "VTN-t":  [{"name": "Thoi Long / Thoi Hung", "code": "VTN-t", "lat": 10.11667, "lon": 105.56667 }],
  "VTN-th": [{"name": "Thanh Hoa", "code": "VTN-th", "lat": 19.34944, "lon": 105.79333 }],
  "VTN-vt": [{"name": "Vung Tau", "code": "VTN-vt", "lat": 10.39472, "lon": 107.14500 }],
  "VTN-x":  [{"name": "Xuan Mai", "code": "VTN-x", "lat": 20.71667, "lon": 105.55000 }],
  "VUT":    [{"name": "Empten Lagoon", "code": "VUT", "lat": -17.75000, "lon": 168.36667 }],
  "YEM-a":  [{"name": "Al Hiswah/Aden", "code": "YEM-a", "lat": 12.83333, "lon": 45.03333 }],
  "YEM-s":  [{"name": "Sanaa", "code": "YEM-s", "lat": 15.36667, "lon": 44.18333 }],
  "ZMB-L":  [{"name": "Lusaka", "code": "ZMB-L", "lat": -15.50000, "lon": 28.25000 }],
  "ZMB-m":  [{"name": "Makeni Ranch", "code": "ZMB-m", "lat": -15.53333, "lon": 28.00000 }],
  "ZWE":    [{"name": "Gweru/Guinea Fowl", "code": "ZWE", "lat": -19.43333, "lon": 29.85000 }],
}

# AUS-vs may stand for any of these sites
EIBI_Locations["AUS-vs"] = (
    EIBI_Locations["AUS-va"] +
    EIBI_Locations["AUS-ee"] + EIBI_Locations["AUS-st"] + EIBI_Locations["AUS-nc"] +
    EIBI_Locations["AUS-ch"] + EIBI_Locations["AUS-ct"] + EIBI_Locations["AUS-pw"] +
    EIBI_Locations["AUS-il"] + EIBI_Locations["AUS-al"] + EIBI_Locations["AUS-sb"]
)

#
# Let first country site represent the entire country
#
MORE_Locations = {}

for code in EIBI_Locations.keys():
    x = code.find("-")
    if x >= 0:
        itu = code[:x]
        if itu not in EIBI_Locations and itu not in MORE_Locations:
            # @@@ Too much output here
            #logger.debug("Assuming '{0}' location stands for '{1}'...".format(code, itu))
            MORE_Locations[itu] = EIBI_Locations[code]

EIBI_Locations.update(MORE_Locations)
del MORE_Locations
